/** @file tacname.c
 *  @brief Working with TAC names in TAC structure.
 */
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include "tpcift.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <time.h>
#include <string.h>
/*****************************************************************************/
#include "tpctac.h"
/*****************************************************************************/

/*****************************************************************************/
/** Verifies that TAC structure contains a name for each TAC, and no other TAC has the same name.
 *  @return 1 if individual names exist, 0 if not.
 *  @author Vesa Oikonen
 *  @sa tacSelectTACs, tacSelectBestReference, roinameMatch, tacEnsureNames
 */ 
int tacIndividualNames(
  /** Pointer to TAC structure. */
  TAC *tac
) {
  if(tac==NULL || tac->tacNr<1) return(0);
  int ri, rj;
  /* First check that names do exist */
  for(ri=0; ri<tac->tacNr; ri++) {
    if(!roinameExists(tac->c[ri].name)) return(0);
  }
  /* If just one TAC, then its name is individual */
  if(tac->tacNr==1) return(1);
  /* Otherwise compare each of the names */
  for(ri=0; ri<tac->tacNr-1; ri++)
    for(rj=ri+1; rj<tac->tacNr; rj++)
      if(strcasecmp(tac->c[ri].name, tac->c[rj].name)==0) return(0);
  return(1);
}
/*****************************************************************************/

/*****************************************************************************/
/** Ensure that TAC structure contains a name for each TAC.
 *  If not available, then TAC index+1 is written as name.
 *  @sa tacIndividualNames, tacSelectTACs, tacSelectBestReference, roinameMatch
 *  @author Vesa Oikonen
 */ 
void tacEnsureNames(
  /** Pointer to TAC structure. */
  TAC *tac
) {
  if(tac==NULL || tac->tacNr<1) return;
  int ri, u, n;
  u=tac->tacNr; n=1; while((u/=10)>=1) n++;
  if(n>MAX_TACNAME_LEN) n=MAX_TACNAME_LEN;
  for(ri=0; ri<tac->tacNr; ri++) {
    if(!roinameExists(tac->c[ri].name))
      sprintf(tac->c[ri].name, "%0*d", n, 1+ri);
  }
  return;
}
/*****************************************************************************/

/*****************************************************************************/
