/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include "tpcextensions.h"
#include "test_tpctac.h"
/*****************************************************************************/

/*****************************************************************************/
int test_tacIsWeighted(
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  if(verbose>0) {
    printf("\n=====================================\n");
    printf("\n%s\n", __func__);
    printf("\n=====================================\n");
  }
  
  TAC tac; tacInit(&tac);

  if(verbose>1) printf("\n trying with empty data\n");
  if(tacIsWeighted(NULL)) return(1);
  if(tacIsWeighted(&tac)) return(2);

  if(verbose>1) printf("\n trying with weights off\n");
  tac.weighting=WEIGHTING_OFF;
  if(tacIsWeighted(&tac)) return(3);

  if(verbose>1) printf("\n trying with weights on\n");
  tac.weighting=WEIGHTING_ON_GENERAL;
  if(!tacIsWeighted(&tac)) return(4);
  tac.weighting=WEIGHTING_ON_COUNTS;
  if(!tacIsWeighted(&tac)) return(5);
  tac.weighting=WEIGHTING_ON_FD;
  if(!tacIsWeighted(&tac)) return(6);

  if(verbose>1) printf("\n trying with rubbish weight setting\n");
  tac.weighting=WEIGHTING_LAST;
  if(tacIsWeighted(&tac)) return(7);
  
  tacFree(&tac);
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
int test_tacWCopy(
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  if(verbose>0) {
    printf("\n=====================================\n");
    printf("\n%s\n", __func__);
    printf("\n=====================================\n");
  }
  
  int ret;
  TAC tac1, tac2; 
  tacInit(&tac1); tacInit(&tac2);

  if(verbose>1) printf("\n trying with empty data\n");
  ret=tacWCopy(&tac1, &tac2, 0, 0);
  if(ret==TPCERROR_OK) {tacFree(&tac1); tacFree(&tac2); return 1;}
  
  if(verbose>1) printf("\n creating test data\n");
  ret=create_tac(&tac1); if(ret==TPCERROR_OK) ret=create_tac(&tac2);  
  if(ret!=TPCERROR_OK) {tacFree(&tac1); tacFree(&tac2); return 10;}

  if(verbose>1) printf("\n checking indices\n");
  ret=tacWCopy(&tac1, &tac2, 2, 0);
  if(ret==TPCERROR_OK) {tacFree(&tac1); tacFree(&tac2); return 11;}
  ret=tacWCopy(&tac1, &tac2, -1, 1);
  if(ret==TPCERROR_OK) {tacFree(&tac1); tacFree(&tac2); return 12;}
  ret=tacWCopy(&tac1, &tac2, 0, tac1._sampleNr);
  if(ret==TPCERROR_OK) {tacFree(&tac1); tacFree(&tac2); return 13;}
  
  if(verbose>1) printf("\n checking copying\n");
  for(int i=0; i<tac2.sampleNr; i++) tac2.w[i]=nan("");
  tac2.weighting=WEIGHTING_OFF;
  for(int i=0; i<tac1.sampleNr; i++) tac1.w[i]=(double)(i+1);
  tac1.weighting=WEIGHTING_ON_GENERAL;
  
  ret=tacWCopy(&tac1, &tac2, 0, tac1.sampleNr-1);
  if(ret!=TPCERROR_OK) {tacFree(&tac1); tacFree(&tac2); return 21;}

  ret=tacCompareWeights(&tac1, &tac2, 1.0E-08, -1.0, NULL);
  if(ret!=0) {tacFree(&tac1); tacFree(&tac2); return 22;}
  
  tacFree(&tac1); tacFree(&tac2);
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
int test_tacWMove(
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  if(verbose>0) {
    printf("\n=====================================\n");
    printf("\n%s\n", __func__);
    printf("\n=====================================\n");
  }
  
  int ret;
  TAC tac; 
  tacInit(&tac);

  if(verbose>1) printf("\n trying with empty data\n");
  ret=tacWMove(&tac, 0, NULL);
  if(ret==TPCERROR_OK) {tacFree(&tac); return 1;}
  
  if(verbose>1) printf("\n creating test data\n");
  ret=create_tac(&tac); if(ret!=TPCERROR_OK) {tacFree(&tac); return 10;}

  if(verbose>1) printf("\n weight column not found \n");
  ret=tacWMove(&tac, 0, NULL);
  if(ret!=TPCERROR_NO_WEIGHTS) {tacFree(&tac); return 11;}

  if(verbose>1) printf("\n weight column is found; overwrite not allowed \n");
  strcpy(tac.c[0].name, "weight");
  tac.weighting=WEIGHTING_ON_GENERAL;
  ret=tacWMove(&tac, 0, NULL);
  if(ret!=TPCERROR_DUPLICATE_DATA) {tacFree(&tac); return 21;}

  if(verbose>1) printf("\n weight column is found; overwrite is allowed \n");
  tac.weighting=WEIGHTING_ON_GENERAL;
  ret=tacWMove(&tac, 1, NULL);
  if(ret!=TPCERROR_OK) {tacFree(&tac); return 22;}

  if(verbose>1) printf("\n weight column is found; no prev weights \n");
  tacFree(&tac);
  ret=create_tac(&tac); if(ret!=TPCERROR_OK) {tacFree(&tac); return 10;}
  strcpy(tac.c[0].name, "weight");
  tac.weighting=WEIGHTING_OFF;
  ret=tacWMove(&tac, 0, NULL);
  if(ret!=TPCERROR_OK) {tacFree(&tac); return 23;}

  tacFree(&tac);
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
int test_tacWByFreq(
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  if(verbose>0) {
    printf("\n=====================================\n");
    printf("\n%s\n", __func__);
    printf("\n=====================================\n");
  }
  
  int i, ret;
  TAC tac; tacInit(&tac);

  if(verbose>1) printf("\n trying with empty data\n");
  ret=tacWByFreq(&tac, ISOTOPE_UNKNOWN, NULL);
  if(ret==TPCERROR_OK) {tacFree(&tac); return 1;}
  
  if(verbose>1) printf("\n creating test data\n");
  ret=create_tac(&tac); if(ret!=TPCERROR_OK) {tacFree(&tac); return 10;}

  if(verbose>1) printf("\n trying with frames\n");
  tac.weighting=WEIGHTING_OFF;
  tac.isframe=1;
  ret=tacWByFreq(&tac, ISOTOPE_UNKNOWN, status);
  if(ret!=TPCERROR_OK) {tacFree(&tac); return 100;}
  if(!tacIsWeighted(&tac)) {tacFree(&tac); return 101;}
  for(i=0; i<tac.sampleNr; i++) {
    if(verbose>2) printf("  %g - %g : %g\n", tac.x1[i], tac.x2[i], tac.w[i]);
    if(!doubleMatch(tac.w[i], 1.0, 0.0001)) {tacFree(&tac); return 102;}
  }

  if(verbose>1) printf("\n trying without frames\n");
  tac.weighting=WEIGHTING_OFF;
  tac.isframe=0;
  ret=tacWByFreq(&tac, ISOTOPE_UNKNOWN, status);
  if(ret!=TPCERROR_OK) {tacFree(&tac); return 200;}
  if(!tacIsWeighted(&tac)) {tacFree(&tac); return 201;}
  for(i=0; i<tac.sampleNr; i++) {
    if(verbose>2) printf("  %g : %g\n", tac.x[i], tac.w[i]);
    if(!doubleMatch(tac.w[i], 1.0, 0.0001)) {tacFree(&tac); return 202;}
  }


  if(verbose>1) printf("\n add weights with isotope\n");
  tac.weighting=WEIGHTING_OFF;
  tac.isframe=1;
  ret=tacWByFreq(&tac, ISOTOPE_C_11, status);
  if(ret!=TPCERROR_OK) {tacFree(&tac); return 301;}
  if(!tacIsWeighted(&tac)) {tacFree(&tac); return 302;}
  if(verbose>2) for(i=0; i<tac.sampleNr; i++) {
    printf("  %g : %g\n", tac.x[i], tac.w[i]);
  }
  if(!doubleMatch(tac.w[0], 1.07, 0.01)) {tacFree(&tac); return 303;}
  if(!doubleMatch(tac.w[tac.sampleNr-1], 0.93, 0.01)) {tacFree(&tac); return 304;}


  tac.weighting=WEIGHTING_OFF;
  tac.isframe=0;
  ret=tacWByFreq(&tac, ISOTOPE_C_11, status);
  if(ret!=TPCERROR_OK) {tacFree(&tac); return 311;}
  if(!tacIsWeighted(&tac)) {tacFree(&tac); return 312;}
  if(verbose>2) for(i=0; i<tac.sampleNr; i++) {
    printf("  %g : %g\n", tac.x[i], tac.w[i]);
  }
  if(!doubleMatch(tac.w[0], 1.07, 0.01)) {tacFree(&tac); return 313;}
  if(!doubleMatch(tac.w[tac.sampleNr-1], 0.93, 0.01)) {tacFree(&tac); return 314;}

  tacFree(&tac);
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
int test_tacWSampleNr(
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  if(verbose>0) {
    printf("\n=====================================\n");
    printf("\n%s\n", __func__);
    printf("\n=====================================\n");
  }
  
  
  TAC tac; tacInit(&tac);
  unsigned int n;
  int ret;

  if(verbose>1) printf("\n trying with empty data\n");
  n=tacWSampleNr(NULL);
  if(n>(unsigned int)0) {return 1;}
  n=tacWSampleNr(&tac);
  if(n>(unsigned int)0) {return 2;}
  
  if(verbose>1) printf("\n creating test data\n");
  ret=create_tac(&tac); if(ret!=TPCERROR_OK) {tacFree(&tac); return 10;}

  if(verbose>1) printf("\n trying with no weights\n");
  tac.weighting=WEIGHTING_OFF;
  n=tacWSampleNr(&tac);
  if(n!=(unsigned int)tac.sampleNr) {tacFree(&tac); return 11;}

  if(verbose>1) printf("\n trying with weights\n");
  ret=tacWByFreq(&tac, ISOTOPE_UNKNOWN, status);
  if(ret!=TPCERROR_OK) {tacFree(&tac); return 100;}
  n=tacWSampleNr(&tac);
  if(n!=(unsigned int)tac.sampleNr) {tacFree(&tac); return 21;}
  tac.w[1]=0.0;
  n=tacWSampleNr(&tac);
  if(n!=(unsigned int)tac.sampleNr-1) {tacFree(&tac); return 22;}

  if(verbose>1) printf("\n trying with unknown weight status\n");
  tac.weighting=WEIGHTING_UNKNOWN;
  n=tacWSampleNr(&tac);
  if(n!=(unsigned int)tac.sampleNr-1) {tacFree(&tac); return 31;}
  for(int i=1; i<tac.sampleNr; i++) tac.w[i]=nan("");
  n=tacWSampleNr(&tac); if(verbose>2) printf(" n=%u\n", n);
  if(n!=(unsigned int)1) {tacFree(&tac); return 32;}
  for(int i=0; i<tac.sampleNr; i++) tac.w[i]=nan("");
  n=tacWSampleNr(&tac); if(verbose>2) printf(" n=%u\n", n);
  if(n!=(unsigned int)tac.sampleNr) {tacFree(&tac); return 33;}


  tacFree(&tac);
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
int test_sifWeight(
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  if(verbose>0) {
    printf("\n=====================================\n");
    printf("\n%s\n", __func__);
    printf("\n=====================================\n");
  }

  TAC sif; tacInit(&sif);

  if(verbose>1) printf("\n trying with empty data\n");
  if(sifWeight(NULL, ISOTOPE_UNKNOWN, status)!=TPCERROR_NO_DATA) {return(1);}
  if(sifWeight(&sif, ISOTOPE_UNKNOWN, status)!=TPCERROR_NO_DATA) {return(2);}
  
  if(verbose>1) printf("creating test data\n");
  if(create_sif(&sif)!=TPCERROR_OK) {tacFree(&sif); return(10);}
  if(verbose>3) tacWrite(&sif, stdout, TAC_FORMAT_TSV_UK, 1, NULL);

  if(verbose>1) printf("add weights without isotope\n");
  if(sifWeight(&sif, ISOTOPE_UNKNOWN, status)!=TPCERROR_OK) {tacFree(&sif); return(100);}
  if(!tacIsWeighted(&sif)) {tacFree(&sif); return(101);}
  if(verbose>3) tacWrite(&sif, stdout, TAC_FORMAT_TSV_UK, 1, NULL);
  if(!doubleMatch(2.53206, sif.w[0], 1.0E-04) || !doubleMatch(1.01282, sif.w[1], 1.0E-04) || 
     !doubleMatch(0.633015, sif.w[2], 1.0E-04) || !doubleMatch(0.460375, sif.w[3], 1.0E-04) ||
     !doubleMatch(0.361723, sif.w[4], 1.0E-04)) {
    tacFree(&sif); return 103;
  }

  if(verbose>1) printf("add weights with isotope\n");
  if(sifWeight(&sif, ISOTOPE_O_15, status)!=TPCERROR_OK) {tacFree(&sif); return(110);}
  if(!tacIsWeighted(&sif)) {tacFree(&sif); return(111);}
  if(verbose>3) tacWrite(&sif, stdout, TAC_FORMAT_TSV_UK, 1, NULL);
  if(!doubleMatch(2.67523, sif.w[0], 1.0E-04) || !doubleMatch(1.01146, sif.w[1], 1.0E-04) || 
     !doubleMatch(0.597521, sif.w[2], 1.0E-04) || !doubleMatch(0.410749, sif.w[3], 1.0E-04) ||
     !doubleMatch(0.305047, sif.w[4], 1.0E-04)) {
    tacFree(&sif); return 113;
  }
  if(!doubleMatch(1.0, doubleMean(sif.w, 5), 1.0E-10)) {tacFree(&sif); return 114;}

  if(verbose>1) printf("one of trues is zero, set to max weight\n");
  sif.c[1].y[0]=sif.c[0].y[0];
  if(sifWeight(&sif, ISOTOPE_UNKNOWN, status)!=TPCERROR_OK) {tacFree(&sif); return(200);}
  if(!tacIsWeighted(&sif)) {tacFree(&sif); return(201);}
  if(verbose>3) tacWrite(&sif, stdout, TAC_FORMAT_TSV_UK, 1, NULL);
  if(!doubleMatch(1.45489, sif.w[0], 1.0E-04) || !doubleMatch(1.45489, sif.w[1], 1.0E-04) || 
     !doubleMatch(0.909306, sif.w[2], 1.0E-04) || !doubleMatch(0.661313, sif.w[3], 1.0E-04) ||
     !doubleMatch(0.519603, sif.w[4], 1.0E-04)) {
    tacFree(&sif); return 203;
  }
  if(!doubleMatch(1.0, doubleMean(sif.w, 5), 1.0E-10)) {tacFree(&sif); return 204;}

  tacFree(&sif);

  statusSet(status, __func__, __FILE__, __LINE__, 0);
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
int test_tacWeightNorm(
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  if(verbose>0) {
    printf("\n=====================================\n");
    printf("\n%s\n", __func__);
    printf("\n=====================================\n");
  }

  TAC tac; tacInit(&tac);

  if(verbose>1) printf("\n trying with empty data\n");
  if(tacWeightNorm(NULL, NULL)!=TPCERROR_NO_DATA) {return 1;}
  if(tacWeightNorm(&tac, NULL)!=TPCERROR_NO_DATA) {return 2;}
  
  if(verbose>1) printf("\n creating test data\n");
  if(create_tac(&tac)!=TPCERROR_OK) {tacFree(&tac); return 3;}
  if(verbose>3) tacWrite(&tac, stdout, TAC_FORMAT_TSV_UK, 1, NULL);

  if(verbose>1) printf("\n trying with no weights\n");
  tac.weighting=WEIGHTING_OFF;
  if(tacWeightNorm(&tac, NULL)!=TPCERROR_NO_WEIGHTS) {tacFree(&tac); return 4;}

  if(verbose>1) printf("\n trying with unknown weight status\n");
  tac.weighting=WEIGHTING_UNKNOWN;
  if(tacWeightNorm(&tac, NULL)!=TPCERROR_NO_WEIGHTS) {tacFree(&tac); return 5;}

  if(verbose>1) printf("\n trying with zero weights\n");
  tac.weighting=WEIGHTING_ON_GENERAL;
  for(int i=0; i<tac.sampleNr; i++) tac.w[i]=0.0;
  if(tacWeightNorm(&tac, status)!=TPCERROR_INVALID_VALUE) {
    fprintf(stderr, "Function returns: %s\n", errorMsg(status->error));
    tacFree(&tac); return 6;
  }

  if(verbose>1) printf("\n trying with NaN weights\n");
  tac.weighting=WEIGHTING_ON_GENERAL;
  for(int i=0; i<tac.sampleNr; i++) tac.w[i]=nan("");
  if(tacWeightNorm(&tac, status)!=TPCERROR_INVALID_VALUE) {
    fprintf(stderr, "Function returns: %s\n", errorMsg(status->error));
    tacFree(&tac); return 7;
  }

  if(verbose>1) printf("\n trying with weights of NaN except for one\n");
  tac.weighting=WEIGHTING_ON_GENERAL;
  for(int i=0; i<tac.sampleNr; i++) tac.w[i]=nan("");
  tac.w[1]=2.0;
  if(tacWeightNorm(&tac, status)!=TPCERROR_OK) {
    fprintf(stderr, "Function returns: %s\n", errorMsg(status->error));
    tacFree(&tac); return 11;
  }
  if(!doubleMatch(1.0, tac.w[1], 1.0E-20)) {tacFree(&tac); return 12;}

  if(verbose>1) printf("\n trying with weights of zero except for one\n");
  tac.weighting=WEIGHTING_ON_GENERAL;
  for(int i=0; i<tac.sampleNr; i++) tac.w[i]=0.0;
  tac.w[1]=2.0; if(verbose>3) tacWrite(&tac, stdout, TAC_FORMAT_TSV_UK, 1, NULL);
  if(tacWeightNorm(&tac, status)!=TPCERROR_OK) {
    fprintf(stderr, "Function returns: %s\n", errorMsg(status->error));
    tacFree(&tac); return 13;
  }
  if(!doubleMatch(1.0, tac.w[1], 1.0E-20)) {
    if(verbose>2) tacWrite(&tac, stdout, TAC_FORMAT_TSV_UK, 1, NULL);
    tacFree(&tac); return 14;
  }

  if(verbose>1) printf("\n trying with a range of weights, in need for norm\n");
  tac.weighting=WEIGHTING_ON_GENERAL;
  for(int i=0; i<tac.sampleNr; i++) tac.w[i]=(double)(1+i);
  if(verbose>3) tacWrite(&tac, stdout, TAC_FORMAT_TSV_UK, 1, NULL);
  if(tacWeightNorm(&tac, status)!=TPCERROR_OK) {
    fprintf(stderr, "Function returns: %s\n", errorMsg(status->error));
    tacFree(&tac); return 21;
  }
  if(!doubleMatch(0.3333333, tac.w[0], 1.0E-06) || !doubleMatch(0.6666667, tac.w[1], 1.0E-06) || 
     !doubleMatch(1.0, tac.w[2], 1.0E-06) || !doubleMatch(1.333333333, tac.w[3], 1.0E-06) ||
     !doubleMatch(1.6666667, tac.w[4], 1.0E-06)) {
    if(verbose>2) tacWrite(&tac, stdout, TAC_FORMAT_TSV_UK, 1, NULL);
    tacFree(&tac); return 22;
  }

  if(verbose>1) printf("\n trying with a range of weights, when no need for norm\n");
  if(tacWeightNorm(&tac, status)!=TPCERROR_OK) {
    fprintf(stderr, "Function returns: %s\n", errorMsg(status->error));
    tacFree(&tac); return 23;
  }
  if(!doubleMatch(0.3333333, tac.w[0], 1.0E-06) || !doubleMatch(0.6666667, tac.w[1], 1.0E-06) || 
     !doubleMatch(1.0, tac.w[2], 1.0E-06) || !doubleMatch(1.333333333, tac.w[3], 1.0E-06) ||
     !doubleMatch(1.6666667, tac.w[4], 1.0E-06)) {
    if(verbose>2) tacWrite(&tac, stdout, TAC_FORMAT_TSV_UK, 1, NULL);
    tacFree(&tac); return 24;
  }

  tacFree(&tac);

  statusSet(status, __func__, __FILE__, __LINE__, 0);
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
int test_tacWeightModerate(
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  if(verbose>0) {
    printf("\n=====================================\n");
    printf("\n%s\n", __func__);
    printf("\n=====================================\n");
  }

//int tacWeightModerate(TAC *tac, const double minprop, const int doZeroes, const int doNaNs, TPCSTATUS *status);

  TAC tac; tacInit(&tac);

  if(verbose>1) printf("\n trying with empty data\n");
  if(tacWeightModerate(NULL, 0.0, 0, 0, NULL)!=TPCERROR_NO_DATA) {return 1;}
  if(tacWeightModerate(&tac, 0.0, 0, 0, NULL)!=TPCERROR_NO_DATA) {return 2;}
  
  if(verbose>1) printf("\n creating test data\n");
  if(create_tac(&tac)!=TPCERROR_OK) {tacFree(&tac); return 3;}
  if(verbose>3) tacWrite(&tac, stdout, TAC_FORMAT_TSV_UK, 1, NULL);

  if(verbose>1) printf("\n trying with no weights\n");
  tac.weighting=WEIGHTING_OFF;
  if(tacWeightModerate(&tac, 0.0, 0, 0, NULL)!=TPCERROR_NO_WEIGHTS) {tacFree(&tac); return 4;}
  if(tacWeightModerate(&tac, 0.1, 0, 0, NULL)!=TPCERROR_NO_WEIGHTS) {tacFree(&tac); return 4;}

  if(verbose>1) printf("\n trying with unknown weight status\n");
  tac.weighting=WEIGHTING_UNKNOWN;
  if(tacWeightModerate(&tac, 0.0, 0, 0, NULL)!=TPCERROR_NO_WEIGHTS) {tacFree(&tac); return 5;}
  if(tacWeightModerate(&tac, 0.1, 0, 0, NULL)!=TPCERROR_NO_WEIGHTS) {tacFree(&tac); return 5;}

  if(verbose>1) printf("\n trying with zero weights\n");
  tac.weighting=WEIGHTING_ON_GENERAL;
  for(int i=0; i<tac.sampleNr; i++) tac.w[i]=0.0;
  if(tacWeightModerate(&tac, 0.0, 0, 0, status)!=TPCERROR_NO_WEIGHTS) {
    fprintf(stderr, "Function returns: %s\n", errorMsg(status->error));
    tacFree(&tac); return 6;
  }

  if(verbose>1) printf("\n trying with NaN weights\n");
  tac.weighting=WEIGHTING_ON_GENERAL;
  for(int i=0; i<tac.sampleNr; i++) tac.w[i]=nan("");
  if(tacWeightModerate(&tac, 0.0, 0, 0, status)!=TPCERROR_NO_WEIGHTS) {
    fprintf(stderr, "Function returns: %s\n", errorMsg(status->error));
    tacFree(&tac); return 7;
  }


  if(verbose>1) printf("\n trying with weights of NaN except for one\n");
  tac.weighting=WEIGHTING_ON_GENERAL;
  for(int i=0; i<tac.sampleNr; i++) tac.w[i]=nan("");
  tac.w[1]=2.0;
  if(tacWeightModerate(&tac, 0.0, 0, 0, status)!=TPCERROR_OK) {
    fprintf(stderr, "Function returns: %s\n", errorMsg(status->error));
    tacFree(&tac); return 11;
  }
  if(!doubleMatch(1.0, tac.w[1], 1.0E-20)) {tacFree(&tac); return 12;}
  if(tacWeightModerate(&tac, 0.1, 0, 0, status)!=TPCERROR_OK) {
    fprintf(stderr, "Function returns: %s\n", errorMsg(status->error));
    tacFree(&tac); return 13;
  }
  if(!doubleMatch(1.0, tac.w[1], 1.0E-20)) {tacFree(&tac); return 14;}

  if(verbose>1) printf("\n trying with weights of zero except for one\n");
  tac.weighting=WEIGHTING_ON_GENERAL;
  for(int i=0; i<tac.sampleNr; i++) tac.w[i]=0.0;
  tac.w[1]=2.0; if(verbose>3) tacWrite(&tac, stdout, TAC_FORMAT_TSV_UK, 1, NULL);
  if(tacWeightModerate(&tac, 0.0, 0, 0, status)!=TPCERROR_OK) {
    fprintf(stderr, "Function returns: %s\n", errorMsg(status->error));
    tacFree(&tac); return 15;
  }
  if(!doubleMatch(1.0, tac.w[1], 1.0E-20)) {
    if(verbose>2) tacWrite(&tac, stdout, TAC_FORMAT_TSV_UK, 1, NULL);
    tacFree(&tac); return 16;
  }
  if(tacWeightModerate(&tac, 0.1, 0, 0, status)!=TPCERROR_OK) {
    fprintf(stderr, "Function returns: %s\n", errorMsg(status->error));
    tacFree(&tac); return 17;
  }
  if(!doubleMatch(1.0, tac.w[1], 1.0E-20)) {
    if(verbose>2) tacWrite(&tac, stdout, TAC_FORMAT_TSV_UK, 1, NULL);
    tacFree(&tac); return 18;
  }


  if(verbose>1) printf("\n trying with a range of weights, but do not moderate\n");
  tac.weighting=WEIGHTING_ON_GENERAL;
  for(int i=0; i<tac.sampleNr; i++) tac.w[i]=(double)(1+i);
  if(verbose>3) tacWrite(&tac, stdout, TAC_FORMAT_TSV_UK, 1, NULL);
  if(tacWeightModerate(&tac, 0.0, 0, 0, status)!=TPCERROR_OK) {
    fprintf(stderr, "Function returns: %s\n", errorMsg(status->error));
    tacFree(&tac); return 21;
  }
  if(!doubleMatch(0.3333333, tac.w[0], 1.0E-06) || !doubleMatch(0.6666667, tac.w[1], 1.0E-06) || 
     !doubleMatch(1.0, tac.w[2], 1.0E-06) || !doubleMatch(1.333333333, tac.w[3], 1.0E-06) ||
     !doubleMatch(1.6666667, tac.w[4], 1.0E-06)) {
    if(verbose>2) tacWrite(&tac, stdout, TAC_FORMAT_TSV_UK, 1, NULL);
    tacFree(&tac); return 22;
  }

  if(verbose>1) printf("\n trying with a range of weights, and moderate\n");
  tac.weighting=WEIGHTING_ON_GENERAL;
  for(int i=0; i<tac.sampleNr; i++) tac.w[i]=(double)(1+i);
  if(verbose>3) tacWrite(&tac, stdout, TAC_FORMAT_TSV_UK, 1, NULL);
  if(tacWeightModerate(&tac, 0.6, 0, 0, status)!=TPCERROR_OK) {
    fprintf(stderr, "Function returns: %s\n", errorMsg(status->error));
    tacFree(&tac); return 31;
  }
  if(!doubleMatch(0.8333333, tac.w[0], 1.0E-06) || !doubleMatch(0.8333333, tac.w[1], 1.0E-06) || 
     !doubleMatch(0.8333333, tac.w[2], 1.0E-06) || !doubleMatch(1.1111111, tac.w[3], 1.0E-06) ||
     !doubleMatch(1.3888889, tac.w[4], 1.0E-06)) {
    if(verbose>2) tacWrite(&tac, stdout, TAC_FORMAT_TSV_UK, 1, NULL);
    tacFree(&tac); return 32;
  }

  if(verbose>1) printf("\n trying with a range of weights, one of which is zero, and moderate\n");
  for(int i=0; i<tac.sampleNr; i++) tac.w[i]=(double)(1+i);
  tac.w[1]=0.0;
  if(verbose>3) tacWrite(&tac, stdout, TAC_FORMAT_TSV_UK, 1, NULL);
  if(tacWeightModerate(&tac, 0.6, 0, 0, status)!=TPCERROR_OK) {
    fprintf(stderr, "Function returns: %s\n", errorMsg(status->error));
    tacFree(&tac); return 33;
  }
  if(!doubleMatch(0.8, tac.w[0], 1.0E-06) || !doubleMatch(0.0, tac.w[1], 1.0E-06) || 
     !doubleMatch(0.8, tac.w[2], 1.0E-06) || !doubleMatch(1.0666667, tac.w[3], 1.0E-06) ||
     !doubleMatch(1.33333333, tac.w[4], 1.0E-06)) {
    if(verbose>2) tacWrite(&tac, stdout, TAC_FORMAT_TSV_UK, 1, NULL);
    tacFree(&tac); return 34;
  }

  if(verbose>1) printf("\n trying with a range of weights, one of which is zero, fixing zeroes\n");
  for(int i=0; i<tac.sampleNr; i++) tac.w[i]=(double)(1+i);
  tac.w[1]=0.0;
  if(verbose>3) tacWrite(&tac, stdout, TAC_FORMAT_TSV_UK, 1, NULL);
  if(tacWeightModerate(&tac, 0.6, 1, 0, status)!=TPCERROR_OK) {
    fprintf(stderr, "Function returns: %s\n", errorMsg(status->error));
    tacFree(&tac); return 41;
  }
  if(!doubleMatch(0.8333333, tac.w[0], 1.0E-06) || !doubleMatch(0.8333333, tac.w[1], 1.0E-06) || 
     !doubleMatch(0.8333333, tac.w[2], 1.0E-06) || !doubleMatch(1.1111111, tac.w[3], 1.0E-06) ||
     !doubleMatch(1.3888889, tac.w[4], 1.0E-06)) {
    if(verbose>2) tacWrite(&tac, stdout, TAC_FORMAT_TSV_UK, 1, NULL);
    tacFree(&tac); return 42;
  }

  if(verbose>1) printf("\n trying with a range of weights, one of which is NaN, fixing it\n");
  for(int i=0; i<tac.sampleNr; i++) tac.w[i]=(double)(1+i);
  tac.w[1]=nan("");
  if(verbose>3) tacWrite(&tac, stdout, TAC_FORMAT_TSV_UK, 1, NULL);
  if(tacWeightModerate(&tac, 0.6, 0, 1, status)!=TPCERROR_OK) {
    fprintf(stderr, "Function returns: %s\n", errorMsg(status->error));
    tacFree(&tac); return 51;
  }
  if(!doubleMatch(0.8333333, tac.w[0], 1.0E-06) || !doubleMatch(0.8333333, tac.w[1], 1.0E-06) || 
     !doubleMatch(0.8333333, tac.w[2], 1.0E-06) || !doubleMatch(1.1111111, tac.w[3], 1.0E-06) ||
     !doubleMatch(1.3888889, tac.w[4], 1.0E-06)) {
    if(verbose>2) tacWrite(&tac, stdout, TAC_FORMAT_TSV_UK, 1, NULL);
    tacFree(&tac); return 52;
  }


  tacFree(&tac);

  statusSet(status, __func__, __FILE__, __LINE__, 0);
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
int test_tacSetWeights(
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  if(verbose>0) {
    printf("\n=====================================\n");
    printf("\n%s\n", __func__);
    printf("\n=====================================\n");
  }

  int ret, weightNr;

  if(verbose>1) printf("\n trying with empty data\n");
  ret=tacSetWeights(NULL, WEIGHTING_UNKNOWN, 0, NULL);
  if(ret==TPCERROR_OK) return(1);

  TAC tac; tacInit(&tac);

  ret=tacSetWeights(&tac, WEIGHTING_OFF, 0, NULL);
  if(ret==TPCERROR_OK) return(2);
  ret=tacSetWeights(&tac, WEIGHTING_ON_F, 999, NULL);
  if(ret==TPCERROR_OK) return(3);
  if(verbose>2) printf("\n ok \n");

  if(verbose>1) printf("\n creating test data\n");
  if(create_tac(&tac)!=TPCERROR_OK) {tacFree(&tac); return(10);}
  if(verbose>3) tacWrite(&tac, stdout, TAC_FORMAT_TSV_UK, 1, NULL);
  if(verbose>2) printf("\n ok \n");

  if(verbose>1) printf("\n trying to remove weights\n");
  ret=tacSetWeights(&tac, WEIGHTING_OFF, 999, status);
  if(ret!=TPCERROR_OK) {tacFree(&tac); return(101);}
  if(tacIsWeighted(&tac)) {tacFree(&tac); return(102);}
  if(verbose>2) printf("\n ok \n");

  if(verbose>1) printf("\n trying to set general weights\n");
  ret=tacSetWeights(&tac, WEIGHTING_ON_GENERAL, 999, status);
  if(ret==TPCERROR_OK) {tacFree(&tac); return(201);}
  if(tacIsWeighted(&tac)) {tacFree(&tac); return(202);}
  if(verbose>2) printf("\n ok \n");


  if(verbose>1) printf("\n trying to set weights based on frames\n");
  tac.isframe=1;
  ret=tacSetWeights(&tac, WEIGHTING_ON_F, 999, status);
  if(ret!=TPCERROR_OK) {tacFree(&tac); return(301);}
  if(!tacIsWeighted(&tac)) {tacFree(&tac); return(302);}
  if(verbose>3) tacWrite(&tac, stdout, TAC_FORMAT_TSV_UK, 1, NULL);
  for(int i=0; i<tac.sampleNr; i++) {
    if(!doubleMatch(tac.w[i], 1.0, 0.0001)) {tacFree(&tac); return 303;}
  }
  if(verbose>2) printf("\n ok \n");

  if(verbose>1) printf("\n applying weightNr to previously set weights\n");
  weightNr=2;
  ret=tacSetWeights(&tac, WEIGHTING_UNKNOWN, weightNr, status);
  if(ret!=TPCERROR_OK) {tacFree(&tac); return(305);}
  if(!tacIsWeighted(&tac)) {tacFree(&tac); return(306);}
  if(verbose>3) tacWrite(&tac, stdout, TAC_FORMAT_TSV_UK, 1, NULL);
  {
    int i;
    for(i=0; i<weightNr; i++) {
      if(!doubleMatch(tac.w[i], 1.0, 0.0001)) {tacFree(&tac); return 307;}
    }
    for( ; i<tac.sampleNr; i++) {
      if(!doubleMatch(tac.w[i], 0.0, 0.0001)) {tacFree(&tac); return 308;}
    }
  }
  if(tacWSampleNr(&tac)!=(unsigned int)weightNr) {tacFree(&tac); return 309;}
  if(verbose>2) printf("\n ok \n");


  if(verbose>1) printf("\n trying to set weights based on frames, but not for all frames\n");
  weightNr=3;
  ret=tacSetWeights(&tac, WEIGHTING_ON_F, weightNr, status);
  if(ret!=TPCERROR_OK) {tacFree(&tac); return(311);}
  if(!tacIsWeighted(&tac)) {tacFree(&tac); return(312);}
  if(verbose>3) tacWrite(&tac, stdout, TAC_FORMAT_TSV_UK, 1, NULL);
  {
    int i;
    for(i=0; i<weightNr; i++) {
      if(!doubleMatch(tac.w[i], 1.0, 0.0001)) {tacFree(&tac); return 313;}
    }
    for( ; i<tac.sampleNr; i++) {
      if(!doubleMatch(tac.w[i], 0.0, 0.0001)) {tacFree(&tac); return 314;}
    }
  }
  if(tacWSampleNr(&tac)!=(unsigned int)weightNr) {tacFree(&tac); return 315;}
  if(verbose>2) printf("\n ok \n");


  if(verbose>1) printf("\n trying to set weights based on decay and frames\n");
  tac.isframe=1;
  weightNr=tac.sampleNr;
  ret=tacSetWeights(&tac, WEIGHTING_ON_FD, weightNr, status);
  if(ret!=TPCERROR_OK) {tacFree(&tac); return(401);}
  if(!tacIsWeighted(&tac)) {tacFree(&tac); return(402);}
  if(verbose>3) tacWrite(&tac, stdout, TAC_FORMAT_TSV_UK, 1, NULL);
  if(!doubleMatch(tac.w[0], 1.07, 0.01)) {tacFree(&tac); return 403;}
  if(!doubleMatch(tac.w[tac.sampleNr-1], 0.93, 0.01)) {tacFree(&tac); return 404;}
  if(verbose>2) printf("\n ok \n");

  if(verbose>1) printf("\n frame mid times should be fine\n");
  tac.isframe=0;
  ret=tacSetWeights(&tac, WEIGHTING_ON_FD, weightNr, status);
  if(ret!=TPCERROR_OK) {tacFree(&tac); return(411);}
  if(!tacIsWeighted(&tac)) {tacFree(&tac); return(412);}
  if(verbose>3) tacWrite(&tac, stdout, TAC_FORMAT_TSV_UK, 1, NULL);
  if(!doubleMatch(tac.w[0], 1.07, 0.01)) {tacFree(&tac); return 413;}
  if(!doubleMatch(tac.w[tac.sampleNr-1], 0.93, 0.01)) {tacFree(&tac); return 414;}
  if(verbose>2) printf("\n ok \n");

  if(verbose>1) printf("\n ... but missing isotope is not fine\n");
  tacSetIsotope(&tac, ISOTOPE_UNKNOWN);
  tac.isframe=1;
  weightNr=tac.sampleNr;
  ret=tacSetWeights(&tac, WEIGHTING_ON_FD, weightNr, status);
  if(ret!=TPCERROR_UNKNOWN_ISOTOPE) {tacFree(&tac); return(421);}
  if(tacIsWeighted(&tac)) {tacFree(&tac); return(422);}
  tacSetIsotope(&tac, ISOTOPE_C_11);
  if(verbose>2) printf("\n ok \n");


  if(verbose>1) printf("\n trying to set weights based on counts when TAC is not SIF\n");
  tacSetIsotope(&tac, ISOTOPE_C_11);
  tac.isframe=1;
  for(int j=0; j<tac.tacNr; j++) tac.c[j].size=0.0;
  if(verbose>10) {
    FILE *fp=fopen("tacsetweights.tac", "w");
    tacWrite(&tac, fp, TAC_FORMAT_PMOD, 1, NULL);
    fclose(fp);
  }
  weightNr=tac.sampleNr;
  ret=tacSetWeights(&tac, WEIGHTING_ON_COUNTS, weightNr, status);
  if(ret!=TPCERROR_OK) {tacFree(&tac); return(601);}
  if(verbose>3) tacWrite(&tac, stdout, TAC_FORMAT_TSV_UK, 1, NULL);
  if(!doubleMatch(tac.w[0], 3.185159, 0.01)) {tacFree(&tac); return 602;}
  if(!doubleMatch(tac.w[tac.sampleNr-1], 0.2450123, 0.001)) {tacFree(&tac); return 603;}
  if(verbose>2) printf("\n ok \n");

  if(verbose>1) printf("\n trying to remove weights\n");
  weightNr=3;
  ret=tacSetWeights(&tac, WEIGHTING_OFF, weightNr, status);
  if(ret!=TPCERROR_OK) {tacFree(&tac); return(611);}
  if(tacIsWeighted(&tac)) {tacFree(&tac); return(612);}
  if(verbose>3) tacWrite(&tac, stdout, TAC_FORMAT_TSV_UK, 1, NULL);
  {
    int i;
    for(i=0; i<weightNr; i++) {
      if(!doubleMatch(tac.w[i], 1.0, 0.0001)) {tacFree(&tac); return 613;}
    }
    for( ; i<tac.sampleNr; i++) {
      if(!doubleMatch(tac.w[i], 0.0, 0.0001)) {tacFree(&tac); return 614;}
    }
  }
  if(verbose>2) printf("\n ok \n");

  tacFree(&tac);

  {
  if(verbose>1) printf("\n trying to set weights based on counts when TAC is actually SIF\n");

  if(verbose>1) printf("creating test data\n");
  TAC sif; tacInit(&sif);
  if(create_sif(&sif)!=TPCERROR_OK) {tacFree(&sif); return(1000);}
  sif.format=TAC_FORMAT_SIF;
  tacSetIsotope(&sif, ISOTOPE_O_15);
  if(verbose>3) tacWrite(&sif, stdout, TAC_FORMAT_TSV_UK, 1, NULL);

  ret=tacSetWeights(&sif, WEIGHTING_ON_COUNTS, 9999, NULL);
  if(ret!=TPCERROR_OK) {
    fprintf(stderr, "Error: %s\n", errorMsg(status->error)); tacFree(&sif); return(1001);}
  if(!tacIsWeighted(&sif)) {tacFree(&sif); return(1002);}
  if(verbose>3) tacWrite(&sif, stdout, TAC_FORMAT_TSV_UK, 1, NULL);
  if(!doubleMatch(2.67523, sif.w[0], 1.0E-04) || !doubleMatch(1.01146, sif.w[1], 1.0E-04) || 
     !doubleMatch(0.597521, sif.w[2], 1.0E-04) || !doubleMatch(0.410749, sif.w[3], 1.0E-04) ||
     !doubleMatch(0.305047, sif.w[4], 1.0E-04)) {
    tacFree(&sif); return(1003);
  }
  if(!doubleMatch(1.0, doubleMean(sif.w, 5), 1.0E-10)) {tacFree(&sif); return 1004;}
  tacFree(&sif);
  }

  statusSet(status, __func__, __FILE__, __LINE__, 0);
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
