/** @file lisim.c
 *  @brief Linear interpolation for simulation.
 */
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <time.h>
#include <string.h>
/*****************************************************************************/
#include "tpcextensions.h"
#include "tpcstatist.h"
#include "tpctac.h"
#include "tpcli.h"
/*****************************************************************************/
#include "tpctacmod.h"
/*****************************************************************************/

/*****************************************************************************/
/** @brief Modify input TAC based on tissue TAC, for use in simulations.
    @note NOT FUNCTIONAL/TESTED YET!
    @sa tacReadModelingInput, tacReadModelingData, tacInterpolateInto, 
     tacInterpolateToEqualLengthFrames
    @return enum tpcerror (TPCERROR_OK when successful).
    @author Vesa Oikonen
 */ 
int tacInput2sim(
  /** Pointer to original input TAC structure. Not modified. Data must not have missing values. */
  TAC *itac,
  /** Pointer to tissue TAC structure. Not modified. Data must not have missing values. */
  TAC *ttac,
  /** Pointer to the new input data, containing input data with new sample times. */
  TAC *stac,
  /** Pointer to status data; obligatory. */
  TPCSTATUS *status
) {
  if(status==NULL) return TPCERROR_FAIL;
  int verbose=status->verbose;
  if(verbose>0) printf("%s(itac, ttac, ...)\n", __func__);

  /* Check the input */
  if(itac==NULL || ttac==NULL || stac==NULL) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_FAIL);
    return TPCERROR_FAIL;
  }
  if(itac->sampleNr<1 || ttac->sampleNr<1 || itac->tacNr<1) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_NO_DATA);
    return TPCERROR_NO_DATA;
  }

  /* Delete any previous data */
  tacFree(stac);

  /* Make a working copy of input data */
  TAC inp; tacInit(&inp);
  status->error=tacDuplicate(itac, &inp); if(status->error!=TPCERROR_OK) return(status->error);
  /* Make sure that input data contains also frame mid times */
  if(inp.isframe) tacSetX(&inp, NULL);
  /* Convert time units to TTAC units */
  tacXUnitConvert(&inp, ttac->tunit, status);
  if(status->error!=TPCERROR_OK) {tacFree(&inp); return(status->error);}

  /* Sort the input data by sample times */
  tacSortByTime(&inp, status); if(status->error!=TPCERROR_OK) {tacFree(&inp); return(status->error);}
  /* Add zero sample if necessary */
  tacAddZeroSample(&inp, status);
  if(status->error!=TPCERROR_OK) {tacFree(&inp); return(status->error);}

  /* Make a combined list of sample times based on input and tissue TACs */
  unsigned int maxn=inp.sampleNr+ttac->sampleNr;
  if(inp.isframe) maxn+=2*inp.sampleNr;
  if(ttac->isframe) maxn+=ttac->sampleNr;
  double x[maxn];

  int n=0;
  for(int i=0; i<inp.sampleNr; i++) if(isfinite(inp.x[i])) {
    int j; for(j=0; j<n; j++) if(fabs(x[j]-inp.x[i])<1.0E-20) break;
    if(j==n) x[n++]=inp.x[i];
  }
  if(inp.isframe) {
    for(int i=0; i<inp.sampleNr; i++) if(isfinite(inp.x1[i])) {
      int j; for(j=0; j<n; j++) if(fabs(x[j]-inp.x1[i])<1.0E-20) break;
      if(j==n) x[n++]=inp.x1[i];
    }
    for(int i=0; i<inp.sampleNr; i++) if(isfinite(inp.x2[i])) {
      int j; for(j=0; j<n; j++) if(fabs(x[j]-inp.x2[i])<1.0E-20) break;
      if(j==n) x[n++]=inp.x2[i];
    }
  }
  if(ttac->isframe) {
    for(int i=0; i<ttac->sampleNr; i++) if(isfinite(ttac->x2[i])) {
      int j; for(j=0; j<n; j++) if(fabs(x[j]-ttac->x2[i])<1.0E-20) break;
      if(j==n) x[n++]=ttac->x2[i];
    }
    for(int i=0; i<ttac->sampleNr; i++) if(isfinite(ttac->x2[i])) {
      int j; for(j=0; j<n; j++) if(fabs(x[j]-ttac->x2[i])<1.0E-20) break;
      if(j==n) x[n++]=ttac->x2[i];
    }
  } else {
    for(int i=0; i<ttac->sampleNr; i++) if(isfinite(ttac->x[i])) {
      int j; for(j=0; j<n; j++) if(fabs(x[j]-ttac->x[i])<1.0E-20) break;
      if(j==n) x[n++]=ttac->x[i];
    }
  }
  if(verbose>2) printf("n=%d\n", n);
  statSortDouble(x, n, 0);

  /* Put new sample times into output data */
  status->error=tacDuplicate(itac, stac);
  if(status->error!=TPCERROR_OK) {tacFree(&inp); return(status->error);}
  status->error=tacAllocateMoreSamples(stac, n-stac->sampleNr);
  if(status->error!=TPCERROR_OK) {tacFree(&inp); return(status->error);}
  stac->sampleNr=n; stac->tunit=inp.tunit; stac->isframe=0;
  for(int i=0; i<n; i++) stac->x[i]=x[i];

  /* Interpolate input data into the new sample times */
  for(int i=0; i<inp.tacNr; i++) {
    status->error=liInterpolate(inp.x, inp.c[i].y, inp.sampleNr,
                            stac->x, stac->c[i].y, NULL, NULL, stac->sampleNr, 4, 1, verbose-10);
    if(status->error!=TPCERROR_OK) break;
  }
  if(status->error!=TPCERROR_OK) {tacFree(&inp); tacFree(stac); return(status->error);}

  tacFree(&inp);

  statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_OK);
  return(TPCERROR_OK);
}
/*****************************************************************************/

/*****************************************************************************/
