/** @file models.c
 *  @brief Listing of PET models.
 *  @copyright (c) Turku PET Centre
 *  @author Vesa Oikonen
 */
/// @cond
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <time.h>
#include <string.h>
/*****************************************************************************/
#include "tpcextensions.h"
#include "tpcmodels.h"
#include "tpcpar.h"
/*****************************************************************************/

/*****************************************************************************/
static char *info[] = {
  "List the PET models and mathematical functions that are identified by",
  "most TPC software.",
  " ",
  "Usage: @P [options] [> outputfile]",
  " ",
  "Options:",
  " -model=<code>",
  "     Information in specified model only; by default all are listed.",
  " -stdoptions", // List standard options like --help, -v, etc
  " ",
  "See also: halflife, fit2dat, sim_rtcm",
  " ",
  "Keywords: modelling, fitting, parameters",
  0};
/*****************************************************************************/

/*****************************************************************************/
/* Turn on the globbing of the command line, since it is disabled by default in
   mingw-w64 (_dowildcard=0); in MinGW32 define _CRT_glob instead, if necessary;
   In Unix&Linux wildcard command line processing is enabled by default. */
/*
#undef _CRT_glob
#define _CRT_glob -1
*/
int _dowildcard = -1;
/*****************************************************************************/

/*****************************************************************************/
/**
 *  Main
 */
int main(int argc, char **argv)
{
  int ai, help=0, version=0, verbose=1;
  unsigned int model=0;
  char *cptr;


  /*
   *  Get arguments
   */
  /* Options */
  for(ai=1; ai<argc; ai++) if(*argv[ai]=='-') {
    if(tpcProcessStdOptions(argv[ai], &help, &version, &verbose)==0) continue;
    cptr=argv[ai]+1; if(*cptr=='-') cptr++; if(!*cptr) continue;
    if(strncasecmp(cptr, "MODEL=", 6)==0) {
      model=modelCodeIndex(cptr+6);
      continue;
    }
    fprintf(stderr, "Error: invalid option '%s'\n", argv[ai]);
    return(1);
  }

  /* Print help or version? */
  if(help==2) {tpcHtmlUsage(argv[0], info, ""); return(0);}
  if(help) {tpcPrintUsage(argv[0], info, stdout); return(0);}
  if(version) {tpcPrintBuild(argv[0], stdout); return(0);}

  /* Non-options */
  for(ai=1; ai<argc; ai++) if(*argv[ai]!='-') {
    fprintf(stderr, "Error: invalid argument '%s'\n", argv[ai]);
    return(1);
  }

  if(verbose>1) {
    if(model>0) printf("model := %u\n", model);
  }

  /* Get the number of listed models */
  unsigned int n=modelNr();
  if(verbose>0) printf("modelNr := %d\n", n);

  /* Print specified model, or all models */
  if(model!=0) {
    fprintf(stdout, "code := %s\n", modelCode(model));
    fprintf(stdout, "desc := %s\n", modelDesc(model));
  } else {
    for(unsigned int i=1; i<n; i++) {
      fprintf(stdout, "code[%d] := %s\n", i, modelCode(i));
      fprintf(stdout, "desc[%d] := %s\n", i, modelDesc(i));
    }
  }
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/// @endcond
