/** @file parformat.c
 *  @brief View or set PAR data units.
 *  @copyright (c) Turku PET Centre
 *  @author Vesa Oikonen
 *  @test Test more file formats.
 */
/// @cond
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
/*****************************************************************************/
#include "tpcextensions.h"
#include "tpcift.h"
#include "tpctac.h"
#include "tpcpar.h"
/*****************************************************************************/

/*****************************************************************************/
static char *info[] = {
  "Convert model or function fit parameter data in filename1 into filename2",
  "in specified file format.",
  "Data can be written only in a few formats (listed below), but more",
  "file formats can be read.",
  " ",
  "Usage: @P [options] filename1 [filename2]",
  " ",
  "Options:",
  " -f=<format-id>",
  "     Accepted format-id's:",
  "     CSV-INT   - CSV format with semicolons and decimal commas.",
  "     CSV-UK    - CSV format with commas and decimal points.",
  "     TSV-INT   - TSV format with tabs and decimal commas.",
  "     TSV-UK    - TSV format with tabs and decimal points.",
  "     RES       - TPC RES format (deprecated).",
  "     FIT       - TPC FIT format (deprecated).",
  "     IFT       - Interfile type format.",
  "     XML       - MS Excel compatible XML format.",
  "     Without this option, only the format of the given file is shown.",
  " -hdr=<Yes|no>",
  "     Extra information is (y, default) or is not stored (n) in lines",
  "     starting with '#' character; not effective with all formats.",
  " -units=<Yes|no>",
  "     Keep (y, default) or remove (n) parameter units.",
  " -stdoptions", // List standard options like --help, -v, etc
  " ",
  "Example: convert RES file into CSV format for exporting to Excel",
  "     @P -f=CSV-UK -hdr=no iea446ki.res iea446ki.csv",
  " ",
  "See also: parsort, paradd, parget, csvtrps, reslist, parmatch, rescoll",
  " ",
  "Keywords: parameter, tool, format, CSV, RES, FIT",
  0};
/*****************************************************************************/

/*****************************************************************************/
/* Turn on the globbing of the command line, since it is disabled by default in
   mingw-w64 (_dowildcard=0); in MinGW32 define _CRT_glob instead, if necessary;
   In Unix&Linux wildcard command line processing is enabled by default. */
/*
#undef _CRT_glob
#define _CRT_glob -1
*/
int _dowildcard = -1;
/*****************************************************************************/

/*****************************************************************************/
/**
 *  Main
 */
int main(int argc, char **argv)
{
  int ai, help=0, version=0, verbose=1;
  int ret;
  int new_format=PAR_FORMAT_UNKNOWN;
  int save_header=1;
  int keep_units=1;
  char *cptr, parfile1[FILENAME_MAX], parfile2[FILENAME_MAX];
  PAR par;

  
  /*
   *  Get arguments
   */
  if(argc==1) {tpcPrintUsage(argv[0], info, stderr); return(1);}
  parInit(&par);
  parfile1[0]=parfile2[0]=(char)0;
  /* Options */
  for(ai=1; ai<argc; ai++) if(*argv[ai]=='-') {
    if(tpcProcessStdOptions(argv[ai], &help, &version, &verbose)==0) continue;
    cptr=argv[ai]+1; if(*cptr=='-') cptr++; if(!*cptr) continue;
    if(strncasecmp(cptr, "F=", 2)==0) {
      new_format=parFormatIdentify(cptr+2);
      if(new_format!=PAR_FORMAT_UNKNOWN) continue;
    } else if(strncasecmp(cptr, "FORMAT=", 7)==0) {
      new_format=parFormatIdentify(cptr+7);
      if(new_format!=PAR_FORMAT_UNKNOWN) continue;
    } else if(strncasecmp(cptr, "HDR=", 4)==0) {
      cptr+=4;
      if(strncasecmp(cptr, "YES", 1)==0) {save_header=1; continue;}
      else if(strncasecmp(cptr, "NO", 1)==0) {save_header=0; continue;}
    } else if(strncasecmp(cptr, "HEADER=", 7)==0) {
      cptr+=7;
      if(strncasecmp(cptr, "YES", 1)==0) {save_header=1; continue;}
      else if(strncasecmp(cptr, "NO", 1)==0) {save_header=0; continue;}
    } else if(strncasecmp(cptr, "UNITS=", 6)==0) {
      cptr+=6;
      if(strncasecmp(cptr, "YES", 1)==0) {keep_units=1; continue;}
      else if(strncasecmp(cptr, "NO", 1)==0) {keep_units=0; continue;}
    }
    fprintf(stderr, "Error: invalid option '%s'.\n", argv[ai]);
    return(1);
  } else break; // tac name argument may start with '-'
  
  /* Print help or version? */
  if(help==2) {tpcHtmlUsage(argv[0], info, ""); return(0);}
  if(help) {tpcPrintUsage(argv[0], info, stdout); return(0);}
  if(version) {tpcPrintBuild(argv[0], stdout); return(0);}

  /* Process other arguments, starting from the first non-option */
  for(; ai<argc; ai++) {
    /* It should be the PAR filename */
    if(!parfile1[0]) {
      strlcpy(parfile1, argv[ai], FILENAME_MAX);
    } else if(!parfile2[0] && new_format!=PAR_FORMAT_UNKNOWN) {
      strlcpy(parfile2, argv[ai], FILENAME_MAX);
    } else {
      fprintf(stderr, "Error: invalid argument '%s'.\n", argv[ai]);
      return(1);
    }
  }  
  /* Is something missing? */
  if(!parfile1[0]) {tpcPrintUsage(argv[0], info, stdout); return(1);}

  /* In verbose mode print arguments and options */
  if(verbose>1) {
    printf("new_format := %s\n", parFormattxt(new_format));
    printf("save_header := %d\n", save_header);
    printf("keep_units := %d\n", keep_units);
    printf("parfile1 := %s\n", parfile1);
    if(parfile2[0]) printf("parfile2 := %s\n", parfile2);
  }

  TPCSTATUS status; statusInit(&status);
  statusSet(&status, __func__, __FILE__, __LINE__, TPCERROR_OK);
  status.verbose=verbose-1;

  /*
   *  Read PAR data
   */
  if(verbose>1) printf("reading %s\n", parfile1);
  ret=parRead(&par, parfile1, &status);
  if(ret) {
    fprintf(stderr, "Error: %s (%s)\n", errorMsg(status.error), parfile1);
    parFree(&par);
    return(2);
  }
  
  /* If new format was not specified, then just print the current format */
  if(new_format==PAR_FORMAT_UNKNOWN) {
    fprintf(stdout, "format := %s\n", parFormattxt(par.format));
    parFree(&par);
    return 0;
  }
  
  /* Remove units, if user does not want to keep those */
  if(keep_units==0) {
    for(int i=0; i<par.parNr; i++) par.n[i].unit=UNIT_UNKNOWN;
  }

  /* If output file name was not given by user, then make it here */
  if(!parfile2[0]) {
    strcpy(parfile2, parfile1);
    filenameRmPath(parfile2); filenameRmExtension(parfile2);
    strcat(parfile2, parDefaultExtension(new_format));
  }
  if(verbose>1) {
    printf("parfile2 := %s\n", parfile2);
  }
  
  /* Print current extra headers for testing */
  if(verbose>4) {
    printf("\n---- par.h ----\n");
    iftWrite(&par.h, stdout, NULL);
    printf("---------------\n\n");
  }

  /* Try to save the data in the new format */
  if(verbose>1) printf("writing %s\n", parfile2);
  FILE *fp; fp=fopen(parfile2, "w");
  if(fp==NULL) {
    fprintf(stderr, "Error: cannot open file for writing (%s)\n", parfile2);
    parFree(&par); return(5);
  }
  ret=parWrite(&par, fp, new_format, save_header, &status);
  fclose(fp);
  parFree(&par);
  if(ret==TPCERROR_UNSUPPORTED) {
    fprintf(stderr, "Error: writing format %s is not supported.\n", parFormattxt(new_format));
    return(6);
  }
  if(ret!=TPCERROR_OK) {
    fprintf(stderr, "Error: %s\n", errorMsg(status.error));
    return(7);
  }
  if(verbose>0) printf("  %s written.\n", parfile2);
  
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/// @endcond
