/** @file parrenp.c
 *  @brief Rename parameter in PAR file.
 *  @copyright (c) Turku PET Centre
 *  @author Vesa Oikonen
 */
/// @cond
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
/*****************************************************************************/
#include "tpcextensions.h"
#include "tpcpar.h"
/*****************************************************************************/

/*****************************************************************************/
static char *info[] = {
  "Rename the specified parameter inside a parameter file.",
  " ",
  "Usage: @P [options] file [oldname newname]",
  " ",
  "Enter either the number or name of the parameter to be renamed, and",
  "the new parameter name. Only the first matching parameter name is renamed.",
  "If neither is given, then program lists current parameter names.",
  "Note that not all parameter file formats can store parameter names.",
  " ",
  "Options:",
  " -dry",
  "     Nothing is actually renamed, application only lists what it would do.",
  " -stdoptions", // List standard options like --help, -v, etc
  " ",
  "See also: paradd, parcoll, iftedit, parformat, tacren",
  " ",
  "Keywords: parameter, tool, rename, simulation",
  0};
/*****************************************************************************/

/*****************************************************************************/
/* Turn on the globbing of the command line, since it is disabled by default in
   mingw-w64 (_dowildcard=0); in MinGW32 define _CRT_glob instead, if necessary;
   In Unix&Linux wildcard command line processing is enabled by default. */
/*
#undef _CRT_glob
#define _CRT_glob -1
*/
int _dowildcard = -1;
/*****************************************************************************/

/*****************************************************************************/
/**
 *  Main
 */
int main(int argc, char **argv)
{
  int ai, help=0, version=0, verbose=1;
  int dryMode=0;
  char parfile[FILENAME_MAX], oldname[MAX_PARNAME_LEN], newname[MAX_PARNAME_LEN];

  
  /*
   *  Get arguments
   */
  if(argc==1) {tpcPrintUsage(argv[0], info, stderr); return(1);}
  parfile[0]=oldname[0]=newname[0]=(char)0;
  /* Options */
  for(ai=1; ai<argc; ai++) if(*argv[ai]=='-') {
    if(tpcProcessStdOptions(argv[ai], &help, &version, &verbose)==0) continue;
    char *cptr=argv[ai]+1; if(*cptr=='-') cptr++; if(!*cptr) continue;
    if(strcasecmp(cptr, "DRY")==0) {
      dryMode=1; continue;
    }
    fprintf(stderr, "Error: invalid option '%s'\n", argv[ai]);
    return(1);
  } else break; // tac name argument may start with '-'

  TPCSTATUS status; statusInit(&status);
  statusSet(&status, __func__, __FILE__, __LINE__, TPCERROR_OK);
  status.verbose=verbose-5;
  
  /* Print help or version? */
  if(help==2) {tpcHtmlUsage(argv[0], info, ""); return(0);}
  if(help) {tpcPrintUsage(argv[0], info, stdout); return(0);}
  if(version) {tpcPrintBuild(argv[0], stdout); return(0);}

  /* Process other arguments, starting from the first non-option */
  if(ai<argc) strlcpy(parfile, argv[ai++], FILENAME_MAX);
  if(ai<argc) strlcpy(oldname, argv[ai++], MAX_PARNAME_LEN);
  if(ai<argc) strlcpy(newname, argv[ai++], MAX_PARNAME_LEN);
  if(ai<argc) {fprintf(stderr, "Error: too many arguments: '%s'.\n", argv[ai]); return(1);}

  /* In verbose mode print arguments and options */
  if(verbose>1) {
    printf("parfile := %s\n", parfile);
    printf("dryMode := %d\n", dryMode);
    if(oldname[0]) printf("oldname := %s\n", oldname);
    if(newname[0]) printf("newname := %s\n", newname);
    fflush(stdout);
  }

  /* Is something missing? */
  if(!parfile[0]) {tpcPrintUsage(argv[0], info, stderr); return(1);}
  if(oldname[0] && !newname[0]) {
    fprintf(stderr, "Error: new parameter name missed.\n");
    return(1);
  }


  /* 
   *  Read the file
   */
  if(verbose>1) printf("reading %s\n", parfile);
  PAR par; parInit(&par);
  if(parRead(&par, parfile, &status)!=TPCERROR_OK) {
    fprintf(stderr, "Error: %s\n", errorMsg(status.error));
    parFree(&par); return(2);
  }
  if(verbose>2) {
    printf("fileformat := %s\n", parFormattxt(par.format));
    printf("parNr := %d\n", par.parNr);
    printf("tacNr := %d\n", par.tacNr);
  }

  /*
   *  List the parameter names and quit, if parameter names were not given
   */
  if(!oldname[0]) {
    fflush(stdout);
    for(int i=0; i<par.parNr; i++)
      printf("%s\n", par.n[i].name);
    fflush(stdout);
    parFree(&par);
    return(0);
  }


  /* 
   *  Select the parameter to be renamed
   */
  int parNr=parSelectParameters(&par, oldname, 1, &status);
  if(verbose>1) printf("%d parameter(s) match '%s'\n", parNr, oldname);
  if(parNr==0) {
    fprintf(stderr, "Error: no parameter matches '%s'.\n", oldname);
    parFree(&par); return(3);
  }



  /*
   *  Rename the first selected parameter
   */
  for(int i=0; i<par.parNr; i++) if(par.n[i].sw) {
    if(verbose>0 || dryMode) {
      fprintf(stdout, "%s -> %s\n", par.n[i].name, newname);
      fflush(stdout);
    }
    strlcpy(par.n[i].name, newname, MAX_PARNAME_LEN);
    break; // only rename the first match
  }

  if(dryMode) {parFree(&par); return(0);} 

  /*
   *  Save modified data 
   */
  if(verbose>1) printf("writing %s\n", parfile);
  {
    FILE *fp=fopen(parfile, "w");
    if(fp==NULL) {
      fprintf(stderr, "Error: cannot open file for writing.\n");
      parFree(&par); return(11);
    }
    int ret=parWrite(&par, fp, PAR_FORMAT_UNKNOWN, 1, &status);
    fclose(fp); parFree(&par);
    if(ret!=TPCERROR_OK) {
      fprintf(stderr, "Error: %s\n", errorMsg(status.error));
      return(12);
    }
    if(verbose>0) printf("modified file written.\n");
  }

  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/// @endcond
