/** @file sifisot.c
    @brief Add isotope to SIF files.
    @copyright (c) Turku PET Centre
    @author Vesa Oikonen
 */
/// @cond
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include <unistd.h>
/*****************************************************************************/
#include "tpcextensions.h"
#include "tpccsv.h"
#include "tpcift.h"
#include "tpctac.h"
/*****************************************************************************/

/*****************************************************************************/
static char *info[] = {
  "Add isotope to the scan information files (SIF), if they do not contain",
  "it already. If isotope code is not given by user, then program shows",
  "the current isotope in SIF.",
  " ",
  "Usage: @P [options] filename(s) [isotope code]",
  " ",
  "Options:",
  " --force",
  "     Isotope is set even if it already exists in SIF.",
  " -stdoptions", // List standard options like --help, -v, etc
  " ",
  "Example:",
  "     @P *.sif C-11",
  " ",
  "See also: sifcat, taclist, eframe, tacframe, tacweigh",
  " ",
  "Keywords: SIF, isotope, physical decay, image, Analyze",
  0};
/*****************************************************************************/

/*****************************************************************************/
/* Turn on the globbing of the command line, since it is disabled by default in
   mingw-w64 (_dowildcard=0); in MinGW32 define _CRT_glob instead, if necessary;
   In Unix&Linux wildcard command line processing is enabled by default. */
/*
#undef _CRT_glob
#define _CRT_glob -1
*/
int _dowildcard = -1;
/*****************************************************************************/

/*****************************************************************************/
/*
 *  Main
 */
int main(int argc, char *argv[])
{
  int ai, help=0, version=0, verbose=1;
  char *cptr, siffile[FILENAME_MAX], isotope_name[MAX_ISOTOPE_LEN];
  int ret, si, ffi=0, fileNr=0, isotope_code=0, forceMode=0;


  /*
   *  Get arguments
   */
  if(argc==1) {tpcPrintUsage(argv[0], info, stderr); return(1);}
  siffile[0]=isotope_name[0]=(char)0;
  /* Options */
  for(ai=1; ai<argc; ai++) if(*argv[ai]=='-') {
    if(tpcProcessStdOptions(argv[ai], &help, &version, &verbose)==0) continue;
    cptr=argv[ai]+1; if(*cptr=='-') cptr++; if(!*cptr) continue;
    if(strcasecmp(cptr, "F")==0 || strcasecmp(cptr, "FORCE")==0) {
      forceMode=1; continue;
    }
    fprintf(stderr, "Error: invalid option '%s'\n", argv[ai]);
    return(1);
  } else break; // tac name argument may start with '-'
  
  /* Print help or version? */
  if(help==2) {tpcHtmlUsage(argv[0], info, ""); return(0);}
  if(help) {tpcPrintUsage(argv[0], info, stdout); return(0);}
  if(version) {tpcPrintBuild(argv[0], stdout); return(0);}

  TPCSTATUS status; statusInit(&status);
  statusSet(&status, __func__, __FILE__, __LINE__, TPCERROR_OK);
  status.verbose=verbose-1;

  /* Process other arguments, starting from the first non-option */
  fileNr=ffi=0;
  for(; ai<argc; ai++) {
    if(ffi<1) ffi=ai;
    fileNr++;
  }
  /* Check that we got at least something */
  if(fileNr<1) {tpcPrintUsage(argv[0], info, stderr); return(1);}
  /* Last argument may or may not be the isotope name */
  if(fileNr>1 && access(argv[ai-1],0)==-1) {
    isotope_code=isotopeIdentify(argv[ai-1]);
    if(isotope_code!=ISOTOPE_UNKNOWN) {
      strcpy(isotope_name, isotopeName(isotope_code));
      fileNr--;
    }
  }

  /* In verbose mode print arguments and options */
  if(verbose>1) {
    printf("fileNr := %d\n", fileNr);
    if(isotope_code!=ISOTOPE_UNKNOWN) 
      printf("isotope_name := %s\n", isotope_name);
    printf("forceMode := %d\n", forceMode);
    fflush(stdout);
  }


  /*
   *  Process each SIF file
   */
  char old_isotope_name[MAX_ISOTOPE_LEN+1];
  int old_isotope_code=0;
  TAC tac; tacInit(&tac);
  for(ai=ffi, si=0; ai<argc && si<fileNr; ai++) {
    strlcpy(siffile, argv[ai], FILENAME_MAX); si++;

    /* Print filename */
    if(verbose>0) fprintf(stdout, "SIF_filename := %s\n", siffile);

    /* Read SIF file as TAC */
    ret=tacRead(&tac, siffile, &status);
    if(ret!=TPCERROR_OK) {
      fprintf(stderr, "Error: %s\n", errorMsg(status.error));
      if(verbose>1) printf("ret := %d\n", ret);
      tacFree(&tac); return(2);
    }
    if(tac.format!=TAC_FORMAT_SIF) {
      fprintf(stderr, "Error: file is not in SIF format.\n");
      tacFree(&tac); return(3);
    }

    /* Read current isotope */
    old_isotope_code=ISOTOPE_UNKNOWN;
    ret=tacGetHeaderIsotope(&tac.h, old_isotope_name, &status);
    if(ret==TPCERROR_OK) old_isotope_code=isotopeIdentify(old_isotope_name);
    strcpy(old_isotope_name, isotopeName(old_isotope_code));

    if(verbose>2) {
      printf("fileformat := %s\n", tacFormattxt(tac.format));
      printf("tacNr := %d\n", tac.tacNr);
      printf("sampleNr := %d\n", tac.sampleNr);
      printf("old_isotope := %s\n", old_isotope_name);
    }

    /* If old and new isotope are the same, then do nothing */
    if(isotope_code==old_isotope_code) {
      if(verbose>0) fprintf(stderr, "Note: isotope was already correct.\n");
      tacFree(&tac);
      continue; /* next SIF */
    }

    /* If new isotope name was not specified, then just show the isotope */
    if(isotope_code==ISOTOPE_UNKNOWN) {
      fprintf(stdout, "isotope := %s\n", old_isotope_name);
      tacFree(&tac);
      continue; /* next SIF */
    }

    /* If isotope was defined in SIF, then do not change it, unless --force */
    if(old_isotope_code!=ISOTOPE_UNKNOWN && forceMode==0) {
      fprintf(stderr, "Warning: isotope was set already; not changed.\n");
      tacFree(&tac);
      continue; /* next SIF */
    }

    /* Change the isotope name */
    ret=tacSetHeaderIsotope(&tac.h, isotope_name);
    if(ret==TPCERROR_OK) {
      if(verbose>0)
        printf("isotope changed from %s to %s\n", 
               old_isotope_name, isotope_name);
    } else {
      fprintf(stderr, "Error: cannot set isotope.\n");
      tacFree(&tac); return(6);
    }

    /*
     *  Save the modified SIF 
     */
    if(verbose>1) printf("writing %s\n", siffile);
    FILE *fp; fp=fopen(siffile, "w");
    if(fp==NULL) {
      fprintf(stderr, "Error: cannot open file for writing.\n");
      tacFree(&tac); return(11);
    }
    ret=tacWrite(&tac, fp, TAC_FORMAT_SIF, 1, &status);
    fclose(fp); 
    if(ret!=TPCERROR_OK) {
      fprintf(stderr, "Error: %s\n", errorMsg(status.error));
      if(verbose>1) printf("ret := %d\n", ret);
      tacFree(&tac); return(12);
    }
    if(verbose>0) printf("modified %s saved.\n", siffile);

    tacFree(&tac);
  } /* next SIF */

  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/// @endcond
