/** @file var4tac.c
    @brief Adds Gaussian noise to simulated dynamic PET TAC.
    @remark Application name was previously var4dat (version 1.3.2 2012-05-07);
     command-line parameters are changed. 
    @copyright (c) Turku PET Centre
    @author Vesa Oikonen
 */
/// @cond
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
/*****************************************************************************/
#include "tpcextensions.h"
#include "tpcift.h"
#include "tpctac.h"
#include "tpcrand.h"
/*****************************************************************************/

/*****************************************************************************/
static char *info[] = {
  "Program for adding Gaussian noise to dynamic PET time-activity curve (TAC)",
  "or TACs using equation",
  "  TAC_noisy(t) = TAC(t) + (CV/100)*TAC(t)*G(0,1) ,", 
  "G(0,1) is a pseudo random number from a Gaussian distribution",
  "with zero mean and SD of one.",
  " ",
  "Usage: @P [Options] tacfile CV outputfile ",
  " ",
  "Options:",
  " -minsd=<Percentage of maximal SD>",
  "     Based on CV only the time frames with little or no activity would have",
  "     very little or no noise. With this option the noise level (SD) in all",
  "     time frames will be at least the given percentage of the maximal SD",
  "     of all time frames and TACs.",
  " -stdoptions", // List standard options like --help, -v, etc
  " ",
  "Frame durations are not used, even if available in the TAC file.",
  "If TAC file contains weight column, then the specified amount of variance is",
  "added to frames with weight=1, and higher or lower variance to those",
  "frames with lower or higher weight.",
  " ",
  "Example:",
  "     @P simulated.tac 10 noisy.tac",
  " ",
  "See also: svar4tac, fvar4tac, sim_3tcm, tacweigh, tacadd, avgttac",
  " ",
  "Keywords: TAC, simulation, noise",
  0};
/*****************************************************************************/

/*****************************************************************************/
/* Turn on the globbing of the command line, since it is disabled by default in
   mingw-w64 (_dowildcard=0); in MinGW32 define _CRT_glob instead, if necessary;
   In Unix&Linux wildcard command line processing is enabled by default. */
/*
#undef _CRT_glob
#define _CRT_glob -1
*/
int _dowildcard = -1;
/*****************************************************************************/

/*****************************************************************************/
/**
 *  Main
 */
int main(int argc, char **argv)
{
  int      ai, help=0, version=0, verbose=1;
  char     tacfile[FILENAME_MAX], simfile[FILENAME_MAX];
  double   minsd=nan("");
  double   cv=nan("");


  /*
   *  Get arguments
   */
  if(argc==1) {tpcPrintUsage(argv[0], info, stderr); return(1);}
  tacfile[0]=simfile[0]=(char)0;
  /* Options */
  for(ai=1; ai<argc; ai++) if(*argv[ai]=='-') {
    if(tpcProcessStdOptions(argv[ai], &help, &version, &verbose)==0) continue;
    char *cptr=argv[ai]+1; if(*cptr=='-') cptr++; if(!*cptr) continue;
    if(strncasecmp(cptr, "MINSD=", 6)==0) {
      minsd=atofVerified(cptr+6); 
      if(isnan(minsd) || minsd<0.0 || minsd>100.0) {
        fprintf(stderr, "Error: invalid minimum SD percentage '%s'.\n", argv[ai]); return(1);}
      if(minsd<0.9) fprintf(stderr, "Warning: minimum SD is set to %g%%.\n", minsd);
      continue;
    }
    fprintf(stderr, "Error: invalid option '%s'.\n", argv[ai]);
    return(1);
  } else break; // tac name argument may start with '-'

  TPCSTATUS status; statusInit(&status);
  statusSet(&status, __func__, __FILE__, __LINE__, TPCERROR_OK);
  status.verbose=verbose-1;
  
  /* Print help or version? */
  if(help==2) {tpcHtmlUsage(argv[0], info, ""); return(0);}
  if(help) {tpcPrintUsage(argv[0], info, stdout); return(0);}
  if(version) {tpcPrintBuild(argv[0], stdout); return(0);}

  /* Process other arguments, starting from the first non-option */
  if(ai<argc) {strlcpy(tacfile, argv[ai++], FILENAME_MAX);}
  if(ai<argc) {
    cv=atofVerified(argv[ai]); 
    if(isnan(cv) || cv<0.0 || cv>1000.0) {
      fprintf(stderr, "Error: invalid CV%% '%s'.\n", argv[ai]); return(1);}
    if(cv<1.0) fprintf(stderr, "Warning: CV is set to %g%%.\n", cv);
    ai++;
  }
  if(ai<argc) {strlcpy(simfile, argv[ai++], FILENAME_MAX);}
  if(ai<argc) {
    fprintf(stderr, "Error: invalid argument '%s'.\n", argv[ai]);
    return(1);
  }

  /* Is something missing? */
  if(!simfile[0]) {
    fprintf(stderr, "Error: missing command-line argument; use option --help\n");
    return(1);
  }

  /* In verbose mode print arguments and options */
  if(verbose>1) {
    printf("tacfile := %s\n", tacfile);
    printf("simfile := %s\n", simfile);
    if(!isnan(minsd)) printf("minsd := %g\n", minsd);
    fflush(stdout);
  }


  /*
   *  Read original TAC
   */
  if(verbose>1) fprintf(stdout, "reading %s\n", tacfile);
  TAC tac; tacInit(&tac);
  if(tacRead(&tac, tacfile, &status)!=TPCERROR_OK) {
    fprintf(stderr, "Error: %s (%s)\n", errorMsg(status.error), tacfile);
    tacFree(&tac); return(2);
  }
  if(verbose>2) {
    printf("fileformat := %s\n", tacFormattxt(tac.format));
    printf("tacNr := %d\n", tac.tacNr);
    printf("sampleNr := %d\n", tac.sampleNr);
    printf("xunit := %s\n", unitName(tac.tunit));
    printf("yunit := %s\n", unitName(tac.cunit));
    if(tacIsWeighted(&tac)) printf("weighting := yes\n");
  }
  /* Check for missing sample times */
  if(tacXNaNs(&tac)>0) {
    fprintf(stderr, "Warning: missing frame times.\n");
  }
  /* Check for missing concentrations */
  if(tacYNaNs(&tac, -1)>0) {
    fprintf(stderr, "Warning: missing concentrations.\n");
  }


  /*
   *  If min SD was given, then the maximum (weighted) concentration in the data is needed.
   *  Find it anyway as an additional data check.
   */
  {
    double f, ymax=-1.0;
    for(int fi=0; fi<tac.sampleNr; fi++)
      for(int ri=0; ri<tac.tacNr; ri++) {
        f=tac.c[ri].y[fi]; 
        if(tacIsWeighted(&tac) && tac.w[fi]>0.0) f/=tac.w[fi];
        if(tac.c[ri].y[fi]>ymax) ymax=tac.c[ri].y[fi];
      }
    if(verbose>1) printf("wymax := %g\n", ymax);
    if(!(ymax>0.0)) {
      fprintf(stderr, "Error: invalid concentration data.\n");
      tacFree(&tac); return(2);
    }
    /* Convert minsd percentage to the actual min SD */
    if(!isnan(minsd)) {
      minsd*=0.01; // min SD was given as percent
      minsd*=(0.01*cv)*ymax; // CV was given as percent
      if(verbose>1) printf("final_minsd := %g\n", minsd);
    }
  }


  /* 
   *  Add noise
   */
  if(verbose>1) printf("adding noise\n");
  double sd;
  MERTWI mt; mertwiInit(&mt); mertwiInitWithSeed64(&mt, mertwiSeed64());
  for(int fi=0; fi<tac.sampleNr; fi++) {
    for(int ri=0; ri<tac.tacNr; ri++) {
      sd=0.01*cv*tac.c[ri].y[fi];
      if(tacIsWeighted(&tac) && tac.w[fi]>0.0) sd/=tac.w[fi];
      if(minsd>sd) sd=minsd;
      tac.c[ri].y[fi] += sd*mertwiRandomGaussian(&mt);
    }
  }


  /*
   *  Write the file
   */
  if(verbose>1) printf("writing noisy data in %s\n", simfile);
  FILE *fp; fp=fopen(simfile, "w");
  if(fp==NULL) {
    fprintf(stderr, "Error: cannot open file for writing (%s)\n", simfile);
    tacFree(&tac); return(11);
  }
  int ret=tacWrite(&tac, fp, TAC_FORMAT_UNKNOWN, 1, &status);
  fclose(fp); tacFree(&tac);
  if(ret!=TPCERROR_OK) {
    fprintf(stderr, "Error (%d): %s\n", ret, errorMsg(status.error));
    return(12);
  }
  if(verbose>=0) printf("Noisy data saved in %s\n", simfile);

  return(0);
}
/*****************************************************************************/
/// @endcond
/*****************************************************************************/
