#!/bin/bash
#: Title      : test_fitmtrap
#: Date       : 2023-03-10
#: Author     : "Vesa Oikonen" <vesa.oikonen@utu.fi>
#: Options    : None

# Set the name of executable to test
if [ -n "${OS+1}" ] && [ "$OS" = "Windows_NT" ]; then
EXT=.exe;
else
EXT=;
fi

PROGRAM=../../fitmtrap$EXT;
if [ ! -f $PROGRAM ]; then
  printf "Failed: executable does not exist.\n"
  exit 1
fi

printf "=====================================================================\n"
printf "creating test data for %s\n" $PROGRAM
printf "=====================================================================\n"

if [ ! -f plasma.fit ] || [ ! -f p2rbc.par ] || [ ! -f cpet.par ] || [ ! -f frames.sif ]; then
  printf "Failed: test data does not exist.\n"
  exit 1
fi

if [ ! -f true100.par ] || [ ! -f true110.par ]; then
  printf "Failed: test data does not exist.\n"
  exit 1
fi


if [ ! -f plasma.bld ]; then
  rm -f blood.bld cpet.tac
  fit2dat -c=0,10,0.02 plasma.fit plasma.bld
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi

if [ ! -f plasma_1-HCT.bld ] || [ ! -f rbc_HCT.bld ] || [ ! -f blood.bld ] || [ ! -f plasmaf.tac ]; then
  rm -f cpet.tac
# Making input function as (1-HCT)*Cp where HCT=0.39
  taccalc plasma.bld x 0.61 plasma_1-HCT.bld
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
# Simulating RBC concentration curve dCrbc=kc*(1-HCT)*Cp
# where kc=0.60 and thus k2=kc*HCT=0.234
  sim_3tcm p2rbc.par plasma_1-HCT.bld plasma_1-HCT.bld rbc.bld
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
# Calculating the blood radioactivity residing in RBCs
  taccalc rbc.bld x 0.39 rbc_HCT.bld
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
# Calculating the blood curve as Cb=(1-HCT)*Cp+HCT*Crbc
  taccalc plasma_1-HCT.bld + rbc_HCT.bld blood.bld
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
# Setting file header information for plotting
  tacren blood.bld 1 Blood
  tacren plasma.bld 1 Plasma
  tacren plasma_1-HCT.bld 1 Plasma
  tacren rbc.bld 1 RBC
  tacren rbc_HCT.bld 1 RBC
  tacstudy --force blood.bld sim1
  tacstudy --force plasma.bld sim1
  tacstudy --force plasma_1-HCT.bld sim1
  tacstudy --force rbc.bld sim1
  tacstudy --force rbc_HCT.bld sim1
# Plotting blood components
  tac2svg -legend=y -yt="SUV" -mt="" -x2=10.2 blood.svg -l blood.bld plasma_1-HCT.bld rbc_HCT.bld
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
# Frames for plasma_1-HCT.bld
  simframe plasma_1-HCT.bld frames.sif plasmaf.tac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi  
fi

if [ ! -f cpet.tac ]; then
  # Simulating regional muscle concentration curves (Cpet)
  sim_3tcm cpet.par plasma_1-HCT.bld blood.bld cpet.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacstudy --force cpet.dat sim1
  simframe cpet.dat frames.sif cpet.tac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tac2svg -legend=y cpet.svg -s cpet.tac -l cpet.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi



printf "\n\n"
printf "=====================================================================\n"
printf "testing %s\n" $PROGRAM
printf "=====================================================================\n"

printf "\n 1.0.0 \n"
printf " Test case 1: Test with simulated data. \n"
printf " Test case 2: Do not constrain k2 in any way. \n"
printf " Test case 3: Fitted TACs saved and plotted. \n"
printf " Expected result 1: Correct results. \n"
printf " Expected result 2: SVG file created. \n"
printf " Expected result 3: Fitted TACs close to original. \n\n"

rm -f output100.svg output100.par fitted100.tac

$PROGRAM -svg=output100.svg -fit=fitted100.tac cpet.tac LVcav output100.par
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
if [ ! -f output100.svg ]; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=1 -rel=1 cpet.tac fitted100.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
parmatch -p=y -tacnames=y -parnames=y -wss=n -abs=0.002 -rel=1 true100.par output100.par
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 1.1.0 \n"
printf " Test case 1: Test with simulated data. \n"
printf " Test case 2: Constrain k2 to known value. \n"
printf " Test case 3: Save estimated PTAC. \n"
printf " Test case 4: Plot input TACs. \n"
printf " Expected result 1: Correct results. \n"
printf " Expected result 2: Estimated PTAC is reasonably close to the true one. \n"
printf " Expected result 3: Input SVG created. \n\n"

rm -f output110.par output110.svg p110.tac i110.svg fitted110.tac

$PROGRAM -k2=0.234 -input=p110.tac -isvg=i110.svg -fit=fitted110.tac -svg=output110.svg cpet.tac LVcav output110.par
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
parmatch -p=y -tacnames=y -parnames=y -wss=n -abs=0.002 -rel=1 true110.par output110.par
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval output110.par k2 0.234
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=1 -rel=1 cpet.tac fitted110.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tac2svg p110.svg plasmaf.tac -l p110.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.5 -rel=0.1 plasmaf.tac p110.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
if [ ! -f i110.svg ]; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n 1.2.0 \n"
printf " Test case 1: Test with simulated data. \n"
printf " Test case 2: Constrain k2 based on certain region. \n"
printf " Expected result 1: Correct results. \n\n"

rm -f output120.par

$PROGRAM -k2=Global cpet.tac LVcav output120.par
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
parmatch -p=y -tacnames=y -parnames=y -wss=n -abs=0.002 -rel=1 true110.par output120.par
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval -abs=0.002 output120.par k2 0.234
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval output120.par k2-constraint Global
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n 1.3.0 \n"
printf " Test case 1: Test with simulated data. \n"
printf " Test case 2: Constrain k2 based on median of all regions. \n"
printf " Expected result 1: Correct results. \n\n"

rm -f output130.par

$PROGRAM -k2=median cpet.tac LVcav output130.par
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
parmatch -p=y -tacnames=y -parnames=y -wss=n -abs=0.002 -rel=1 true110.par output130.par
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval -abs=0.002 output130.par k2 0.234
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval output130.par k2-constraint median
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n\n=====================================================================\n"

printf "\n 2.0.0 \n"
printf " Test case 1: Test with actual data. \n"
printf " Test case 2: k2 not constrained. \n"
printf " Expected result: Reasonable fits. \n\n"

tacformat -f=PMOD 4dm.csv 4dm.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
tacunit -xconv=min -yconv=kBq/mL 4dm.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
rm -f output200.svg output200.par fitted200.tac i200.svg
printf "\n ok \n"

$PROGRAM -svg=output200.svg -isvg=i200.svg -fit=fitted200.tac 4dm.csv LV output200.par
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
if [ ! -f output200.par ]; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
if [ ! -f output200.svg ]; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
if [ ! -f i200.svg ]; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=20 4dm.tac fitted200.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n 2.0.1 \n"
printf " Test case 1: Test with actual data. \n"
printf " Test case 2: k2 constrained to large ROI. \n"
printf " Expected result: Reasonable fits. \n\n"

tacformat -f=PMOD 4dm.csv 4dm.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
tacunit -xconv=min -yconv=kBq/mL 4dm.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
rm -f output201.svg output201.par fitted201.tac i201.svg
printf "\n ok \n"

$PROGRAM -k2=Global -svg=output201.svg -isvg=i201.svg -fit=fitted201.tac 4dm.csv LV output201.par
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
if [ ! -f output201.par ]; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
if [ ! -f output201.svg ]; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
if [ ! -f i201.svg ]; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=20 4dm.tac fitted201.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n===================================================================\n"
printf "   All passed! \n"
printf "===================================================================\n"
exit 0

