#!/bin/bash
#: Title      : test_taccbvp
#: Date       : 2023-03-14
#: Author     : "Vesa Oikonen" <vesa.oikonen@utu.fi>
#: Options    : None

# Set the name of executable to test
if [ -n "${OS+1}" ] && [ "$OS" = "Windows_NT" ]; then
EXT=.exe;
else
EXT=;
fi

PROGRAM=../../taccbvp$EXT;
if [ ! -f $PROGRAM ]; then
  printf "Failed: executable does not exist.\n"
  exit 1
fi


printf "=====================================================================\n"
printf "creating test data for %s\n" $PROGRAM
printf "=====================================================================\n"

if [ ! -f plasma.fit ] || [ ! -f p2rbc.par ] || [ ! -f cpet.par ] || [ ! -f frames.sif ]; then
  printf "Failed: test data does not exist.\n"
  exit 1
fi

if [ ! -f plasma.bld ]; then
  rm -f blood.bld cpet.tac
  fit2dat -c=0,10,0.02 plasma.fit plasma.bld
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi

if [ ! -f plasmac.bld ] || [ ! -f rbcc.bld ] || [ ! -f blood.bld ]; then
  rm -f cpet.tac ct.tac
# Computing PTAC as component of blood as (1-HCT)*Cp where HCT=0.39
  taccalc plasma.bld x 0.61 plasmac.bld
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
# Simulating RBC concentration curve dCrbc=kc*(1-HCT)*Cp
# where kc=0.60 and thus k2=kc*HCT=0.234
  sim_3tcm p2rbc.par plasmac.bld plasmac.bld rbc.bld
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
# Calculating the blood radioactivity residing in RBCs
  taccalc rbc.bld x 0.39 rbcc.bld
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
# Calculating the blood curve as Cb=(1-HCT)*Cp+HCT*Crbc
  taccalc plasmac.bld + rbcc.bld blood.bld
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
# Setting file header information for plotting
  tacren blood.bld 1 Blood
  tacren plasma.bld 1 Plasma
  tacren plasmac.bld 1 Plasma
  tacren rbc.bld 1 RBC
  tacren rbcc.bld 1 RBC
  tacstudy --force blood.bld sim1
  tacstudy --force plasma.bld sim1
  tacstudy --force plasmac.bld sim1
  tacstudy --force rbc.bld sim1
  tacstudy --force rbcc.bld sim1
# Plotting blood components
  tac2svg -legend=y -yt="SUV" -mt="" -x2=10.2 blood.svg -l blood.bld plasmac.bld rbcc.bld
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi

if [ ! -f cpet.tac ] || [ ! -f ct.tac ] || [ ! -f truevb.par ]; then
# Simulating regional muscle concentration curves (Cpet)
  sim_3tcm cpet.par plasmac.bld blood.bld cpet.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacstudy --force cpet.dat sim1
  simframe cpet.dat frames.sif cpet.tac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tac2svg -legend=y cpet.svg -s cpet.tac -l cpet.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
# Make parameter file containing only Vb
  paradd -ovr -par=Vb truevb.par cpet.par
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
# Simulate regional (1-Vb)*Ct data, without BTAC
  taccalc blood.bld x 0 zero.bld
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  sim_3tcm cpet.par plasmac.bld zero.bld ct.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacstudy --force ct.dat sim1
  simframe ct.dat frames.sif ct.tac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacdel ct.tac LVcav
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi




printf "\n\n"
printf "=====================================================================\n"
printf "testing %s\n" $PROGRAM
printf "=====================================================================\n"

printf "\n 1.0.0 \n"
printf " Test case 1: Estimate Vb and calculate corrected TACs. \n"
printf " Expected result: reasonable results are obtained. \n\n"

rm -f o100.par o100.tac
$PROGRAM -d3 -par=o100.par cpet.tac LVcav o100.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

tacdel o100.tac LVcav
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
tac2svg -legend=y o100.svg -s ct.tac -l o100.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=1 -rel=1 ct.tac o100.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

parmatch -abs=0.05 -rel=5 truevb.par o100.par
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n===================================================================\n"
printf "   All passed! \n"
printf "===================================================================\n"
exit 0

