/** @file tac2xml.c
 *  @brief Save Excel-compatible XML file containing data from one or more TAC files.
 *  @copyright (c) Turku PET Centre
 *  @author Vesa Oikonen
 */
/// @cond
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include <unistd.h>
/*****************************************************************************/
#include "tpcextensions.h"
#include "tpcift.h"
#include "tpctac.h"
/*****************************************************************************/

/*****************************************************************************/
static char *info[] = {
  "Save TAC data in Excel compatible XML format. Each TAC file will have",
  "its own sheet in the XML file.",
  "TAC files may contain variable number of TACs and samples.",
  " ",
  "Usage: @P [options] outputfile tacfiles",
  " ",
  "Options:",
  " -mid",
  "     Frame mid times are always used instead of frame start and end times.",
  " -stdoptions", // List standard options like --help, -v, etc
  " ",
  "See also: tacformat, tacjoin, tacblend, tacadd, tacunit, tacnames",
  " ",
  "Keywords: TAC, tool, format, Excel, XML, plotting",
  0};
/*****************************************************************************/

/*****************************************************************************/
/* Turn on the globbing of the command line, since it is disabled by default in
   mingw-w64 (_dowildcard=0); in MinGW32 define _CRT_glob instead, if necessary;
   In Unix&Linux wildcard command line processing is enabled by default. */
/*
#undef _CRT_glob
#define _CRT_glob -1
*/
int _dowildcard = -1;
/*****************************************************************************/

/*****************************************************************************/
/// @endcond
/** Write TAC data as a new sheet inside Excel compatible XML file.
    @return enum tpcerror (TPCERROR_OK when successful).
    @author Vesa Oikonen
 */
int tacWriteSheetIntoXML(
  /** Pointer to TAC struct, contents of which are to be written. */
  TAC *tac,
  /** Pointer to string containing name of sheet. */
  char *sheetname,
  /** Pointer to file opened for write. File must already contain XML header,
      and the footer must be written before it is closed. */
  FILE *fp,
  /** Pointer to status data; enter NULL if not needed. */
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  if(verbose>0) printf("%s(%s)\n", __func__, sheetname);
  if(fp==NULL) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_CANNOT_WRITE);
    return TPCERROR_CANNOT_WRITE;
  }
  if(tac==NULL || tac->tacNr<1 || tac->sampleNr<1) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_NO_DATA);
    return TPCERROR_NO_DATA;
  }

  char tunit[128], cunit[128];
  strcpy(tunit, unitName(tac->tunit));
  strcpy(cunit, unitName(tac->cunit));

  /* Make sure that TAC names are available */
  if(verbose>2) printf("constructing TAC names\n");
  tacEnsureNames(tac);

  /* Start worksheet and table */
  {
    int n=0;
    n=fprintf(fp, "  <ss:Worksheet ss:Name=\"%s\">\n", sheetname);
    n+=fprintf(fp, "    <ss:Table>\n");
    if(n<10) {
      statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_CANNOT_WRITE);
      return TPCERROR_CANNOT_WRITE;
    }
  }

  /* Set column widths */
  {
    int n=tac->tacNr+1;
    if(tac->isframe) n++;
    if(tacIsWeighted(tac)) n++;
    for(int i=0; i<n; i++) fprintf(fp, "      <ss:Column ss:Width=\"80\"/>\n");
  }

  /* Write the title line */
  fprintf(fp, "      <ss:Row ss:StyleID=\"1\">\n");
  if(tac->isframe==0) {
    fprintf(fp, "        <ss:Cell>\n");
    fprintf(fp, "          <ss:Data ss:Type=\"String\">time[%s]</ss:Data>\n", tunit);
    fprintf(fp, "        </ss:Cell>\n");
  } else {
    fprintf(fp, "        <ss:Cell>\n");
    fprintf(fp, "          <ss:Data ss:Type=\"String\">start[%s]</ss:Data>\n", tunit);
    fprintf(fp, "        </ss:Cell>\n");
    fprintf(fp, "        <ss:Cell>\n");
    fprintf(fp, "          <ss:Data ss:Type=\"String\">end[%s]</ss:Data>\n", cunit);
    fprintf(fp, "        </ss:Cell>\n");
  }
  for(int ri=0; ri<tac->tacNr; ri++) {
    /* write TAC names */
    fprintf(fp, "        <ss:Cell>\n");
    if(ri==0 && tac->isframe==0)
      fprintf(fp, "          <ss:Data ss:Type=\"String\">%s[%s]</ss:Data>\n", tac->c[ri].name, cunit);
    else
      fprintf(fp, "          <ss:Data ss:Type=\"String\">%s</ss:Data>\n", tac->c[ri].name);
    fprintf(fp, "        </ss:Cell>\n");
  }
  if(tacIsWeighted(tac)) {
    fprintf(fp, "        <ss:Cell>\n");
    fprintf(fp, "          <ss:Data ss:Type=\"String\">weight</ss:Data>\n");
    fprintf(fp, "        </ss:Cell>\n");
  }
  if(fprintf(fp, "      </ss:Row>\n")<1) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_CANNOT_WRITE);
    return TPCERROR_CANNOT_WRITE;
  }

  /* Write data */
  if(verbose>2) printf("writing data table\n");
  for(int fi=0; fi<tac->sampleNr; fi++) {

    fprintf(fp, "      <ss:Row>\n");

    /* Note: missing values are written as empty cells, with string type, because with
       number type Excel seems to read those as zeroes. */

    /* Time(s) (x, or x1 and x2) */
    double v; if(tac->isframe==0) v=tac->x[fi]; else v=tac->x1[fi];
    fprintf(fp, "        <ss:Cell>\n");
    if(isnan(v)) fprintf(fp, "          <ss:Data ss:Type=\"String\"></ss:Data>\n");
    else fprintf(fp, "          <ss:Data ss:Type=\"Number\">%g</ss:Data>\n", v);
    fprintf(fp, "        </ss:Cell>\n");

    if(tac->isframe) {
      v=tac->x2[fi];
      fprintf(fp, "        <ss:Cell>\n");
      if(isnan(v)) fprintf(fp, "          <ss:Data ss:Type=\"String\"></ss:Data>\n");
      else fprintf(fp, "          <ss:Data ss:Type=\"Number\">%g</ss:Data>\n", v);
      fprintf(fp, "        </ss:Cell>\n");
    }

    /* Concentrations (y values) */
    for(int ri=0; ri<tac->tacNr; ri++) {
      fprintf(fp, "        <ss:Cell>\n");
      if(isnan(tac->c[ri].y[fi])) fprintf(fp, "          <ss:Data ss:Type=\"String\"></ss:Data>\n");
      else fprintf(fp, "          <ss:Data ss:Type=\"Number\">%g</ss:Data>\n", tac->c[ri].y[fi]);
      fprintf(fp, "        </ss:Cell>\n");
    }

    /* Weight */
    if(tacIsWeighted(tac)) {
      fprintf(fp, "        <ss:Cell>\n");
      if(isnan(tac->w[fi])) fprintf(fp, "          <ss:Data ss:Type=\"String\"></ss:Data>\n");
      else fprintf(fp, "          <ss:Data ss:Type=\"Number\">%g</ss:Data>\n", tac->w[fi]);
      fprintf(fp, "        </ss:Cell>\n");
    }

    fprintf(fp, "      </ss:Row>\n");
  }

  /* Write table and worksheet end part */
  fprintf(fp, "    </ss:Table>\n");
  if(fprintf(fp, "  </ss:Worksheet>\n")<1) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_CANNOT_WRITE);
    return TPCERROR_CANNOT_WRITE;
  }

  statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_OK);
  return(TPCERROR_OK);
}
/// @cond
/*****************************************************************************/

/*****************************************************************************/
/**
 *  Main
 */
int main(int argc, char **argv)
{
  int ai, help=0, version=0, verbose=1;
  int mid_time=0; // Write mid frame time
  char tacfile[FILENAME_MAX], outfile[FILENAME_MAX];
  int fileNr=0, file1=0;


  /*
   *  Get arguments
   */
  if(argc==1) {tpcPrintUsage(argv[0], info, stderr); return(1);}
  tacfile[0]=outfile[0]=(char)0;
  /* Options */
  for(ai=1; ai<argc; ai++) if(*argv[ai]=='-') {
    if(tpcProcessStdOptions(argv[ai], &help, &version, &verbose)==0) continue;
    char *cptr=argv[ai]+1; if(*cptr=='-') cptr++; if(!*cptr) continue;
    if(strncasecmp(cptr, "MIDDLE", 3)==0) {
      mid_time=1; continue;
    }
    fprintf(stderr, "Error: invalid option '%s'\n", argv[ai]);
    return(1);
  } else break; // tac name argument may start with '-'

  TPCSTATUS status; statusInit(&status);
  statusSet(&status, __func__, __FILE__, __LINE__, TPCERROR_OK);
  status.verbose=verbose-3;
  
  /* Print help or version? */
  if(help==2) {tpcHtmlUsage(argv[0], info, ""); return(0);}
  if(help) {tpcPrintUsage(argv[0], info, stdout); return(0);}
  if(version) {tpcPrintBuild(argv[0], stdout); return(0);}

  /* Process other arguments, starting from the first non-option */
  if(ai<argc) strlcpy(outfile, argv[ai++], FILENAME_MAX);
  for(; ai<argc; ai++) {
    if(fileNr==0) file1=ai;
    fileNr++;
  }


  /* In verbose mode print arguments and options */
  if(verbose>1) {
    for(ai=0; ai<argc; ai++) printf("%s ", argv[ai]); 
    printf("\n");
    printf("fileNr := %d\n", fileNr);
    printf("outfile := %s\n", outfile);
    printf("mid_time := %d\n", mid_time);
    fflush(stdout);
  }

  /* Is something missing? */
  if(!outfile[0]) {tpcPrintUsage(argv[0], info, stdout); return(1);}
  if(fileNr==0) {
    fprintf(stderr, "Error: missing command-line argument; try %s --help\n", argv[0]);
    return(1);
  }


  /* Check that output file name extension is .xml */
  {
    char *cptr=filenameGetExtension(outfile);
    if(strcasecmp(cptr, ".XML")) {
      fprintf(stderr, "Error: output file name extension must be .xml\n");
      return(2);
    }
  }
  /* Check that all input files do exist, and that their name does not match output file name */
  for(ai=file1; ai<argc; ai++) {
    strlcpy(tacfile, argv[ai], FILENAME_MAX);
    if(access(tacfile, 0) == -1) {
      fprintf(stderr, "Error: input file %s does not exist.\n", tacfile);
      return(2);
    }
    if(strcasecmp(outfile, tacfile)==0) {
      fprintf(stderr, "Error: input file would be overwritten.\n");
      return(2);
    }
  }


  /*
   *  Open XML file for writing
   */
  if(verbose>1) printf("opening %s\n", outfile);
  FILE *fp; fp=fopen(outfile, "w");
  if(fp==NULL) {
    fprintf(stderr, "Error: cannot open file for writing.\n");
    return(11);
  }
  /* Write XML header */
  {
    int n=fprintf(fp, "<?xml version=\"1.0\"?>\n");
    if(n<1) {
      fprintf(stderr, "Error: cannot write file.\n");
      fclose(fp); return(12);
    }
    fprintf(fp, "<ss:Workbook xmlns:ss=\"urn:schemas-microsoft-com:office:spreadsheet\">\n");
    fprintf(fp, "  <ss:Styles>\n");
    fprintf(fp, "    <ss:Style ss:ID=\"1\">\n");
    fprintf(fp, "      <ss:Font ss:Bold=\"1\"/>\n");
    fprintf(fp, "    </ss:Style>\n");
    fprintf(fp, "  </ss:Styles>\n");
  }

  /*
   *  Go through all TAC files
   */
  int errorCounter=0;
  for(ai=file1; ai<argc; ai++) {
    strlcpy(tacfile, argv[ai], FILENAME_MAX);
    if(verbose>1) {printf("%s\n", tacfile); fflush(stdout);}
    /* Read TAC file */
    TAC tac; tacInit(&tac);
    if(tacRead(&tac, tacfile, &status)!=TPCERROR_OK) {
      if(verbose<=1) {printf("%s\n", tacfile); fflush(stdout);}
      fprintf(stderr, "Error: %s\n", errorMsg(status.error));
      tacFree(&tac); errorCounter++; break;
    }
    if(verbose>3) {
      printf("fileformat := %s\n", tacFormattxt(tac.format));
      printf("tacNr := %d\n", tac.tacNr);
      printf("sampleNr := %d\n", tac.sampleNr);
      printf("xunit := %s\n", unitName(tac.tunit));
      printf("yunit := %s\n", unitName(tac.cunit));
    }
    /* Use mid times if requested */
    if(mid_time) tac.isframe=0;
    /* Make sheet name */
    char sheetname[FILENAME_MAX]; strcpy(sheetname, tacfile);
    filenameRmPath(sheetname); filenameRmExtensions(sheetname);
    /* Write into XML */
    int ret=TPCERROR_OK;
    ret=tacWriteSheetIntoXML(&tac, sheetname, fp, &status);
    if(ret!=TPCERROR_OK) {
      fprintf(stderr, "Error: %s\n", errorMsg(status.error));
      tacFree(&tac); errorCounter++; break;
    }
    tacFree(&tac);
  }

  /* Write XML footer */
  if(errorCounter==0) {
    int n=fprintf(fp, "</ss:Workbook>\n");
    if(n<1) {
      fprintf(stderr, "Error: cannot write file.\n");
      errorCounter++;
    }
  }

  /* Close XML file */
  fclose(fp);

  /* Delete the file in case of errors */
  if(errorCounter>0) {
    remove(outfile);
    return(12);
  }

  if(verbose>=0) {
    printf("Contents of %d TAC files stored in %s\n", fileNr, outfile);
    fflush(stdout);
  }

  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/// @endcond

