/** @file tac4frpl.c
 *  @brief Transform TACs suitable for plotting with frames as bars.
 *  @remark Imported and renamed program dft2frpl (0.1.1 2006-12-28). 
 *  @copyright (c) Turku PET Centre
 *  @author Vesa Oikonen
 */
/// @cond
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
/*****************************************************************************/
#include "tpcextensions.h"
#include "tpctac.h"
/*****************************************************************************/

/*****************************************************************************/
static char *info[] = {
  "Transform TAC file into suitable format for plotting with bars",
  "representing both frame start and end times.",
  " ",
  "Usage: @P [Options] tacfile outputfile",
  " ",
  "Options:",
  " -stdoptions", // List standard options like --help, -v, etc
  " ",
  "See also: tacfr2x, tacframe, simframe, tac2svg, tacformat, tacsetx",
  " ",
  "Keywords: TAC, plotting, simulation, time",
  0};
/*****************************************************************************/

/*****************************************************************************/
/* Turn on the globbing of the command line, since it is disabled by default in
   mingw-w64 (_dowildcard=0); in MinGW32 define _CRT_glob instead, if necessary;
   In Unix&Linux wildcard command line processing is enabled by default. */
/*
#undef _CRT_glob
#define _CRT_glob -1
*/
int _dowildcard = -1;
/*****************************************************************************/

/*****************************************************************************/
/**
 *  Main
 */
int main(int argc, char **argv)
{
  int   ai, help=0, version=0, verbose=1;
  char  tacfile1[FILENAME_MAX], tacfile2[FILENAME_MAX];
  int   ret;

  /*
   *  Get arguments
   */
  if(argc==1) {tpcPrintUsage(argv[0], info, stderr); return(1);}
  tacfile1[0]=tacfile2[0]=(char)0;
  /* Options */
  for(ai=1; ai<argc; ai++) if(*argv[ai]=='-') {
    if(tpcProcessStdOptions(argv[ai], &help, &version, &verbose)==0) continue;
    //char *cptr=argv[ai]+1; if(*cptr=='-') cptr++; if(!*cptr) continue;
    fprintf(stderr, "Error: invalid option '%s'\n", argv[ai]);
    return(1);
  } else break; // tac name argument may start with '-'

  TPCSTATUS status; statusInit(&status);
  statusSet(&status, __func__, __FILE__, __LINE__, TPCERROR_OK);
  status.verbose=verbose-1;
  
  /* Print help or version? */
  if(help==2) {tpcHtmlUsage(argv[0], info, ""); return(0);}
  if(help) {tpcPrintUsage(argv[0], info, stdout); return(0);}
  if(version) {tpcPrintBuild(argv[0], stdout); return(0);}

  /* Arguments */
  if(ai<argc) {strlcpy(tacfile1, argv[ai++], FILENAME_MAX);}
  if(ai<argc) {strlcpy(tacfile2, argv[ai++], FILENAME_MAX);}
  if(ai<argc) {fprintf(stderr, "Error: invalid argument '%s'.\n", argv[ai]); return(1);}

  /* Is something missing? */
  if(!tacfile2[0]) {tpcPrintUsage(argv[0], info, stdout); return(1);}

  /* In verbose mode print arguments and options */
  if(verbose>1) {
    for(ai=0; ai<argc; ai++) printf("%s ", argv[ai]); 
    printf("\n");
    printf("tacfile1 := %s\n", tacfile1);
    printf("tacfile2 := %s\n", tacfile2);
    fflush(stdout);
  }


  /*
   *  Read the file
   */
  if(verbose>1) printf("reading %s\n", tacfile1);
  TAC tac1; tacInit(&tac1);
  ret=tacRead(&tac1, tacfile1, &status);
  if(ret!=TPCERROR_OK) {
    fprintf(stderr, "Error: %s\n", errorMsg(status.error));
    if(verbose>1) printf("ret := %d\n", ret);
    tacFree(&tac1); return(2);
  }
  if(verbose>2) {
    printf("fileformat := %s\n", tacFormattxt(tac1.format));
    printf("tacNr := %d\n", tac1.tacNr);
    printf("sampleNr := %d\n", tac1.sampleNr);
    printf("xunit := %s\n", unitName(tac1.tunit));
    printf("yunit := %s\n", unitName(tac1.cunit));
    printf("isframe := %d\n", tac1.isframe);
  }

  /* Check the data */
  if(tac1.sampleNr<1 || tac1.tacNr<1) {
    fprintf(stderr, "Error: invalid data.\n");
    tacFree(&tac1); return(3);
  }
  /* Make sure that frame start and end times are present */
  if(tac1.isframe==0) {
    fprintf(stderr, "Error: file does not contain frame start and end times.\n");
    tacFree(&tac1); return(3);
  }
  if(tacSortByTime(&tac1, &status)!=TPCERROR_OK) {
    fprintf(stderr, "Error: %s\n", errorMsg(status.error));
    if(verbose>1) printf("ret := %d\n", ret);
    tacFree(&tac1); return(3);
  }

#if(1)
  TAC tac2; tacInit(&tac2);
  ret=tacToBars(&tac1, &tac2);
  if(ret!=0) {
    fprintf(stderr, "Error: cannot produce meaningful bar data for plotting.\n");
    tacFree(&tac1); tacFree(&tac2); return(4);
  }

#else

  /* Make a copy of the TAC data in memory */
  if(verbose>1) printf("allocating memory for new TAC\n");
  TAC tac2; tacInit(&tac2);
  ret=tacDuplicate(&tac1, &tac2);
  if(ret!=0) {
    fprintf(stderr, "Error: cannot allocate memory.\n");
    tacFree(&tac1); return(4);
  }
  /* Add room for more frames; we plot each frame as its own bar
     and therefore we may need 4x the original space */
  ret=tacAllocateMoreSamples(&tac2, 3*tac1.sampleNr);
  if(ret!=0) {
    fprintf(stderr, "Error: cannot allocate memory.\n");
    tacFree(&tac1); return(4);
  }
  if(verbose>2) printf("_sampleNr := %d\n", tac2._sampleNr);

  /* Make separate 'samples' for each frame start and end time */
  int ri, fi, fj;
  tac2.isframe=0;

  for(fi=0, fj=0; fi<tac1.sampleNr; fi++) {
    /* Do nothing if missing values */
    if(isnan(tac1.x1[fi]) || isnan(tac1.x2[fi])) continue;
    if(tac1.tacNr==1 && isnan(tac1.c[0].y[fi])) continue;
    /* Otherwise make bar of each frame */
    tac2.x[fj]=tac1.x1[fi];
    for(ri=0; ri<tac2.tacNr; ri++) tac2.c[ri].y[fj]=0.0;
    fj++;
    tac2.x[fj]=tac1.x1[fi];
    for(ri=0; ri<tac2.tacNr; ri++) tac2.c[ri].y[fj]=tac1.c[ri].y[fi];
    fj++;
    tac2.x[fj]=tac1.x2[fi];
    for(ri=0; ri<tac2.tacNr; ri++) tac2.c[ri].y[fj]=tac1.c[ri].y[fi];
    fj++;
    tac2.x[fj]=tac1.x2[fi];
    for(ri=0; ri<tac2.tacNr; ri++) tac2.c[ri].y[fj]=0.0;
    fj++;
  }
  tac2.sampleNr=fj;
#endif

  tacFree(&tac1);

  /*
   *  Save data 
   */
  if(verbose>2) printf("writing %s\n", tacfile2);
  FILE *fp; fp=fopen(tacfile2, "w");
  if(fp==NULL) {
    fprintf(stderr, "Error: cannot open file for writing (%s)\n", tacfile2);
    tacFree(&tac2); return(11);
  }
  ret=tacWrite(&tac2, fp, TAC_FORMAT_UNKNOWN, 1, &status);
  fclose(fp); tacFree(&tac2);
  if(ret!=TPCERROR_OK) {
    fprintf(stderr, "Error: %s\n", errorMsg(status.error));
    return(12);
  }

  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/// @endcond
