/** @file tacadd.c
 *  @brief Insert one or more TACs from one or more TAC file(s) into another.
 *  @copyright (c) Turku PET Centre
 *  @author Vesa Oikonen
 *  @test Add tests for options. 
 */
/// @cond
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <math.h>
/*****************************************************************************/
#include "tpcextensions.h"
#include "tpcift.h"
#include "tpctac.h"
/*****************************************************************************/

/*****************************************************************************/
static char *info[] = {
  "Insert one or more TACs from TAC file 2 into another, optionally new,",
  "TAC file 1. Files must have the same number of samples (time frames).",
  "If the TAC number or name is not specified, then all TACs in file2 are added.",
  "If file1 does not exist, it will be created.",
  "Alternatively, all TACs from several files can be added to file1.",
  " ",
  "Usage: @P [options] file1 file2 [tacname or tacnumber]",
  "or",
  "Usage: @P [options] file1 file2 [file3 [file4 ...]]",
  " ",
  "Options:",
  " -sn",
  "     Copied region names are changed to contain the study number.",
  " -namestart",
  "     TAC name must match the initial part of names in file2.",
  " -ovr",
  "     Existing file1 is overwritten.",
  " --force",
  "     Program does not mind if the time or calibration units do not match.",
  " -move",
  "     Selected TAC(s) are removed from file2.",
  " -stdoptions", // List standard options like --help, -v, etc
  " ",
  "Example 1: Add TACs with name 'striatum' from file a123_12.dat into",
  "file a123_06.dat",
  "     @P a123_06.dat a123_12.dat striatum",
  " ",
  "Example 2a: Combine specified TAC files into a new file",
  "     @P combined.tac s6789*.tac",
  " ",
  "Example 2b: Combine specified TAC files into a new file",
  "using for loop in MS Windows command prompt window",
  "     for %g in (s6789*.tac) do @P combined.tac %g",
  " ",
  "Example 2c: Combine all putamen TACs from TAC files into a new file,",
  "storing study number as region name for each TAC,",
  "using for loop in MS Windows command prompt window",
  "     for %g in (a*.tac) do @P -sn putamen.tac %g put",
  " ",
  "See also: taclist, tacdel, tacsplit, tacsort, tacblend, tacunit, tac2svg",
  " ",
  "Keywords: TAC, tool",
  0};
/*****************************************************************************/

/*****************************************************************************/
/* Turn on the globbing of the command line, since it is disabled by default in
   mingw-w64 (_dowildcard=0); in MinGW32 define _CRT_glob instead, if necessary;
   In Unix&Linux wildcard command line processing is enabled by default. */
/*
#undef _CRT_glob
#define _CRT_glob -1
*/
int _dowildcard = -1;
/*****************************************************************************/

/*****************************************************************************/
/**
 *  Main
 */
int main(int argc, char **argv)
{
  int ai, help=0, version=0, verbose=1;
  int ret, addNr=0;
  int newfile=0; // 0=TAC(s) are added; 1=new file is created
  int sn2vn=0; // 0=TAC names are not changed; 1=TAC name is set to study number
  int overwrite=0; // 0=TAC(s) are added; 1=existing file is overwritten
  int tacmove=0; // 0=file2 is not modified; 1=selected TAC(s) are removed from file2
  int checkunits=1; // 0=Units do not need to match; 1=units must match
  int rnameStartGiven=0; // 1=Given TAC name needs to match the start of TAC names in file
  char *cptr, tacfile1[FILENAME_MAX], tacfile2[FILENAME_MAX];
  char rname[1024];
  int firstfile=0, fileNr=0;

  
  /*
   *  Get arguments
   */
  if(argc==1) {tpcPrintUsage(argv[0], info, stderr); return(1);}
  tacfile1[0]=tacfile2[0]=rname[0]=(char)0;
  /* Options */
  for(ai=1; ai<argc; ai++) if(*argv[ai]=='-') {
    if(tpcProcessStdOptions(argv[ai], &help, &version, &verbose)==0) continue;
    cptr=argv[ai]+1; if(*cptr=='-') cptr++; if(!*cptr) continue;
    if(strcasecmp(cptr, "SN")==0 || strncasecmp(cptr, "STUDYNUMBER", 5)==0) {
      sn2vn=1; continue;
    } else if(strncasecmp(cptr, "OVR", 2)==0) {
      overwrite=1; continue;
    } else if(strcasecmp(cptr, "MOVE")==0) {
      tacmove=1; continue;
    } else if(strncasecmp(cptr, "FORCE", 1)==0) {
      checkunits=0; continue;
    } else if(strncasecmp(cptr, "-FORCE", 2)==0) {
      checkunits=0; continue;
    } else if(strcasecmp(cptr, "NT")==0) {
      checkunits=0; continue;
    } else if(strcasecmp(cptr, "NAMESTART")==0) {
      rnameStartGiven=1; continue;
    }
    fprintf(stderr, "Error: invalid option '%s'\n", argv[ai]);
    return(1);
  } else break; // tac name argument may start with '-'

  TPCSTATUS status; statusInit(&status);
  statusSet(&status, __func__, __FILE__, __LINE__, TPCERROR_OK);
  status.verbose=verbose-1;
  
  /* Print help or version? */
  if(help==2) {tpcHtmlUsage(argv[0], info, ""); return(0);}
  if(help) {tpcPrintUsage(argv[0], info, stdout); return(0);}
  if(version) {tpcPrintBuild(argv[0], stdout); return(0);}

  /* Process other arguments, starting from the first non-option */
  if(ai<argc) {strlcpy(tacfile1, argv[ai++], FILENAME_MAX);}
  if(ai<argc) {
    firstfile=ai; fileNr=1;
    strlcpy(tacfile2, argv[ai++], FILENAME_MAX);
  } else {
    fprintf(stderr, "Error: missing filename.\n"); return(1);
  }
  /* More file names? */
  ret=0;
  for(; ai<argc; ai++) {
    if(access(argv[ai], 0) == -1) ret++; else fileNr++;
  }
  if(ret>0) { // at least not existing files, thus assuming those are TAC names
    fileNr=1; ai=firstfile+1;
    strlcpy(rname, argv[ai++], 1024);
    for(int n=1; ai<argc && n<3; ai++, n++) {
      if((strlen(rname)+strlen(argv[ai]))<1000) {
        strlcat(rname, " ", 1024); 
        strlcat(rname, argv[ai], 1024);
      }
    }
  }
  if(ai<argc) {
    fprintf(stderr, "Error: extra command-line argument.\n");
    return(1);
  }
  /* Check that we got what we need */
  if(!tacfile2[0]) {tpcPrintUsage(argv[0], info, stderr); return(1);}

  /* If combining several files, check that user did not forget to enter the name for new file */
  if(fileNr>1) {
    for(ai=firstfile; ai<argc; ai++) {
      /* Check that filename is not the same as the output filename */
      if(strcasecmp(argv[ai], tacfile1)==0) {
        fprintf(stderr, "Error: the same name for input and output file.\n");
        return(1);
      }
    }
  }
  /* If combining several files, move mode is not applicable */
  if(fileNr>1 && tacmove!=0) {
    fprintf(stderr, "Warning: ignoring option -move.\n");
    tacmove=0;
  }

  /* In verbose mode print arguments and options */
  if(verbose>1) {
    printf("tacfile1 := %s\n", tacfile1);
    printf("tacfile2 := %s\n", tacfile2);
    printf("overwrite := %d\n", overwrite);
    printf("checkunits := %d\n", checkunits);
    printf("sn2vn := %d\n", sn2vn);
    if(rname[0]) printf("rname := %s\n", rname);
    if(fileNr>1) printf("fileNr := %d\n", fileNr);
    if(fileNr==1) printf("move := %d\n", tacmove);
  }

  /* 
   *  Try to read the TAC file in which TACs should be added
   */
  if(verbose>1) printf("trying to read %s\n", tacfile1);
  TAC tac1; tacInit(&tac1);
  ret=tacRead(&tac1, tacfile1, &status);
  if(ret==TPCERROR_OK) {
    /* File 1 could be read */
    if(overwrite!=0) {
      if(verbose>=0) printf("File %s is overwritten.\n", tacfile1);
      tacFree(&tac1); newfile=1;
    } else if(verbose>1) {
      printf("fileformat1 := %s\n", tacFormattxt(tac1.format));
      printf("tacNr1 := %d\n", tac1.tacNr);
      printf("sampleNr1 := %d\n", tac1.sampleNr);
    }
  } else if(ret==TPCERROR_CANNOT_OPEN) {
    /* File 1 does not exist */
    if(verbose>=0) printf("File %s did not exist and will be created.\n", tacfile1);
    newfile=1;
  } else if(overwrite==0) {
    /* File 1 exists but cannot be read */
    fprintf(stderr, "Error (%d): %s\n", ret, errorMsg(status.error));
    tacFree(&tac1); return(2);
  } else {
    if(verbose>0) printf("File %s will be overwritten.\n", tacfile1);
    tacFree(&tac1); newfile=1;
  }


  /* 
   *  Read the (first) TAC file from where TAC(s) are imported
   */
  if(verbose>1) printf("reading %s\n", tacfile2);
  TAC tac2; tacInit(&tac2);
  ret=tacRead(&tac2, tacfile2, &status);
  if(ret!=TPCERROR_OK) {
    fprintf(stderr, "Error: %s\n", errorMsg(status.error));
    tacFree(&tac1); tacFree(&tac2); return(3);
  }
  if(verbose>1) {
    printf("fileformat2 := %s\n", tacFormattxt(tac2.format));
    printf("tacNr2 := %d\n", tac2.tacNr);
    printf("sampleNr2 := %d\n", tac2.sampleNr);
  }

  /* Check frames, unless new file is created */
  if(!newfile && tac1.sampleNr!=tac2.sampleNr) {
    fprintf(stderr, "Error: TAC files have different sample number.\n");
    tacFree(&tac1); tacFree(&tac2); return(4);
  }
  /* If new file is created, set the units */
  if(newfile) {
    tac1.cunit=tac2.cunit;
    tac1.tunit=tac2.tunit;
    /* and do not try to check the units */
    checkunits=0;
  }


  /* 
   *  Select the TACs in file 2 to be copied 
   */
  if(fileNr==1 && rname[0]) {
    if(verbose>1) {printf("selecting regions to be copied.\n"); fflush(stdout);}
    if(rnameStartGiven) {
      char rname2[strlen(rname)+3];
      strcpy(rname2, rname); strcat(rname2, "*");
      addNr=tacSelectTACs(&tac2, rname2, 1, &status);
    } else {
      addNr=tacSelectTACs(&tac2, rname, 1, &status);
    }
    if(addNr<=0) {
      fprintf(stderr, "Error: specified TACs not found in %s.\n", tacfile2);
      tacFree(&tac1); tacFree(&tac2); return(5);
    }
    if(addNr==tac2.tacNr) {
      fprintf(stderr, "Warning: adding all TACs from %s.\n", tacfile2);
      /* Move mode is not supported in this case */
      if(tacmove!=0) {
        fprintf(stderr, "Warning: ignoring option -move.\n");
        tacmove=0;
      }
    }
    if(verbose>1) printf("%d tac(s) match name '%s'\n", addNr, rname);
  } else {
    /* TAC name(s) or number(s) not specified by user, so all TACs are selected to be copied */
    for(int i=0; i<tac2.tacNr; i++) tac2.c[i].sw=1;
    addNr=tac2.tacNr;
    /* Move mode is not supported in this case */
    if(tacmove!=0) {
      fprintf(stderr, "Warning: ignoring option -move.\n");
      tacmove=0;
    }
  }
  if(verbose>1) printf("  %d TAC(s) will be added.\n", addNr);
  int sumAddNr=addNr;


  /* 
   *  Change TAC name to contain study number, if required
   */
  if(sn2vn) {
    if(verbose>1) printf("changing region name to include study number\n");
    char studynr[MAX_STUDYNR_LEN+1];
    ret=tacGetHeaderStudynr(&tac2.h, studynr, &status);
    if(ret!=TPCERROR_OK) {
      fprintf(stderr, "Error: TAC file does not contain valid studynumber.\n");
      tacFree(&tac1); tacFree(&tac2); return(6);
    }
    if(verbose>2) printf("studynr2 := %s\n", studynr);
    /* Add the study number to TAC names */
    int len;
    for(int i=0; i<tac2.tacNr; i++) if(tac2.c[i].sw) {
      /* Check that TAC names have space to add the study number */
      len=1+strlen(tac2.c[i].name);
      len+=strlen(studynr);
      if(len<MAX_TACNAME_LEN) {
        /* yes, add it */
        char tmp[MAX_TACNAME_LEN+1];
        strcpy(tmp, studynr); strcat(tmp, "_"); strcat(tmp, tac2.c[i].name);
        strcpy(tac2.c[i].name, tmp);
      } else if(strlen(studynr)<=MAX_TACNAME_LEN) {
        /* no, just put the study number */
        strcpy(tac2.c[i].name, studynr);
      }
    } // next selected TAC
  }


  /*
   *  Allocate memory in TAC struct for the new TACs
   */
  if(newfile) {
    if(verbose>2) printf("creating space for %d TACs\n", addNr);
    /* New file ; allocate memory and copy header */
    ret=tacAllocate(&tac1, tac2.sampleNr, addNr);
    if(ret!=TPCERROR_OK) {
      fprintf(stderr, "Error: cannot allocate memory.\n");
      tacFree(&tac1); tacFree(&tac2); return(7);
    }
    tac1.sampleNr=tac2.sampleNr; tac1.tacNr=0;
    ret=tacCopyHdr(&tac2, &tac1);
    if(ret!=TPCERROR_OK) {
      fprintf(stderr, "Error: cannot copy TAC header.\n");
      tacFree(&tac1); tacFree(&tac2); return(7);
    }
    /* Copy sample times */
    ret=tacXCopy(&tac2, &tac1, 0, tac1.sampleNr-1);
    if(ret!=TPCERROR_OK) {
      fprintf(stderr, "Error: cannot copy sample times.\n");
      tacFree(&tac1); tacFree(&tac2); return(7);
    }
  } else {
    if(verbose>2) printf("adding space for %d TACs\n", addNr);
    /* Add memory to previous TAC struct */
    ret=tacAllocateMore(&tac1, addNr);
    if(ret!=TPCERROR_OK) {
      fprintf(stderr, "Error: cannot allocate memory.\n");
      tacFree(&tac1); tacFree(&tac2); return(7);
    }
    /* Convert y units */
    ret=tacYUnitConvert(&tac2, tac1.cunit, &status);
    if(ret==TPCERROR_UNKNOWN_UNIT) ret=TPCERROR_OK;
    if(ret!=TPCERROR_OK) { // failed
      /* failing is a problem if units are to be verified */
      if(checkunits!=0) {
        fprintf(stderr, "Error: units do match.\n");
        tacFree(&tac1); tacFree(&tac2); return(8);
      } else {
        /* Since units are NOT to be checked, that is not a problem.
           But lets assume that if units are given, those are correct */
        if(verbose>0) fprintf(stderr, "Warning: units do match.\n");
        if(tac1.tunit==UNIT_UNKNOWN) tac1.tunit=tac2.tunit;
        else tac2.tunit=tac1.tunit;
        if(tac1.cunit==UNIT_UNKNOWN) tac1.cunit=tac2.cunit;
        else tac2.cunit=tac1.cunit;
      }
    }    
  }

  /* Add specified TAC(s) */
  for(int i=0; i<tac2.tacNr; i++) if(tac2.c[i].sw) {
    if(verbose>3) printf("  copying TAC %d\n", 1+i);
    ret=tacCopyTacc(&tac2.c[i], &tac1.c[tac1.tacNr], tac2.sampleNr);
    if(ret!=TPCERROR_OK) {
      fprintf(stderr, "Error: cannot copy TAC.\n");
      tacFree(&tac1); tacFree(&tac2); return(9);
    }
    tac1.tacNr++;
  } // next selected TAC

  /* Add weights, if necessary */
  if(newfile || (!tacIsWeighted(&tac1) && tacIsWeighted(&tac2))) {
    tac1.weighting=tac2.weighting;
    for(int j=0; j<tac2.sampleNr; j++) tac1.w[j]=tac2.w[j];
  }

  /* Remove specified TAC(s) from the source file, if requested, and possible */
  if(tacmove!=0) { // despite of user request, previously set to 0, if not applicable
    int n=0; 
    int i=tac2.tacNr-1;
    ret=TPCERROR_OK;
    while(i>=0) {
      if(tac2.c[i].sw) {
        if(verbose>2) printf("%s deleted\n", tac2.c[i].name);
        ret=tacDeleteTACC(&tac2, i); if(ret!=TPCERROR_OK) break; else n++;
      }
      i--;
    }
    if(ret!=TPCERROR_OK) {
      fprintf(stderr, "Warning: cannot delete TAC.\n");
    } else if(n>0 && tac2.tacNr>0) {
      if(verbose>1) printf("writing %s\n", tacfile2);
      FILE *fp; fp=fopen(tacfile2, "w");
      if(fp==NULL) {
        fprintf(stderr, "Warning: cannot open file for writing (%s)\n", tacfile2);
      } else {
        ret=tacWrite(&tac2, fp, TAC_FORMAT_UNKNOWN, 1, &status);
        fclose(fp);
        if(ret!=TPCERROR_OK) fprintf(stderr, "Warning: cannot write file %s\n", tacfile2);
      }
    }
  }

  tacFree(&tac2);


  /*
   *  If more TAC files, then add data from those, too
   */
  ai=firstfile+1;
  for(int tfi=1; tfi<fileNr; tfi++, ai++) {
    /* Read the next TAC file from where TAC(s) are imported */
    strlcpy(tacfile2, argv[ai], FILENAME_MAX);
    if(verbose>1) printf("reading %s\n", tacfile2);
    ret=tacRead(&tac2, tacfile2, &status);
    if(ret!=TPCERROR_OK) {
      fprintf(stderr, "Error (%d): %s\n", ret, errorMsg(status.error));
      tacFree(&tac1); tacFree(&tac2); return(3);
    }
    if(verbose>1) {
      printf("fileformat%d := %s\n", 1+tfi, tacFormattxt(tac2.format));
      printf("tacNr%d := %d\n", 1+tfi, tac2.tacNr);
      printf("sampleNr%d := %d\n", 1+tfi, tac2.sampleNr);
    }
    /* Check frames */
    if(tac1.sampleNr!=tac2.sampleNr) {
      fprintf(stderr, "Error: TAC files have different sample number.\n");
      tacFree(&tac1); tacFree(&tac2); return(4);
    }
    addNr=tac2.tacNr;
    if(verbose>1) printf("  %d TAC(s) will be added.\n", addNr);
    sumAddNr+=addNr;

    /* Change TAC name to contain study number, if required */
    if(sn2vn) {
      if(verbose>2) printf("changing region name to include study number\n");
      char studynr[MAX_STUDYNR_LEN+1];
      ret=tacGetHeaderStudynr(&tac2.h, studynr, &status);
      if(ret!=TPCERROR_OK) {
        fprintf(stderr, "Error: TAC file does not contain valid studynumber.\n");
        tacFree(&tac1); tacFree(&tac2); return(6);
      }
      if(verbose>2) printf("studynr%d := %s\n", 1+tfi, studynr);
      /* Add the study number to TAC names */
      int len;
      for(int i=0; i<tac2.tacNr; i++) {
        /* Check that TAC names have space to add the study number */
        len=1+strlen(tac2.c[i].name);
        len+=strlen(studynr);
        if(len<MAX_TACNAME_LEN) {
          /* yes, add it */
          char tmp[MAX_TACNAME_LEN+1];
          strcpy(tmp, studynr); strcat(tmp, "_"); strcat(tmp, tac2.c[i].name);
          strcpy(tac2.c[i].name, tmp);
        } else if(strlen(studynr)<=MAX_TACNAME_LEN) {
          /* no, just put the study number */
          strcpy(tac2.c[i].name, studynr);
        }
      } // next TAC
    }

    /* Allocate memory in TAC struct for the new TACs */
    if(verbose>2) printf("adding space for %d TACs\n", addNr);
    /* Add memory to previous TAC struct */
    ret=tacAllocateMore(&tac1, addNr);
    if(ret!=TPCERROR_OK) {
      fprintf(stderr, "Error: cannot allocate memory.\n");
      tacFree(&tac1); tacFree(&tac2); return(7);
    }
    /* Convert y units */
    ret=tacYUnitConvert(&tac2, tac1.cunit, &status);
    if(ret==TPCERROR_UNKNOWN_UNIT) ret=TPCERROR_OK;
    if(ret!=TPCERROR_OK) { // failed
      /* failing is a problem if units are to be verified */
      if(checkunits!=0) {
        fprintf(stderr, "Error: units do match.\n");
        tacFree(&tac1); tacFree(&tac2); return(8);
      } else {
        /* Since units are NOT to be checked, that is not a problem.
           But lets assume that if units are given, those are correct */
        if(verbose>0) fprintf(stderr, "Warning: units do match.\n");
        if(tac1.tunit==UNIT_UNKNOWN) tac1.tunit=tac2.tunit;
        else tac2.tunit=tac1.tunit;
        if(tac1.cunit==UNIT_UNKNOWN) tac1.cunit=tac2.cunit;
        else tac2.cunit=tac1.cunit;
      }
    }

    /* Add the TAC(s) */
    for(int i=0; i<tac2.tacNr; i++) {
      if(verbose>3) printf("  copying TAC %d\n", 1+i);
      ret=tacCopyTacc(&tac2.c[i], &tac1.c[tac1.tacNr], tac2.sampleNr);
      if(ret!=TPCERROR_OK) {
        fprintf(stderr, "Error: cannot copy TAC.\n");
        tacFree(&tac1); tacFree(&tac2); return(9);
      }
      tac1.tacNr++;
    } // next TAC

    /* Add weights, if necessary */
    if(!tacIsWeighted(&tac1) && tacIsWeighted(&tac2)) {
      tac1.weighting=tac2.weighting;
      for(int j=0; j<tac2.sampleNr; j++) tac1.w[j]=tac2.w[j];
    }

    tacFree(&tac2);
  } // next file


  /* Save data */
  if(verbose>1) printf("writing %s\n", tacfile1);
  FILE *fp; fp=fopen(tacfile1, "w");
  if(fp==NULL) {
    fprintf(stderr, "Error: cannot open file for writing (%s)\n", tacfile1);
    tacFree(&tac1); return(11);
  }
  if(tac1.format==TAC_FORMAT_UNKNOWN) tac1.format=tac2.format;
  ret=tacWrite(&tac1, fp, TAC_FORMAT_UNKNOWN, 1, &status);
  fclose(fp);
  if(ret!=TPCERROR_OK) {
    fprintf(stderr, "Error (%d): %s\n", ret, errorMsg(status.error));
    tacFree(&tac1); return(12);
  }

  /* Free memory */
  tacFree(&tac1); 

  if(verbose>=0) printf("  %d TAC(s) added to %s\n", sumAddNr, tacfile1);
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/// @endcond
