/** @file tacfcont.c
 *  @brief Ensure that TAC time frames are continuous, without overlaps or gaps.
 *  @copyright (c) Turku PET Centre
 *  @author Vesa Oikonen
 */
/// @cond
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
/*****************************************************************************/
#include "tpcextensions.h"
#include "tpcift.h"
#include "tpctac.h"
/*****************************************************************************/

/*****************************************************************************/
static char *info[] = {
  "Ensure that TAC time frames are continuous, without overlaps or gaps.",
  "Program return code is 0, if time frames are contiguous, or optionally",
  "were made contiguous.",
  " ",
  "Usage: @P [options] file",
  " ",
  "Options:",
  " -fix",
  "     Gaps and overlaps between time frames are fixed, when possible.",
  "     Tiny gaps/overlaps are fixed by editing frame times.",
  "     Larger gaps are filled with new frames, based on linear interpolation",
  "     between frame middle points of adjacent frames.",
  " -stdoptions", // List standard options like --help, -v, etc
  " ",
  "See also: tacframe, tacdelna, simframe, tacsetx",
  " ",
  "Keywords: TAC, tool, time",
  0};
/*****************************************************************************/

/*****************************************************************************/
/* Turn on the globbing of the command line, since it is disabled by default in
   mingw-w64 (_dowildcard=0); in MinGW32 define _CRT_glob instead, if necessary;
   In Unix&Linux wildcard command line processing is enabled by default. */
/*
#undef _CRT_glob
#define _CRT_glob -1
*/
int _dowildcard = -1;
/*****************************************************************************/

/*****************************************************************************/
/**
 *  Main
 */
int main(int argc, char **argv)
{
  int ai, help=0, version=0, verbose=1;
  int ret, fixMode=0;
  char *cptr, tacfile[FILENAME_MAX];

  /*
   *  Get arguments
   */
  if(argc==1) {tpcPrintUsage(argv[0], info, stderr); return(1);}
  tacfile[0]=(char)0;
  /* Options */
  for(ai=1; ai<argc; ai++) if(*argv[ai]=='-') {
    if(tpcProcessStdOptions(argv[ai], &help, &version, &verbose)==0) continue;
    cptr=argv[ai]+1; if(*cptr=='-') cptr++; if(!*cptr) continue;
    if(strcasecmp(cptr, "FIX")==0) {
      fixMode=1; continue;
    }
    fprintf(stderr, "Error: invalid option '%s'\n", argv[ai]);
    return(1);
  } else break; // tac name argument may start with '-'

  TPCSTATUS status; statusInit(&status);
  statusSet(&status, __func__, __FILE__, __LINE__, TPCERROR_OK);
  status.verbose=verbose-3;
  
  /* Print help or version? */
  if(help==2) {tpcHtmlUsage(argv[0], info, ""); return(0);}
  if(help) {tpcPrintUsage(argv[0], info, stdout); return(0);}
  if(version) {tpcPrintBuild(argv[0], stdout); return(0);}

  /* The first argument (non-option) is the filename */
  if(ai<argc) {strlcpy(tacfile, argv[ai], FILENAME_MAX); ai++;}
  else {fprintf(stderr, "Error: missing filename.\n"); return(1);}
  if(ai<argc) {fprintf(stderr, "Error: extra command-line argument.\n"); return(1);}
  
  /* In verbose mode print arguments and options */
  if(verbose>1) {
    printf("tacfile := %s\n", tacfile);
    printf("fixMode := %d\n", fixMode);
  }

  /* Read the file */
  if(verbose>1) printf("reading %s\n", tacfile);
  TAC tac; tacInit(&tac);
  ret=tacRead(&tac, tacfile, &status);
  if(ret!=TPCERROR_OK) {
    fprintf(stderr, "Error (%d): %s\n", ret, errorMsg(status.error));
    tacFree(&tac); return(2);
  }
  if(verbose>2) {
    printf("fileformat := %s\n", tacFormattxt(tac.format));
    printf("tacNr := %d\n", tac.tacNr);
    printf("sampleNr := %d\n", tac.sampleNr);
  }

  /*
   *  Check if there are gaps or overlaps
   */
  ret=tacIsXContiguous(&tac);
  if(ret==TPCERROR_OK) {
    fprintf(stdout, "Note: frame times are contiguous.\n");
    tacFree(&tac); return(0);
  }
  if(fixMode==0) {
    fprintf(stderr, "Warning: frame times are not contiguous.\n");
    tacFree(&tac); return(ret);
  }
  if(ret!=TPCERROR_OVERLAPPING_DATA && ret!=TPCERROR_LARGE_GAP) {
    fprintf(stderr, "Error: time frames cannot be fixed.\n");
    tacFree(&tac); return(ret);
  }

  /*
   *  Fix frame times
   */
  fprintf(stdout, "Note: fixing frame times that are not contiguous.\n");
  ret=tacSetXContiguous(&tac);
  if(ret==TPCERROR_OVERLAPPING_DATA) {
    fprintf(stderr, "Error: large frame overlap.\n");
    tacFree(&tac); return(ret);
  }
  if(ret!=TPCERROR_OK) {
    fprintf(stderr, "Error: cannot fix frame times.\n");
    tacFree(&tac); return(ret);
  }


  /*
   *  Save data 
   */
  if(verbose>1) printf("writing fixed %s\n", tacfile);
  FILE *fp; fp=fopen(tacfile, "w");
  if(fp==NULL) {
    fprintf(stderr, "Error: cannot open file for writing (%s)\n", tacfile);
    tacFree(&tac); return(11);
  }
  ret=tacWrite(&tac, fp, TAC_FORMAT_UNKNOWN, 1, &status);
  fclose(fp); tacFree(&tac);
  if(ret!=TPCERROR_OK) {
    fprintf(stderr, "Error (%d): %s\n", ret, errorMsg(status.error));
    return(12);
  }

  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/// @endcond
