/** @file tacframe.c
 *  @brief Print or edit frame times in TAC file.
 *  @remark Previously dftframe.
 *  @copyright (c) Turku PET Centre
 *  @author Vesa Oikonen
 */
/// @cond
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <math.h>
/*****************************************************************************/
#include "tpcextensions.h"
//#include "tpcift.h"
#include "tpctac.h"
/*****************************************************************************/

/*****************************************************************************/
static char *info[] = {
  "Print or edit the frame times in regional TAC datafile.",
  " ",
  "By default, frame start times and lengths are printed on screen, and",
  "optionally saved in the SIF (scan information file), if it does not exist.",
  "If an existing SIF name is specified, the SIF frames are written into",
  "regional TAC file; also normal TAC file containing frame start and end times",
  "can be given instead of SIF.",
  " ",
  "Usage: @P [options] tacfile sif",
  " ",
  "Options:",
  " -mid",
  "     Frame mid time is written in TAC file instead of frame start and end",
  "     times that are read from SIF. Not effective when writing SIF.",
  " -stdoptions", // List standard options like --help, -v, etc
  " ",
  "Example:",
  "     @P iea567.tac iea567dy.img.sif",
  " ",
  "See also: tactime, tacsetx, interpol, taccat, taccut, simframe, tacformat",
  " ",
  "Keywords: TAC, SIF, tool, time",
  0};
/*****************************************************************************/

/*****************************************************************************/
/* Turn on the globbing of the command line, since it is disabled by default in
   mingw-w64 (_dowildcard=0); in MinGW32 define _CRT_glob instead, if necessary;
   In Unix&Linux wildcard command line processing is enabled by default. */
/*
#undef _CRT_glob
#define _CRT_glob -1
*/
int _dowildcard = -1;
/*****************************************************************************/

/*****************************************************************************/
/**
 *  Main
 */
int main(int argc, char **argv)
{
  int ai, help=0, version=0, verbose=1;
  int ret;
  int opMode=1; // 0=change, 1=list but do not save, 2=list and save
  int isframe=1; // 0=save frame mid times, 1=save frame start and end times
  char *cptr, tacfile[FILENAME_MAX], siffile[FILENAME_MAX];
  TAC tac, sif;

  
  /*
   *  Get arguments
   */
  if(argc==1) {tpcPrintUsage(argv[0], info, stderr); return(1);}
  tacInit(&tac); tacInit(&sif);
  tacfile[0]=siffile[0]=(char)0;
  /* Options */
  for(ai=1; ai<argc; ai++) if(*argv[ai]=='-') {
    if(tpcProcessStdOptions(argv[ai], &help, &version, &verbose)==0) continue;
    cptr=argv[ai]+1; if(*cptr=='-') cptr++; if(!*cptr) continue;
    if(strncasecmp(cptr, "MIDDLE", 3)==0) {
      isframe=0; continue;
    }
    fprintf(stderr, "Error: invalid option '%s'\n", argv[ai]);
    return(1);
  } else break; // tac name argument may start with '-'

  TPCSTATUS status; statusInit(&status);
  statusSet(&status, __func__, __FILE__, __LINE__, TPCERROR_OK);
  status.verbose=verbose-3;

  /* Print help or version? */
  if(help==2) {tpcHtmlUsage(argv[0], info, ""); return(0);}
  if(help) {tpcPrintUsage(argv[0], info, stdout); return(0);}
  if(version) {tpcPrintBuild(argv[0], stdout); return(0);}

  /* Process other arguments, starting from the first non-option */
  for(; ai<argc; ai++) {
    if(!tacfile[0]) {
      strlcpy(tacfile, argv[ai], FILENAME_MAX); continue;
    } else if(!siffile[0]) {
      strlcpy(siffile, argv[ai], FILENAME_MAX); continue;
    } 
    fprintf(stderr, "Error: too many arguments: '%s'.\n", argv[ai]);
    return(1);
  }

  /* In verbose mode print arguments and options */
  if(verbose>1) {
    printf("tacfile := %s\n", tacfile);
    printf("siffile := %s\n", siffile);
    printf("isframe := %d\n", isframe);
  }

  /* Did we get all the information that we need? */
  if(!tacfile[0]) {
    fprintf(stderr, "Error: missing TAC name.\n");
    return(1);
  }

  /* Read the TAC file */
  if(verbose>1) printf("reading %s\n", tacfile);
  ret=tacRead(&tac, tacfile, &status);
  if(ret!=TPCERROR_OK) {
    fprintf(stderr, "Error (%d): %s\n", ret, errorMsg(status.error));
    tacFree(&tac); return(2);
  }
  if(verbose>2) {
    printf("fileformat := %s\n", tacFormattxt(tac.format));
    printf("tacNr := %d\n", tac.tacNr);
    printf("sampleNr := %d\n", tac.sampleNr);
    printf("tac_frames := %d\n", tac.isframe);
  }
  if(tac.tunit==UNIT_UNKNOWN) {
    if(verbose>=0)
      fprintf(stdout, "Warning: unknown time unit in %s\n", tacfile);
  }

  /* If SIF name was given by user, check if it exists */
  if(siffile[0]) {
    /* Check if file exists */
    if(access(siffile, 0) != -1) {
      opMode=0; /* SIF exists; change frames */
    } else {
      opMode=2; /* not existing; so create it */
    }
  } else
    opMode=1; /* SIF not specified; just list frames */
  if(verbose>2) printf("opMode := %d\n", opMode);


  /*
   *  Read new frames from frame file, set frames and save TAC file
   */
  if(opMode==0) {
    if(verbose>1) printf("reading %s\n", siffile);
    ret=tacRead(&sif, siffile, &status);
    if(ret!=TPCERROR_OK) {
      fprintf(stderr, "Error (%d): %s\n", ret, errorMsg(status.error));
      tacFree(&tac); tacFree(&sif); return(3);
    }
    if(verbose>2) {
      printf("siffileformat := %s\n", tacFormattxt(sif.format));
      printf("siftacNr := %d\n", sif.tacNr);
      printf("sifsampleNr := %d\n", sif.sampleNr);
      printf("sifisframe := %d\n", sif.isframe);
    }
    if(sif.tunit==UNIT_UNKNOWN) {
      if(verbose>=0)
        fprintf(stdout, "Warning: unknown time unit in %s\n", siffile);
    }
    /* Check that frame number is the same */
    if(tac.sampleNr>sif.sampleNr) {
      fprintf(stderr, "Error: different frame numbers.\n");
      tacFree(&tac); tacFree(&sif); return(3);
    } else if(tac.sampleNr<sif.sampleNr) {
      fprintf(stderr, "Warning: different frame numbers.\n");
    }
    /* Check that frame start and end times are there, if required */
    if(isframe==1 && sif.isframe==0) {
      fprintf(stderr, "Error: no frame start and end times in %s\n", siffile);
      tacFree(&tac); tacFree(&sif); return(3);
    }
    /* Convert SIF times to min if necessary */
    if(tac.tunit==UNIT_MIN && sif.tunit==UNIT_SEC)
      tacXUnitConvert(&sif, UNIT_MIN, &status);
    /* Set TAC time units, if not previously known */
    if(tac.tunit==UNIT_UNKNOWN) tac.tunit=sif.tunit;
    /* Copy frame times */
    ret=tacXCopy(&sif, &tac, 0, tac.sampleNr-1);
    if(ret!=TPCERROR_OK) {
      fprintf(stderr, "Error: cannot copy frame times.\n");
      tacFree(&tac); tacFree(&sif); return(4);
    }
    tacFree(&sif);
    /* Set output format */
    tac.isframe=isframe;
    if(isframe!=0) {
      if(tac.format==TAC_FORMAT_SIMPLE || tac.format==TAC_FORMAT_IFT)
        tac.format=TAC_FORMAT_DFT;
    }
    if(!tacFormatWriteSupported(tac.format)) tac.format=TAC_FORMAT_PMOD;
    /* Save edited TAC data */
    if(verbose>1) printf("writing %s\n", tacfile);
    FILE *fp; fp=fopen(tacfile, "w");
    if(fp==NULL) {
      fprintf(stderr, "Error: cannot open file for writing (%s)\n", tacfile);
      tacFree(&tac); return(5);
    }
    ret=tacWrite(&tac, fp, TAC_FORMAT_UNKNOWN, 1, &status);
    fclose(fp);
    if(ret!=TPCERROR_OK) {
      fprintf(stderr, "Error (%d): %s\n", ret, errorMsg(status.error));
      tacFree(&tac); return(6);
    }
  }


  /*
   *  Create SIF file based on TAC file if necessary
   */
  if(opMode==2) {
    if(verbose>1) printf("creating SIF data from regional TAC data\n");
    /* Set file format */
    tac.format=TAC_FORMAT_SIF;
    /* Convert time units to sec if necessary */
    int m2s=0;
    if(tac.tunit==UNIT_MIN) {tacXUnitConvert(&sif, UNIT_SEC, &status); m2s=1;}
    /* Check that data contains frame start and end times */
    if(tac.isframe==0) {
      if(verbose>2) 
        printf("original data does not contain frame start and end times\n");
      if(tacSetX(&tac, &status)!=TPCERROR_OK) {
        fprintf(stderr, "Error: data does not contain valid frame times.\n");
        tacFree(&tac); return(4);
      }
    }
    tac.isframe=1;
    /* Set tacNr to zero to save only frame times */
    int bnr=tac.tacNr; tac.tacNr=0;
    /* Save edited TAC data as SIF */
    if(verbose>1) printf("writing %s\n", siffile);
    FILE *fp; fp=fopen(siffile, "w");
    if(fp==NULL) {
      fprintf(stderr, "Error: cannot open file for writing (%s)\n", siffile);
      tacFree(&tac); return(5);
    }
    ret=tacWriteSIF(&tac, fp, 0, &status);
    fclose(fp);
    if(ret!=TPCERROR_OK) {
      fprintf(stderr, "Error (%d): %s\n", ret, errorMsg(status.error));
      tacFree(&tac); return(6);
    }
    tac.tacNr=bnr;
    if(m2s!=0) tacXUnitConvert(&sif, UNIT_MIN, &status);
  }


  /*
   *  List frame lengths on screen
   */
  if(opMode==1 && tac.isframe==0) {
    fprintf(stderr, "Error: no frame start and end times in %s\n", tacfile);
    tacFree(&tac); return(9);
  }
  if(opMode==1 || verbose>1) {
    fprintf(stdout, "# Frame start and length\n");
    for(int fi=0; fi<tac.sampleNr; fi++) {
      fprintf(stdout, "%g %g\n", tac.x1[fi], tac.x2[fi]-tac.x1[fi]);
    }
  }


  /* Tell the user what he did */
  if(verbose>0) {
    if(opMode==0) {
      printf("New frames saved in datafile %s\n", tacfile);
    } else if(opMode==1) {
      fprintf(stdout, "# %d frames were found in %s\n", tac.sampleNr, tacfile);
    } else if(opMode==2) {
      printf("# %d frames were found in %s; saved in %s\n",
             tac.sampleNr, tacfile, siffile);
    }
  }
       
  tacFree(&tac);
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/// @endcond
