/** @file tacjoin.c
 *  @brief Join all TACs from given TAC files into one file.
 *  @copyright (c) Turku PET Centre
 *  @author Vesa Oikonen
 */
/// @cond
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include <unistd.h>
/*****************************************************************************/
#include "tpcextensions.h"
#include "tpcift.h"
#include "tpctac.h"
/*****************************************************************************/

/*****************************************************************************/
static char *info[] = {
  "Join all TACs in TAC files into one file, keeping each TAC separate.",
  "Sample times can be different. Mid frame time is always used.",
  " ",
  "Usage: @P [options] outputfile tacfiles",
  " ",
  "Options:",
  " --force",
  "     Program does not mind if the time or calibration units cannot be",
  "     converted to match.",
  " -stdoptions", // List standard options like --help, -v, etc
  " ",
  "See also: tacblend, tacadd, avgttac, tacnames, tacformat, tac2xml",
  " ",
  "Keywords: TAC, tool, plotting",
  0};
/*****************************************************************************/

/*****************************************************************************/
/* Turn on the globbing of the command line, since it is disabled by default in
   mingw-w64 (_dowildcard=0); in MinGW32 define _CRT_glob instead, if necessary;
   In Unix&Linux wildcard command line processing is enabled by default. */
/*
#undef _CRT_glob
#define _CRT_glob -1
*/
int _dowildcard = -1;
/*****************************************************************************/

/*****************************************************************************/
/**
 *  Main
 */
int main(int argc, char **argv)
{
  int ai, help=0, version=0, verbose=1;
  int forceMode=0;
  char tacfile[FILENAME_MAX], outfile[FILENAME_MAX];
  int fileNr=0, file1=0;


  /*
   *  Get arguments
   */
  if(argc==1) {tpcPrintUsage(argv[0], info, stderr); return(1);}
  tacfile[0]=outfile[0]=(char)0;
  /* Options */
  for(ai=1; ai<argc; ai++) if(*argv[ai]=='-') {
    if(tpcProcessStdOptions(argv[ai], &help, &version, &verbose)==0) continue;
    char *cptr=argv[ai]+1; if(*cptr=='-') cptr++; if(!*cptr) continue;
    if(strcasecmp(cptr, "F")==0 || strcasecmp(cptr, "FORCE")==0) {
      forceMode=1; continue;
    }
    fprintf(stderr, "Error: invalid option '%s'\n", argv[ai]);
    return(1);
  } else break; // tac name argument may start with '-'

  TPCSTATUS status; statusInit(&status);
  statusSet(&status, __func__, __FILE__, __LINE__, TPCERROR_OK);
  status.verbose=verbose-3;
  
  /* Print help or version? */
  if(help==2) {tpcHtmlUsage(argv[0], info, ""); return(0);}
  if(help) {tpcPrintUsage(argv[0], info, stdout); return(0);}
  if(version) {tpcPrintBuild(argv[0], stdout); return(0);}

  /* Process other arguments, starting from the first non-option */
  if(ai<argc) strlcpy(outfile, argv[ai++], FILENAME_MAX);
  for(; ai<argc; ai++) {
    if(fileNr==0) file1=ai;
    fileNr++;
  }

  /* In verbose mode print arguments and options */
  if(verbose>1) {
    for(ai=0; ai<argc; ai++) printf("%s ", argv[ai]); 
    printf("\n");
    printf("fileNr := %d\n", fileNr);
    printf("outfile := %s\n", outfile);
    printf("forceMode := %d\n", forceMode);
    fflush(stdout);
  }

  /* Is something missing? */
  if(!outfile[0]) {tpcPrintUsage(argv[0], info, stdout); return(1);}
  if(fileNr==0) {
    fprintf(stderr, "Error: missing command-line argument; try %s --help\n", argv[0]);
    return(1);
  }
  if(fileNr==1) {
    fprintf(stderr, "Error: only one input file specified.\n");
    return(1);
  }
  
  /* Check that all input files do exist, and that their name does not match output file name */
  for(ai=file1; ai<argc; ai++) {
    strlcpy(tacfile, argv[ai], FILENAME_MAX);
    if(access(tacfile, 0) == -1) {
      fprintf(stderr, "Error: input file %s does not exist.\n", tacfile);
      return(2);
    }
    if(strcasecmp(outfile, tacfile)==0) {
      fprintf(stderr, "Error: input file would be overwritten.\n");
      return(2);
    }
  }


  /*
   *  Read all TAC files
   */
  /* Allocate memory for an array of TACs */
  TAC *taclist;
  taclist=(TAC*)malloc(fileNr*sizeof(TAC));
  if(taclist==NULL) {
    fprintf(stderr, "Error: out of memory.\n");
    return(3);
  }
  /* Initiate TAC structures */
  for(int i=0; i<fileNr; i++) tacInit(taclist+i);
  /* Read TAC files */
  for(ai=file1; ai<argc; ai++) {
    strlcpy(tacfile, argv[ai], FILENAME_MAX);
    if(verbose>1) printf("reading %s\n", tacfile);
    int i=ai-file1;
    if(tacRead(taclist+i, tacfile, &status)) {
      fprintf(stderr, "Error: %s\n", errorMsg(status.error));
      for(int j=0; j<i; j++) tacFree(taclist+j);
      free(taclist); return(3);
    }
    if(verbose>2) {
      printf("fileformat := %s\n", tacFormattxt(taclist[i].format));
      printf("tacNr := %d\n", taclist[i].tacNr);
      printf("sampleNr := %d\n", taclist[i].sampleNr);
    }
    /* Use mid times */
    taclist[i].isframe=0;
  }

  if(verbose>1) printf("checking data...\n");

  /* Convert TAC units when necessary */
  for(int i=1; i<fileNr; i++) {
    int ret=tacXUnitConvert(taclist+i, taclist[0].tunit, &status);
    if(ret==TPCERROR_UNKNOWN_UNIT) ret=TPCERROR_OK;
    if(ret==TPCERROR_OK) {
      ret=tacYUnitConvert(taclist+i, taclist[0].cunit, &status);
      if(ret==TPCERROR_UNKNOWN_UNIT) ret=TPCERROR_OK;
    }
    if(ret!=TPCERROR_OK) { // failed
      /* failing is a problem if units are to be verified */
      if(!forceMode) {
        fprintf(stderr, "Error: non-compatible TAC units.\n");
        for(int i=0; i<fileNr; i++) tacFree(taclist+i);
        free(taclist); return(4);
      }
    }
  }

  /* Add the TACs to the first structure */
  int rs[fileNr], cs[fileNr]; // place for storing data start indices
  rs[0]=cs[0]=0;
  if(verbose>1) {
    printf("adding %d TACs with %d samples\n", taclist[0].tacNr, taclist[0].sampleNr);
    fflush(stdout);
  }
  for(int i=1; i<fileNr; i++) {
    cs[i]=taclist[0].tacNr; // always add to the first free column
    if(taclist[i].tacNr<1) continue;
    if(verbose>1) {
      printf("adding %d TACs with %d samples\n", taclist[i].tacNr, taclist[i].sampleNr);
      fflush(stdout);
    }
    /* See if any of the previous structures had the same sample times */
    int ti=-1;
    for(int j=0; j<i; j++) {
      if(taclist[i].tacNr<1 || taclist[j].tacNr<1) continue;
      if(tacCompareTimes(taclist+j, taclist+i, 0.001, 0.001, NULL)==0) {ti=j; break;}
    }
    if(verbose>1 && ti>=0) {
      printf("  equal sample times in %s and %s\n", argv[file1+i], argv[file1+ti]); fflush(stdout);}
    if(ti<0) rs[i]=taclist[0].sampleNr; else rs[i]=rs[ti];
    if(verbose>2) printf("  adding TACs to index %d,%d\n", rs[i], cs[i]);
    /* Add space for TACs, and for samples, if different sample times */
    int ret=TPCERROR_OK;
    if(ti<0) ret=tacAllocateMoreSamples(taclist+0, taclist[i].sampleNr);
    if(ret==TPCERROR_OK) ret=tacAllocateMore(taclist+0, taclist[i].tacNr);
    if(ret!=TPCERROR_OK) {
      fprintf(stderr, "Error: cannot allocate memory.\n");
      for(int j=0; j<i; j++) tacFree(taclist+j);
      free(taclist); return(6);
    }
    if(verbose>3) {
      printf("  allocated for %d TACs and %d samples\n", taclist[0]._tacNr, taclist[0]._sampleNr);
      fflush(stdout);
    }
    /* If different sample times, copy those 'below' the previous data */
    if(ti<0) {
      if(verbose>3) {printf("  adding new sample times\n"); fflush(stdout);}
      for(int j=0; j<taclist[i].sampleNr; j++) {
        taclist[0].x[taclist[0].sampleNr+j]=taclist[i].x[j];
        taclist[0].x1[taclist[0].sampleNr+j]=taclist[i].x1[j];
        taclist[0].x2[taclist[0].sampleNr+j]=taclist[i].x2[j];
      }
      taclist[0].sampleNr+=taclist[i].sampleNr;
    }
    /* Copy the y values and TAC names */
    if(verbose>3) {printf("  adding new TACs\n"); fflush(stdout);}
    for(int ci=0; ci<taclist[i].tacNr; ci++) {
      strcpy(taclist[0].c[cs[i]+ci].name, taclist[i].c[ci].name);
      for(int ri=0; ri<taclist[i].sampleNr; ri++) {
        taclist[0].c[cs[i]+ci].y[rs[i]+ri]=taclist[i].c[ci].y[ri];
      }
    }
    taclist[0].tacNr+=taclist[i].tacNr;
  }

  /* Remove weights */
  taclist[0].weighting=WEIGHTING_OFF;
  /* Free the header data */
  iftFree(&taclist[0].h);


  /*
   *  Write joined data
   */
  {
    if(verbose>1) printf("writing %s\n", outfile);
    FILE *fp; fp=fopen(outfile, "w");
    if(fp==NULL) {
      fprintf(stderr, "Error: cannot open file for writing.\n");
      for(int i=0; i<fileNr; i++) tacFree(taclist+i);
      free(taclist);
      return(11);
    }
    int ret=tacWrite(taclist+0, fp, TAC_FORMAT_UNKNOWN, 1, &status);
    fclose(fp);
    if(ret!=TPCERROR_OK) {
      fprintf(stderr, "Error: %s\n", errorMsg(status.error));
      for(int i=0; i<fileNr; i++) tacFree(taclist+i);
      free(taclist);
      return(12);
    }
    if(verbose>=0) {
      printf("%d TACs from %d files saved in %s\n", taclist[0].tacNr, fileNr, outfile);
    }
  }

  /* Free memory */
  for(int i=0; i<fileNr; i++) tacFree(taclist+i);
  free(taclist);

  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/// @endcond
