/** @file taclist.c
    @brief List tac file regions.
    @copyright (c) Turku PET Centre
    @author Vesa Oikonen
 */
/// @cond
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
/*****************************************************************************/
#include "tpcextensions.h"
#include "tpcift.h"
#include "tpctac.h"
/*****************************************************************************/

/*****************************************************************************/
static char *info[] = {
  "List on screen the specified or all TACs of a TAC data file.",
  "The concentrations are shown, if file contains <5 samples (time frames).",
  "Accepts at least PMOD, DFT, and SIF formats.",
  " ",
  "Usage: @P [options] filename [tacname1 [tacname2...]] [> outputfile]",
  " ",
  "Options:",
  " -nv",
  "     Do not print region volume.",
  " -nn",
  "     Do not print region name.",
  " -na",
  "     Activity concentrations are never printed.",
  " -ift",
  "     Interfile-type print.",
  " -weights",
  "     List weights with Interfile-type print.",
  " -stdoptions", // List standard options like --help, -v, etc
  " ",
  "Example:",
  "     @P iea345.tac cer",
  " ",
  "See also: tacmatch, tacformat, tacren, tacnames, dft2csv, tac2svg, tacsort",
  " ",
  "Keywords: TAC, SIF, tool",
  0};
/*****************************************************************************/

/*****************************************************************************/
/* Turn on the globbing of the command line, since it is disabled by default in
   mingw-w64 (_dowildcard=0); in MinGW32 define _CRT_glob instead, if necessary;
   In Unix&Linux wildcard command line processing is enabled by default. */
/*
#undef _CRT_glob
#define _CRT_glob -1
*/
int _dowildcard = -1;
/*****************************************************************************/

/*****************************************************************************/
/**
 *  Main
 */
int main(int argc, char **argv)
{
  int ai, help=0, version=0, verbose=1;
  int ret;
  int printVol=1, printName=1, printType=0, printAct=1, printWeights=0;
  char *cptr, tacfile[FILENAME_MAX];

#ifdef MINGW
  // Use Unix/Linux default of two-digit exponents in MinGW on Windows
  _set_output_format(_TWO_DIGIT_EXPONENT);
#endif

  
  /*
   *  Get arguments
   */
  if(argc==1) {tpcPrintUsage(argv[0], info, stderr); return(1);}
  tacfile[0]=(char)0;
  /* Options */
  for(ai=1; ai<argc; ai++) if(*argv[ai]=='-') {
    if(tpcProcessStdOptions(argv[ai], &help, &version, &verbose)==0) continue;
    cptr=argv[ai]+1; if(*cptr=='-') cptr++; if(!*cptr) continue;
    if(strcasecmp(cptr, "NV")==0) {
      printVol=0; continue;
    } else if(strcasecmp(cptr, "NN")==0) {
      printName=0; continue;
    } else if(strcasecmp(cptr, "NA")==0) {
      printAct=0; continue;
    } else if(strncasecmp(cptr, "WEIGHTS", 1)==0) {
      printWeights=1; continue;
    } else if(strcasecmp(cptr, "IFT")==0) {
      printType=1; continue;
    }
    fprintf(stderr, "Error: invalid option '%s'\n", argv[ai]);
    return(1);
  } else break; // tac name argument may start with '-'
  
  /* Print help or version? */
  if(help==2) {tpcHtmlUsage(argv[0], info, ""); return(0);}
  if(help) {tpcPrintUsage(argv[0], info, stdout); return(0);}
  if(version) {tpcPrintBuild(argv[0], stdout); return(0);}

  /* In verbose mode print arguments and options */
  if(verbose>1) {
    printf("print_name := %d\n", printName);
    printf("print_volume := %d\n", printVol);
    printf("print_value := %d\n", printAct);
    printf("print_type := %d\n", printType);
    printf("print_weight := %d\n", printWeights);
  }

  TPCSTATUS status; statusInit(&status);
  statusSet(&status, __func__, __FILE__, __LINE__, TPCERROR_OK);
  status.verbose=verbose-1;

  /* Process other arguments, starting from the first non-option */
  TAC tac; tacInit(&tac);
  int n=0, m;
  for(; ai<argc; ai++) {
    if(!tacfile[0]) {
      strlcpy(tacfile, argv[ai], FILENAME_MAX);
      /* Read TAC file */
      if(verbose>1) printf("reading %s\n", tacfile);
      ret=tacRead(&tac, tacfile, &status);
      if(ret) {
        fprintf(stderr, "Error (%d): %s\n", ret, errorMsg(status.error));
        tacFree(&tac); return(2);
      }
      if(verbose>0) {
        printf("fileformat := %s\n", tacFormattxt(tac.format));
        printf("tacNr := %d\n", tac.tacNr);
        printf("sampleNr := %d\n", tac.sampleNr);
      }
    } else { /* Next parameters are region names; select them */
      m=tacSelectTACs(&tac, argv[ai], 0, &status);
      if(verbose>0) printf("%d tac(s) match name '%s'\n", m, argv[ai]);
      n++;
    }
  }
  /* Check that we got what we need */
  if(!tacfile[0]) {tpcPrintUsage(argv[0], info, stderr); tacFree(&tac); return(1);}
  if(printType==0 && printWeights) {
    fprintf(stderr, "Warning: weights can only be listed in Interfile-style.\n");
    fflush(stderr);
  }

  /* If TAC names were specified but none matched, then that is an error */
  if(n>0 && tacSelectedTACs(&tac)<1) {
    fprintf(stderr, "Error: specified TAC names were not found.\n");
    tacFree(&tac); return(3);
  }  
    
  /* If no tac names were specified, then select all TACs */
  if(n==0) m=tacSelectTACs(&tac, NULL, 0, &status);
    
    
  /*
   *  List all selected TACs
   */

  /* Print title (not in silent mode or when printing IFT) */
  if(verbose>=0 && printType==0) {
    printf("Nr\t");
    if(printName!=0) printf("TAC-name\t");
    if(printVol!=0) printf("Volume\t");
    n=5; if(printVol==0) n++;
    if(printAct!=0 && tac.sampleNr<n) {
      printf("TAC-values");
      if(tac.cunit!=UNIT_UNKNOWN) printf("[%s]", unitName(tac.cunit));
    }
    printf("\n");
  }

  /* If SIF out, then print header info */
  if(printType==1) {
    iftWrite(&tac.h, stdout, NULL);
  }
   
  /* List selected ROIs */
  for(int ri=0; ri<tac.tacNr; ri++) if(tac.c[ri].sw) {
    if(printType==0) {
      printf("%d", ri+1);
      if(printName) {
        if(strlen(tac.c[ri].name)>0) printf("\t%s", tac.c[ri].name);
        else printf("\t--------");
      }
      n=5; if(printVol==0) n++;
      if(printAct!=0 && tac.sampleNr<n) {
        if(printVol) printf("\t%g", tac.c[ri].size);
        for(int fi=0; fi<tac.sampleNr; fi++)
          printf("\t%.3e", tac.c[ri].y[fi]);
        printf("\n");
      } else {
        if(printVol) printf("\t%g\n", tac.c[ri].size);
        else printf("\n");
      }
    } else { /* IFT format */
      if(printName) printf("tacname[%d] := %s\n", ri+1, tac.c[ri].name);
      if(printVol) printf("volume[%d] := %g\n", ri+1, tac.c[ri].size);
      if(printAct!=0) {
        for(int fi=0; fi<tac.sampleNr; fi++)
          printf("value[%d][%d] := %g\n", ri+1, fi+1, tac.c[ri].y[fi]);
      }
    }
  }
  fflush(stdout);

  /* List weights */
  if(printType!=0 && printWeights) {
    printf("weights := ");
    if(tacIsWeighted(&tac)) printf("yes\n"); else printf("no\n");
    if(tacIsWeighted(&tac)) {
      for(int fi=0; fi<tac.sampleNr; fi++)
        printf("weight[%d] := %g\n", fi+1, tac.w[fi]);
    }
    fflush(stdout);
  }
   
  tacFree(&tac);
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/// @endcond
