/** @file tacln.c
 *  @brief Natural logarithm of TAC data.
 *  @copyright (c) Turku PET Centre
 *  @author Vesa Oikonen
 */
/// @cond
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
/*****************************************************************************/
#include "tpcextensions.h"
#include "tpctac.h"
/*****************************************************************************/

/*****************************************************************************/
static char *info[] = {
  "Calculates natural logarithm of y values (concentrations) in TAC file, or",
  "optionally of TAC x values (sample times).",
  " ",
  "Usage: @P [options] tacfile outputfile",
  " ",
  "Options:",
  " -x=<y|N>",
  "     Calculate (y) or do not calculate (n, default) natural logarithm of x",
  "     values.",
  " -y=<Y|n>",
  "     Calculate (y, default) or do not calculate (n) natural logarithm of y",
  "     values.",
  " -stdoptions", // List standard options like --help, -v, etc
  " ",
  "See also: taccalc, tacinv, tacadd, taccut, tacsetx, fit_line, fit_exp",
  " ",
  "Keywords: TAC, tool",
  0};
/*****************************************************************************/

/*****************************************************************************/
/* Turn on the globbing of the command line, since it is disabled by default in
   mingw-w64 (_dowildcard=0); in MinGW32 define _CRT_glob instead, if necessary;
   In Unix&Linux wildcard command line processing is enabled by default. */
/*
#undef _CRT_glob
#define _CRT_glob -1
*/
int _dowildcard = -1;
/*****************************************************************************/

/*****************************************************************************/
/**
 *  Main
 */
int main(int argc, char **argv)
{
  int   ai, help=0, version=0, verbose=1;
  int   ret;
  char  tacfile[FILENAME_MAX], outfile[FILENAME_MAX];
  char *cptr;
  TAC   tac;
  int   xvals=0;
  int   yvals=1;


  /*
   *  Get arguments
   */
  if(argc==1) {tpcPrintUsage(argv[0], info, stderr); return(1);}
  tacInit(&tac);
  tacfile[0]=outfile[0]=(char)0;
  /* Options */
  for(ai=1; ai<argc; ai++) if(*argv[ai]=='-') {
    if(tpcProcessStdOptions(argv[ai], &help, &version, &verbose)==0) continue;
    cptr=argv[ai]+1; if(*cptr=='-') cptr++; if(!*cptr) continue;
    if(strncasecmp(cptr, "X=", 2)==0) {
      cptr+=2;
      if(strncasecmp(cptr, "YES", 1)==0) {xvals=1; continue;}
      else if(strncasecmp(cptr, "NO", 1)==0) {xvals=0; continue;}
    } else if(strncasecmp(cptr, "Y=", 2)==0) {
      cptr+=2;
      if(strncasecmp(cptr, "YES", 1)==0) {yvals=1; continue;}
      else if(strncasecmp(cptr, "NO", 1)==0) {yvals=0; continue;}
    }
    fprintf(stderr, "Error: invalid option '%s'.\n", argv[ai]);
    return(1);
  } else break; // tac name argument may start with '-'

  TPCSTATUS status; statusInit(&status);
  statusSet(&status, __func__, __FILE__, __LINE__, TPCERROR_OK);
  status.verbose=verbose-1;
  
  /* Print help or version? */
  if(help==2) {tpcHtmlUsage(argv[0], info, ""); return(0);}
  if(help) {tpcPrintUsage(argv[0], info, stdout); return(0);}
  if(version) {tpcPrintBuild(argv[0], stdout); return(0);}

  /* Arguments */
  for(; ai<argc; ai++) {
    if(!tacfile[0]) {
      strcpy(tacfile, argv[ai]); continue;
    } else if(!outfile[0]) {
      strcpy(outfile, argv[ai]); continue;
    }
    fprintf(stderr, "Error: invalid argument '%s'.\n", argv[ai]);
    return(1);
  }

  /* Is something missing? */
  if(!outfile[0]) {tpcPrintUsage(argv[0], info, stdout); return(1);}

  
  /* In verbose mode print arguments and options */
  if(verbose>1) {
    for(ai=0; ai<argc; ai++)
      printf("%s ", argv[ai]); 
    printf("\n");
    printf("tacfile := %s\n", tacfile);
    printf("outfile := %s\n", outfile);
    printf("xvals := %d\n", xvals);
    printf("yvals := %d\n", yvals);
  }
  if(xvals==0 && yvals==0) {
    fprintf(stderr, "Error: no transforms requested.\n");
    return(2);
  }


  /*
   *  Read the file
   */
  if(verbose>1) printf("reading %s\n", tacfile);
  ret=tacRead(&tac, tacfile, &status);
  if(ret!=TPCERROR_OK) {
    fprintf(stderr, "Error (%d): %s\n", ret, errorMsg(status.error));
    tacFree(&tac); return(2);
  }
  if(verbose>2) {
    printf("fileformat := %s\n", tacFormattxt(tac.format));
    printf("tacNr := %d\n", tac.tacNr);
    printf("sampleNr := %d\n", tac.sampleNr);
    printf("xunit := %s\n", unitName(tac.tunit));
    printf("yunit := %s\n", unitName(tac.cunit));
  }


  int okNr=0, failNr=0;
  /*
   *  Invert the y data, if required
   */
  if(yvals) {
    for(int ri=0; ri<tac.tacNr; ri++) {
      for(int fi=0; fi<tac.sampleNr; fi++) {
        if(isnan(tac.c[ri].y[fi])) {failNr++; continue;}
        if(tac.c[ri].y[fi]<1.0E-20) {
          tac.c[ri].y[fi]=nan(""); failNr++; continue;}
        tac.c[ri].y[fi]=log(tac.c[ri].y[fi]);
        if(isfinite(tac.c[ri].y[fi])) okNr++; 
        else {tac.c[ri].y[fi]=nan(""); failNr++;}
      }
    }
  }
  /*
   *  Invert the x data, if required
   */
  if(xvals) {
    if(tac.isframe==0) {
      for(int fi=0; fi<tac.sampleNr; fi++) {
        if(isnan(tac.x[fi])) {failNr++; continue;}
        if(tac.x[fi]<1.0E-20) {tac.x[fi]=nan(""); failNr++; continue;}
        tac.x[fi]=log(tac.x[fi]);
        if(isfinite(tac.x[fi])) okNr++; 
        else {tac.x[fi]=nan(""); failNr++;}
      }
    } else {
      for(int fi=0; fi<tac.sampleNr; fi++) {
        if(isnan(tac.x1[fi])) {failNr++;}
        else if(tac.x1[fi]<1.0E-20) {tac.x1[fi]=nan(""); failNr++;}
        else {
          tac.x1[fi]=log(tac.x1[fi]);
          if(isfinite(tac.x1[fi])) okNr++; 
          else {tac.x1[fi]=nan(""); failNr++;}
        }
        if(isnan(tac.x2[fi])) {failNr++;}
        else if(tac.x2[fi]<1.0E-20) {tac.x2[fi]=nan(""); failNr++;}
        else {
          tac.x2[fi]=log(tac.x2[fi]);
          if(isfinite(tac.x2[fi])) okNr++; 
          else {tac.x2[fi]=nan(""); failNr++;}
        }
      }
    }
  }

  if(verbose>1) {
    printf("  failNr := %d\n", failNr);
    printf("  okNr := %d\n", okNr);
  }
  if(okNr<1) {
    fprintf(stderr, "Error: all transformations failed.\n");
    tacFree(&tac); return(3);
  }


  /*
   *  Save data 
   */
  if(verbose>1) printf("writing %s\n", outfile);
  FILE *fp; fp=fopen(outfile, "w");
  if(fp==NULL) {
    fprintf(stderr, "Error: cannot open file for writing (%s)\n", outfile);
    tacFree(&tac); return(11);
  }
  ret=tacWrite(&tac, fp, TAC_FORMAT_UNKNOWN, 1, &status);
  fclose(fp); tacFree(&tac);
  if(ret!=TPCERROR_OK) {
    fprintf(stderr, "Error (%d): %s\n", ret, errorMsg(status.error));
    return(12);
  }
  if(verbose>=0) printf("%s saved.\n", outfile);

  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/// @endcond
