/** @file tacmsamp.c
 *  @brief Replace multiple samples in TAC file with their mean.
 *  @copyright (c) Turku PET Centre
 *  @author Vesa Oikonen
 */
/// @cond
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
/*****************************************************************************/
#include "tpcextensions.h"
#include "tpcift.h"
#include "tpctac.h"
/*****************************************************************************/

/*****************************************************************************/
static char *info[] = {
  "Inspect TAC file for multiple samples with the same sample time.",
  " ",
  "Usage: @P [options] file",
  " ",
  "Options:",
  " -mean",
  "     Replace multiple samples with their mean.",
  " -stdoptions", // List standard options like --help, -v, etc
  " ",
  "Example 1. Check all TAC files for multiple samples in bash shell:",
  "     for file in ./*.tac; do @P $file; done",
  " ",
  "Example 2. Combine multiple samples inside TAC files in Windows command",
  "           prompt window: ",
  "     for %g in (*.tac) do @P -mean %g",
  " ",
  "See also: interpol, tacblend, taccut, taccat, tacsetx, tacframe",
  " ",
  "Keywords: TAC, tool, time",
  0};
/*****************************************************************************/

/*****************************************************************************/
/* Turn on the globbing of the command line, since it is disabled by default in
   mingw-w64 (_dowildcard=0); in MinGW32 define _CRT_glob instead, if necessary;
   In Unix&Linux wildcard command line processing is enabled by default. */
/*
#undef _CRT_glob
#define _CRT_glob -1
*/
int _dowildcard = -1;
/*****************************************************************************/

/*****************************************************************************/
/**
 *  Main
 */
int main(int argc, char **argv)
{
  int ai, help=0, version=0, verbose=1;
  int fixMode=0; // 0= change nothing, 1=replace with mean
  char tacfile[FILENAME_MAX];

  
  /*
   *  Get arguments
   */
  if(argc==1) {tpcPrintUsage(argv[0], info, stderr); return(1);}
  tacfile[0]=(char)0;
  /* Options */
  for(ai=1; ai<argc; ai++) if(*argv[ai]=='-') {
    if(tpcProcessStdOptions(argv[ai], &help, &version, &verbose)==0) continue;
    char *cptr=argv[ai]+1; if(*cptr=='-') cptr++; if(!*cptr) continue;
    if(strcasecmp(cptr, "MEAN")==0) {
      fixMode=1; continue;
    }
    fprintf(stderr, "Error: invalid option '%s'\n", argv[ai]);
    return(1);
  } else break; // tac name argument may start with '-'

  TPCSTATUS status; statusInit(&status);
  statusSet(&status, __func__, __FILE__, __LINE__, TPCERROR_OK);
  status.verbose=verbose-3;
  
  /* Print help or version? */
  if(help==2) {tpcHtmlUsage(argv[0], info, ""); return(0);}
  if(help) {tpcPrintUsage(argv[0], info, stdout); return(0);}
  if(version) {tpcPrintBuild(argv[0], stdout); return(0);}

  /* The first argument (non-option) is the filename */
  if(ai<argc) {strlcpy(tacfile, argv[ai], FILENAME_MAX); ai++;}
  else {fprintf(stderr, "Error: missing filename.\n"); return(1);}
  if(ai<argc) {fprintf(stderr, "Error: extra command-line argument.\n"); return(1);}
  
  /* In verbose mode print arguments and options */
  if(verbose>1) {
    printf("tacfile := %s\n", tacfile);
    printf("fixMode := %d\n", fixMode);
    fflush(stdout);
  }

  /* 
   *  Read the file
   */
  if(verbose>1) printf("reading %s\n", tacfile);
  TAC tac; tacInit(&tac);
  if(tacRead(&tac, tacfile, &status)!=TPCERROR_OK) {
    fprintf(stderr, "Error: %s\n", errorMsg(status.error));
    tacFree(&tac); return(2);
  }
  if(verbose>2) {
    printf("fileformat := %s\n", tacFormattxt(tac.format));
    printf("tacNr := %d\n", tac.tacNr);
    printf("sampleNr := %d\n", tac.sampleNr);
  }

  /*
   *  Check for multiple samples
   */
  int isMultiple=tacMultipleSamples(&tac, 0, NULL, verbose-1);
  if(fixMode==0 || verbose>0) {
    printf("%s : ", tacfile);
    if(isMultiple) printf("multiple samples found\n"); else printf("no multiple samples\n");
    fflush(stdout);
  }
  if(fixMode==0 || !isMultiple) {tacFree(&tac); return(0);}

  /*
   *  Fix multiple samples
   */
  TAC tac2; tacInit(&tac2);
  if(tacMultipleSamples(&tac, fixMode, &tac2, verbose-1)!=0) {
    fprintf(stderr, "Error: cannot fix multiple samples.\n");
    tacFree(&tac); tacFree(&tac2); return(9);
  }
  /* Compare the sample numbers */
  if(tac.sampleNr==tac2.sampleNr) {
    printf("%s : no multiple samples fixed\n", tacfile);
    tacFree(&tac); tacFree(&tac2); return(0);
  }
  if(verbose>0) printf("%d samples reduced to %d\n", tac.sampleNr, tac2.sampleNr);
  tacFree(&tac);


  /*
   *  Save data 
   */
  {
    if(verbose>1) printf("writing %s\n", tacfile);
    FILE *fp; fp=fopen(tacfile, "w");
    if(fp==NULL) {
      fprintf(stderr, "Error: cannot open file for writing (%s)\n", tacfile);
      tacFree(&tac2); return(11);
    }
    int ret=tacWrite(&tac2, fp, TAC_FORMAT_UNKNOWN, 1, &status);
    fclose(fp); tacFree(&tac2);
    if(ret!=TPCERROR_OK) {
      fprintf(stderr, "Error: %s\n", errorMsg(status.error));
      return(12);
    }
  }

  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/// @endcond
