/** @file tacnames.c
 *  @brief List TAC names inside TAC files.
 *  @copyright (c) Turku PET Centre
 *  @author Vesa Oikonen
 */
/// @cond
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <math.h>
/*****************************************************************************/
#include "tpcextensions.h"
#include "tpcift.h"
#include "tpctac.h"
/*****************************************************************************/

/*****************************************************************************/
static char *info[] = {
  "List TAC names inside a TAC files.",
  " ",
  "Usage: @P [options] files",
  " ",
  "Options:",
  " -missing",
  "     List only files that contain TAC(s) with missing name.",
  " -errorifmissing",
  "     Return error code if TAC(s) with missing name.",
  " -stdoptions", // List standard options like --help, -v, etc
  " ",
  "See also: taclist, tacren, tacadd, tacformat, iftren",
  " ",
  "Keywords: TAC, tool, rename, simulation",
  0};
/*****************************************************************************/

/*****************************************************************************/
/* Turn on the globbing of the command line, since it is disabled by default in
   mingw-w64 (_dowildcard=0); in MinGW32 define _CRT_glob instead, if necessary;
   In Unix&Linux wildcard command line processing is enabled by default. */
/*
#undef _CRT_glob
#define _CRT_glob -1
*/
int _dowildcard = -1;
/*****************************************************************************/

/*****************************************************************************/
/**
 *  Main
 */
int main(int argc, char **argv)
{
  int ai, help=0, version=0, verbose=1;
  int ret, listMissing=0, errorIfMissing=0;
  char *cptr;

  
  /*
   *  Get arguments
   */
  if(argc==1) {tpcPrintUsage(argv[0], info, stderr); return(1);}
  /* Options */
  for(ai=1; ai<argc; ai++) if(*argv[ai]=='-') {
    if(tpcProcessStdOptions(argv[ai], &help, &version, &verbose)==0) continue;
    cptr=argv[ai]+1; if(*cptr=='-') cptr++; if(!*cptr) continue;
    if(strncasecmp(cptr, "MISSING", 3)==0) {
      listMissing=1; continue;
    } else if(strncasecmp(cptr, "ERRORIFMISSING", 3)==0) {
      errorIfMissing=1; continue;
    }
    fprintf(stderr, "Error: invalid option '%s'\n", argv[ai]);
    return(1);
  } else break; // tac name argument may start with '-'

  TPCSTATUS status; statusInit(&status);
  statusSet(&status, __func__, __FILE__, __LINE__, TPCERROR_OK);
  status.verbose=verbose-1;
  
  /* Print help or version? */
  if(help==2) {tpcHtmlUsage(argv[0], info, ""); return(0);}
  if(help) {tpcPrintUsage(argv[0], info, stdout); return(0);}
  if(version) {tpcPrintBuild(argv[0], stdout); return(0);}

  /* Store file names */
  if(ai>=argc) {fprintf(stderr, "Error: missing filenames.\n"); return(1);}
  IFT fnames; iftInit(&fnames);
  for(; ai<argc; ai++) iftPut(&fnames, "", argv[ai], 0, &status);
  
  
  /* In verbose mode print arguments and options */
  if(verbose>1) {
    printf("listMissing := %d\n", listMissing);
    printf("errorIfMissing := %d\n", errorIfMissing);
    printf("fileNr := %d\n", fnames.keyNr);
  }


  /* Collect TAC names */
  int missingNr=0;
  IFT tnames; iftInit(&tnames);
  {
    TAC tac; tacInit(&tac);
    char tacfile[FILENAME_MAX], tacname[256];
    for(int fi=0; fi<fnames.keyNr; fi++) {
      strlcpy(tacfile, fnames.item[fi].value, FILENAME_MAX);
      if(verbose>2) printf("reading %s\n", tacfile);
      ret=tacRead(&tac, tacfile, &status);
      if(ret!=TPCERROR_OK) {
        fprintf(stderr, "Error: cannot read %s\n", tacfile); 
        fprintf(stderr, "Error: %s\n", errorMsg(status.error)); 
        iftFree(&fnames); iftFree(&tnames); 
        return(1);
      }
      for(int ti=0; ti<tac.tacNr; ti++) {
        strlcpy(tacname, tac.c[ti].name, 256);
        if(verbose>3) printf("\t%d\t%s\n", 1+ti, tacname);
        if(!tacname[0] || strcmp(tacname, ".")==0) {missingNr++; continue;}
        /* Check that TAC name is not already listed */
        if(iftFindKey(&tnames, tacname, 0)>=0) continue;
        iftPut(&tnames, tacname, "", 0, NULL);
      }
      tacFree(&tac);
    }
  }
  if(verbose>2) {
    printf("\n%d missing TAC names.\n", missingNr);
    printf("\nTAC names:\n==========\n");
    for(int ti=0; ti<tnames.keyNr; ti++) printf("%s\n", tnames.item[ti].key);
  }
  if(listMissing && missingNr==0) {
    fprintf(stdout, "Note: no missing TAC names.\n");
    iftFree(&fnames); iftFree(&tnames);
    return(0);
  }

  /* List TAC names, and files that contain each name */
  {
    TAC tac; tacInit(&tac);
    char tacfile[FILENAME_MAX], tacname[256];

    if(listMissing==0) for(int ti=0; ti<tnames.keyNr; ti++) {
      strlcpy(tacname, tnames.item[ti].key, 256);
      printf("%s\n", tacname);
      for(int fi=0; fi<fnames.keyNr; fi++) {
        strlcpy(tacfile, fnames.item[fi].value, FILENAME_MAX);
        if(verbose>10) printf("reading %s\n", tacfile);
        ret=tacRead(&tac, tacfile, &status);
        if(ret!=TPCERROR_OK) {
          fprintf(stderr, "Error: cannot read %s\n", tacfile); 
          fprintf(stderr, "Error: %s\n", errorMsg(status.error)); 
          iftFree(&fnames); iftFree(&tnames); 
          return(1);
        }
        for(int ti=0; ti<tac.tacNr; ti++) {
          if(strcasecmp(tacname, tac.c[ti].name)==0) {
            printf("\t%s\n", tacfile); 
            break;
          }
        }
        tacFree(&tac);
      }
    }

    if(missingNr>0) {
      printf("Missing TAC name\n");
      for(int fi=0; fi<fnames.keyNr; fi++) {
        strlcpy(tacfile, fnames.item[fi].value, FILENAME_MAX);
        if(verbose>10) printf("reading %s\n", tacfile);
        ret=tacRead(&tac, tacfile, &status);
        if(ret!=TPCERROR_OK) {
          fprintf(stderr, "Error: cannot read %s\n", tacfile); 
          fprintf(stderr, "Error: %s\n", errorMsg(status.error)); 
          iftFree(&fnames); iftFree(&tnames); 
          return(1);
        }
        for(int ti=0; ti<tac.tacNr; ti++) {
          strlcpy(tacname, tac.c[ti].name, 256);
          if(verbose>11) printf("\t%d\t%s\n", 1+ti, tacname);
          if(!tacname[0] || strcmp(tacname, ".")==0) {
            printf("\t%s\n", tacfile); 
            break;
          }
        }
        tacFree(&tac);
      }
    }
  }

  iftFree(&fnames); iftFree(&tnames);

  if(errorIfMissing!=0 && missingNr>0) {
    fprintf(stderr, "Error: missing TAC name(s).\n");
    return(9);
  }
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/// @endcond
