/** @file tacsort.c
 *  @brief Sort TACs by region name or sample time.
 *  @copyright (c) Turku PET Centre
 *  @author Vesa Oikonen
 */
/// @cond
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
/*****************************************************************************/
#include "tpcextensions.h"
#include "tpcift.h"
#include "tpctac.h"
/*****************************************************************************/

/*****************************************************************************/
static char *info[] = {
  "Sort PET TAC data by the region name or sample time.",
  " ",
  "Usage: @P [options] filename(s)",
  " ",
  "Options:",
  " -sort=<name|auc|time>",
  "     TACs are sorted by regional name (default), AUC, or TAC samples",
  "     (\"frames\") are sorted by increasing sample time.",
  " -stdoptions", // List standard options like --help, -v, etc
  " ",
  "Example:",
  "     @P -sort=time iea*.tac",
  " ",
  "See also: tacmatch, taclist, tacren, taccat, tacadd, tacframe",
  " ",
  "Keywords: TAC, tool, sorting, time",
  0};
/*****************************************************************************/

/*****************************************************************************/
/* Turn on the globbing of the command line, since it is disabled by default in
   mingw-w64 (_dowildcard=0); in MinGW32 define _CRT_glob instead, if necessary;
   In Unix&Linux wildcard command line processing is enabled by default. */
/*
#undef _CRT_glob
#define _CRT_glob -1
*/
int _dowildcard = -1;
/*****************************************************************************/

/*****************************************************************************/
/**
 *  Main
 */
int main(int argc, char **argv)
{
  int ai, help=0, version=0, verbose=1;
  int ret;
  int sortBy=1; // 1=name, 2=name2, 3=name3, 4=time, 5=AUC
  char *cptr, tacfile[FILENAME_MAX];
  TAC tac;

  
  /*
   *  Get arguments
   */
  if(argc==1) {tpcPrintUsage(argv[0], info, stderr); return(1);}
  tacInit(&tac);
  tacfile[0]=(char)0;
  /* Options */
  for(ai=1; ai<argc; ai++) if(*argv[ai]=='-') {
    if(tpcProcessStdOptions(argv[ai], &help, &version, &verbose)==0) continue;
    cptr=argv[ai]+1; if(*cptr=='-') cptr++; if(!*cptr) continue;
    if(strncasecmp(cptr, "SORT=", 5)==0) {
      cptr+=5;
      if(strcasecmp(cptr, "NAME")==0) {sortBy=1; continue;}
      if(strcasecmp(cptr, "NAME2")==0) {sortBy=2; continue;}
      if(strcasecmp(cptr, "NAME3")==0) {sortBy=3; continue;}
      if(strcasecmp(cptr, "TIME")==0) {sortBy=4; continue;}
      if(strcasecmp(cptr, "X")==0) {sortBy=4; continue;}
      if(strcasecmp(cptr, "AUC")==0) {sortBy=5; continue;}
    }
    fprintf(stderr, "Error: invalid option '%s'\n", argv[ai]);
    return(1);
  } else break; // tac name argument may start with '-'

  TPCSTATUS status; statusInit(&status);
  statusSet(&status, __func__, __FILE__, __LINE__, TPCERROR_OK);
  status.verbose=verbose-1;

  /* Print help or version? */
  if(help==2) {tpcHtmlUsage(argv[0], info, ""); return(0);}
  if(help) {tpcPrintUsage(argv[0], info, stdout); return(0);}
  if(version) {tpcPrintBuild(argv[0], stdout); return(0);}

  /* In verbose mode print arguments and options */
  if(verbose>1) {
    printf("sortBy := %d\n", sortBy);
  }

  /* Check options */
  if(sortBy==2 || sortBy==3) {
    fprintf(stderr, "Error: specified sort option is not yet supported.\n");
    return(1);
  }

  /* Process other arguments, starting from the first non-option */
  int n=0;
  for(; ai<argc; ai++) {
    /* It should be the TAC filename */
    strcpy(tacfile, argv[ai]);
    /* Read TAC file */
    if(verbose>1) printf("reading %s\n", tacfile);
    else if(verbose==1) printf("  %s\n", tacfile);
    ret=tacRead(&tac, tacfile, &status);
    if(ret!=TPCERROR_OK) {
      fprintf(stderr, "Error (%d): %s\n", ret, errorMsg(status.error));
      tacFree(&tac); return(2);
    }
    if(verbose>1) {
      printf("fileformat := %s\n", tacFormattxt(tac.format));
      printf("tacNr := %d\n", tac.tacNr);
      printf("sampleNr := %d\n", tac.sampleNr);
    }
    /* Sort */
    if(verbose>1) printf("sorting %s\n", tacfile);
    if(sortBy==1)
      ret=tacSortByName(&tac, &status);
    else if(sortBy==4)
      ret=tacSortByTime(&tac, &status);
    else if(sortBy==5)
      ret=tacSortByAUC(&tac, &status);
    if(ret) {
      fprintf(stderr, "Error (%d): %s\n", ret, errorMsg(status.error));
      tacFree(&tac); return(3);
    }
    /* Save sorted data */
    if(verbose>1) printf("writing %s\n", tacfile);
    FILE *fp; fp=fopen(tacfile, "w");
    if(fp==NULL) {
      fprintf(stderr, "Error: cannot open file for writing (%s)\n", tacfile);
      tacFree(&tac); return(5);
    }
    ret=tacWrite(&tac, fp, TAC_FORMAT_UNKNOWN, 1, &status);
    fclose(fp);
    if(ret!=TPCERROR_OK) {
      fprintf(stderr, "Error (%d): %s\n", ret, errorMsg(status.error));
      tacFree(&tac); return(6);
    }
    tacFree(&tac);
    n++;
  } // next file
  if(n==0) {
    fprintf(stderr, "Error: no TAC(s) were sorted.\n");
    tacFree(&tac); return(2);
  }  
       
  tacFree(&tac);
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/// @endcond
