#!/bin/bash
#: Title      : test_tacformat
#: Date       : 2023-02-13
#: Author     : "Vesa Oikonen" <vesa.oikonen@utu.fi>
#: Options    : None

# Set the name of executable to test
if [ -n "${OS+1}" ] && [ "$OS" = "Windows_NT" ]; then
EXT=.exe;
else
EXT=;
fi
PROGRAM=../../tacformat$EXT;
if [ ! -f $PROGRAM ]; then
  printf "Failed: executable does not exist.\n"
  exit 1
fi


printf "=====================================================================\n"
printf "creating test data for  %s\n" $PROGRAM
printf "=====================================================================\n"

printf "\n verifying that required data exists \n\n"

if [ ! -f tacsort1.dft ] || [ ! -f test1a.bld ]; then
  printf "Failed: required data does not exist.\n"
  exit 1
fi
if [ ! -f hrrt1.hc ] || [ ! -f carimas1.txt ]; then
  printf "Failed: required data does not exist.\n"
  exit 1
fi
if [ ! -f test3.csv ] || [ ! -f test3.csv ]; then
  printf "Failed: required data does not exist.\n"
  exit 1
fi
if [ ! -f weights.dft ] ; then
  printf "Failed: required data does not exist.\n"
  exit 1
fi
if [ ! -f qview.csv ] || [ ! -f correct_qview.tac ] || [ ! -f qview.sif ]; then
  printf "Failed: required data does not exist.\n"
  exit 1
fi
if [ ! -f us1721.mat ] || [ ! -f correct_us1721.tac ]; then
  printf "Failed: required data does not exist.\n"
  exit 1
fi
if [ ! -f 4dm.csv ] || [ ! -f correct_4dm.tac ]; then
  printf "Failed: required data does not exist.\n"
  exit 1
fi


printf "\n making test files \n\n"

if [ ! -f simple1.dat ]; then
  printf "1 2 3\n4 5 6\n7 8 9\n10 11 12\n" > simple1.dat
fi
if [ ! -f test1.csv ]; then
  printf "1,2,3\n4,5,6\n7,8,9\n10,11,12\n" > test1.csv
fi
if [ ! -f test2.csv ]; then
  printf "1,2,3\n4,5,6\n7,8,9\n10,11,12\n" > test2.csv
  printf "# timeunit := min\n" >> test2.csv
  printf "# unit := Bq/cc\n" >> test2.csv
  printf "# Hand-made for testing\n" >> test2.csv
fi



printf "\n\n"
printf "=====================================================================\n"
printf "testing %s\n" $PROGRAM
printf "=====================================================================\n"

printf "\n 0.1.0 \n"
printf " Test case: Nonexisting argument file. \n"
printf " Expected result: Error. \n\n"

$PROGRAM -d nonexistingfile.tac
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "passed.\n\n"


printf "\n 0.1.1 \n"
printf " Test case: Binary file as argument file. \n"
printf " Expected result: Must not crash. \n\n"

$PROGRAM -d $PROGRAM
printf "passed.\n\n"


printf "=====================================================================\n"
printf "  Identify certain file formats\n"
printf "=====================================================================\n"

printf "\n 1.0.0 \n"
printf " Test case: DFT file. \n"
printf " Expected result: Format identified. \n\n"

f=$( $PROGRAM tacsort1.dft )
printf "Format: $f \n"
if [ "$f" != "DFT" ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "passed.\n\n"


printf "\n 1.1.0 \n"
printf " Test case: PMOD BLD file. \n"
printf " Expected result: Format identified. \n\n"

f=$( $PROGRAM test1a.bld )
printf "Format: $f \n"
if [ "$f" != "PMOD" ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "passed.\n\n"


printf "\n 1.2.0 \n"
printf " Test case: Simple TAC file. \n"
printf " Expected result: Format identified. \n\n"

f=$( $PROGRAM simple1.dat )
printf "Format: $f \n"
if [ "$f" != "Simple" ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "passed.\n\n"


printf "\n 1.3.0 \n"
printf " Test case: Carimas txt TAC file. \n"
printf " Expected result: Format identified. \n\n"

f=$( $PROGRAM carimas1.txt )
printf "Format: $f \n"
if [ "$f" != "Carimas-txt" ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "passed.\n\n"


printf "\n 1.4.0 \n"
printf " Test case: HRRT head curve file. \n"
printf " Expected result: Format identified. \n\n"

f=$( $PROGRAM hrrt1.hc )
printf "Format: $f \n"
if [ "$f" != "HRRT-HC" ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "passed.\n\n"


printf "\n 1.5.0 \n"
printf " Test case: Simple CSV file without comments. \n"
printf " Expected result: Format identified. \n\n"

f=$( $PROGRAM test1.csv )
printf "Format: $f \n"
if [ "$f" != "Simple" ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "passed.\n\n"


printf "\n 1.5.1 \n"
printf " Test case: Simple CSV file with comments. \n"
printf " Expected result: Format identified. \n\n"

f=$( $PROGRAM test2.csv )
printf "Format: $f \n"
if [ "$f" != "Simple" ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "passed.\n\n"


printf "\n 1.5.2 \n"
printf " Test case: CSV file in PMOD-like format with comments. \n"
printf " Expected result: Format identified. \n\n"

f=$( $PROGRAM test3.csv )
printf "Format: $f \n"
if [ "$f" != "PMOD" ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "passed.\n\n"


printf "\n 1.6.0 \n"
printf " Test case: QView TAC file. \n"
printf " Expected result: Format identified. \n\n"

f=$( $PROGRAM qview.csv )
printf "Format: $f \n"
if [ "$f" != "QView" ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "passed.\n\n"


printf "\n 1.7.0 \n"
printf " Test case: 4DM TAC file. \n"
printf " Expected result: Format identified. \n\n"

f=$( $PROGRAM 4dm.csv )
printf "Format: $f \n"
if [ "$f" != "4DM" ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "passed.\n\n"


printf "\n 1.8.0 \n"
printf " Test case: Mat TAC file. \n"
printf " Expected result: Format identified. \n\n"

f=$( $PROGRAM us1721.mat )
printf "Format: $f \n"
if [ "$f" != "Mat" ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "passed.\n\n"


printf "=====================================================================\n"
printf "  Basic format conversions \n"
printf "=====================================================================\n"

printf "\n 2.0.0 \n"
printf " Test case: DFT to PMOD and back. \n"
printf " Expected result: Correct data files. \n\n"

rm -f output.tac output.dft
$PROGRAM -f=pmod tacsort1.dft output.tac
 if [ $? -ne 0 ] ; then printf "\nFailed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.0001 -rel=0.01 -tacnames=y -yunit=y -xunit=y tacsort1.dft output.tac
if [ $? -ne 0 ] ; then printf "\nFailed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"
$PROGRAM -f=DFT output.tac output.dft
 if [ $? -ne 0 ] ; then printf "\nFailed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.0001 -rel=0.01 -tacnames=y -yunit=y -xunit=y tacsort1.dft output.dft
if [ $? -ne 0 ] ; then printf "\nFailed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 2.0.1 \n"
printf " Test case: Simple TAC without headers to DFT and back. \n"
printf " Expected result: Correct data files. \n\n"

rm -f output.dat output.dft
$PROGRAM -f=DFT simple1.dat output.dft
if [ $? -ne 0 ] ; then printf "\nFailed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.0001 -rel=0.01 -tacnames=n -yunit=n -xunit=n simple1.dat output.dft
if [ $? -ne 0 ] ; then printf "\nFailed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"
$PROGRAM -f=Simple output.dft output.dat
 if [ $? -ne 0 ] ; then printf "\nFailed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.0001 -rel=0.01 -tacnames=n -yunit=n -xunit=n simple1.dat output.dat
if [ $? -ne 0 ] ; then printf "\nFailed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 2.1.0 \n"
printf " Test case: PMOD CSV with headers to DFT and back. \n"
printf " Expected result: Correct data files. \n\n"

rm -f output.dft output.csv
$PROGRAM -f=dft -hdr=y test3.csv output.dft
if [ $? -ne 0 ] ; then printf "\nFailed!\n" ; exit 1 ; fi
printf "\n ok \n"
printf "checking that extra headers are copied\n"
iftmatch test3.csv output.dft Isotope 
if [ $? -ne 0 ] ; then printf "\nFailed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftmatch test3.csv output.dft Tracer 
if [ $? -ne 0 ] ; then printf "\nFailed!\n" ; exit 1 ; fi
printf "\n ok \n"
printf "checking that data are copied\n"
tacmatch -abs=0.0001 -rel=0.01 -tacnames=n -yunit=n -xunit=y test3.csv output.dft
if [ $? -ne 0 ] ; then printf "\nFailed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

$PROGRAM -f=PMOD -hdr=y output.dft output.csv
 if [ $? -ne 0 ] ; then printf "\nFailed!\n" ; exit 1 ; fi
printf "\n ok \n"
printf "checking that extra headers are copied\n"
iftmatch test3.csv output.csv Isotope 
if [ $? -ne 0 ] ; then printf "\nFailed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftmatch test3.csv output.csv Tracer 
if [ $? -ne 0 ] ; then printf "\nFailed!\n" ; exit 1 ; fi
printf "\n ok \n"
printf "checking that data are copied\n"
tacmatch -abs=0.0001 -rel=0.01 -tacnames=n -yunit=n -xunit=y test3.csv output.csv
if [ $? -ne 0 ] ; then printf "\nFailed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n 2.1.1 \n"
printf " Test case: DFT with extra headers to PMOD and back. \n"
printf " Expected result: Correct data files. \n\n"

rm -f output.tac output.dft
$PROGRAM -f=PMOD -hdr=y tacsort1.dft output.tac
if [ $? -ne 0 ] ; then printf "\nFailed!\n" ; exit 1 ; fi
printf "\n ok \n"
printf "checking that extra headers are copied\n"
iftmatch tacsort1.dft output.tac isotope 
if [ $? -ne 0 ] ; then printf "\nFailed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftmatch tacsort1.dft output.tac patient_id 
if [ $? -ne 0 ] ; then printf "\nFailed!\n" ; exit 1 ; fi
printf "\n ok \n"
printf "checking that data are copied\n"
tacmatch -abs=0.0001 -rel=0.01 -tacnames=y -yunit=y -xunit=y tacsort1.dft output.tac
if [ $? -ne 0 ] ; then printf "\nFailed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

$PROGRAM -f=DFT -hdr=y output.tac output.dft
 if [ $? -ne 0 ] ; then printf "\nFailed!\n" ; exit 1 ; fi
printf "\n ok \n"
printf "checking that extra headers are copied\n"
iftmatch tacsort1.dft output.dft isotope 
if [ $? -ne 0 ] ; then printf "\nFailed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftmatch tacsort1.dft output.dft patient_id 
if [ $? -ne 0 ] ; then printf "\nFailed!\n" ; exit 1 ; fi
printf "\n ok \n"
printf "checking that data are copied\n"
tacmatch -abs=0.0001 -rel=0.01 -tacnames=y -yunit=y -xunit=y tacsort1.dft output.dft
if [ $? -ne 0 ] ; then printf "\nFailed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 2.2.1 \n"
printf " Test case: DFT with weights to PMOD and back. \n"
printf " Expected result: Correct data files. \n\n"

rm -f output.tac output.dft
$PROGRAM -f=PMOD -hdr=y weights.dft output.tac
if [ $? -ne 0 ] ; then printf "\nFailed!\n" ; exit 1 ; fi
printf "\n ok \n"
printf "checking that weights and data are copied\n"
tacmatch -abs=0.001 -rel=0.1 -w=y -tacnames=y -yunit=y -xunit=y weights.dft output.tac
if [ $? -ne 0 ] ; then printf "\nFailed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

$PROGRAM -f=DFT -hdr=y output.tac output.dft
if [ $? -ne 0 ] ; then printf "\nFailed!\n" ; exit 1 ; fi
printf "\n ok \n"
printf "checking that weights and data are copied\n"
tacmatch -abs=0.001 -rel=0.1 -w=y -tacnames=y -yunit=y -xunit=y weights.dft output.dft
if [ $? -ne 0 ] ; then printf "\nFailed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n 2.3.0 \n"
printf " Test case: QView to PMOD. \n"
printf " Expected result: Correct data files. \n\n"

rm -f output.tac output.dft
$PROGRAM -f=PMOD -hdr=n qview.csv output.tac
if [ $? -ne 0 ] ; then printf "\nFailed!\n" ; exit 1 ; fi
printf "\n ok \n"

printf " remove other than group TACs. \n"
tacdel output.tac 1 2 3 4 5 6 7 8 9 10
if [ $? -ne 0 ] ; then printf "\nFailed!\n" ; exit 1 ; fi
printf "\n ok \n"

printf "checking that data is correct\n"
tacmatch -abs=0.001 -rel=0.1 -tacnames=y -yunit=y -xunit=y correct_qview.tac output.tac
if [ $? -ne 0 ] ; then printf "\nFailed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n 2.3.1 \n"
printf " Test case: Add frame times from SIF to previously made file. \n"
printf " Expected result: No error. \n\n"

tacframe output.tac qview.sif
if [ $? -ne 0 ] ; then printf "\nFailed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n 2.4.0 \n"
printf " Test case: Mat to PMOD. \n"
printf " Expected result: Correct data files. \n\n"

rm -f output.tac
$PROGRAM -f=PMOD us1721.mat output.tac
if [ $? -ne 0 ] ; then printf "\nFailed!\n" ; exit 1 ; fi
printf "\n ok \n"

printf "checking that data is correct\n"
tacmatch -abs=0.001 -rel=0.1 -tacnames=y -yunit=n -xunit=n correct_us1721.tac output.tac
if [ $? -ne 0 ] ; then printf "\nFailed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"




printf "\n 2.5.0 \n"
printf " Test case: 4DM to PMOD. \n"
printf " Expected result: Correct data files. \n\n"

rm -f output.tac
$PROGRAM -f=PMOD 4dm.csv output.tac
if [ $? -ne 0 ] ; then printf "\nFailed!\n" ; exit 1 ; fi
printf "\n ok \n"

printf "checking that data is correct\n"
tacmatch -abs=0.001 -rel=0.1 -tacnames=y -yunit=y -xunit=y correct_4dm.tac output.tac
if [ $? -ne 0 ] ; then printf "\nFailed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n 3.0.0 \n"
printf " Test case: DFT with weights to XML.\n"
printf " Expected result: file can be created. \n"

rm -f output.xml
$PROGRAM -f=XML -hdr=n weights.dft output.xml
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
if [ ! -f output.xml ]; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"




printf "=====================================================================\n"
printf "  All passed!\n"
printf "=====================================================================\n"
exit 0
