/** @file ecat2ana.c
 *  @brief Convert ECAT 6.3 or 7 image format to Analyze 7.5.
 *  @copyright (c) Turku PET Centre
 *  @author Vesa Oikonen
 */
/// @cond
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <string.h>
#include <time.h>
#include <sys/stat.h>
#include <unistd.h>
/*****************************************************************************/
#include "libtpcmisc.h"
#include "libtpcimgio.h"
#include "libtpcimgp.h"
/*****************************************************************************/

/*****************************************************************************/
static char *info[] = {
  "Converts ECAT 6.3 images or ECAT 7 image volumes to Analyze 7.5 image format.",
  " ",
  "The resulting Analyze database consists of two files, Analyze image (*.img)",
  "and header file (*.hdr).",
  "Analyze image format does not contain information on the frame times.",
  "Frame times can be retrieved from SIF file, which can be created optionally.",
  "SIF can also be created using other software.",
  " ",
  "By default, data is saved in big endian byte order (Sun Sparc).",
  " ",
  "Usage: @P [Options] ecatfile(s)",
  " ",
  "Options:",
  " -Little or -pc or -intel",
  "     Data is saved in little endian (PC Intel) byte order.",
  " -Big or -sun or -sparc",
  "     Data is saved in big endian (Sun Sparc, Motorola, PowerPC) byte order",
  "     (default).",
  " -O=<output path>",
  "     Data directory for Analyze files; by default the output directory",
  "     ana_unix_files or ana_pc_files is created under input directory.",
  " -flip=<y|n>",
  "     Override the default and environment variable ANALYZE_FLIP setting",
  "     by always flipping/not flipping image in z-direction (planes).",
  "     If environment variable is not set, then default is y.",
  "     Images are always flipped in x,y-directions.",
  " -sif",
  "     SIF is saved with Analyze files; note that existing SIF will be",
  "     overwritten.",
  " -frames",
  "     Time frames are saved as separate Analyze files.",
  " -stdoptions", // List standard options like --help, -v, etc
  " ",
  "Example:",
  "  @P -o=anal *.v",
  " ",
  "Specific extensions to Analyze 7.5 format:",
  " -Scale factor to retain quantitation in image_dimension.funused1",
  " -Isotope halflife (sec) in image_dimension.funused3;",
  "  this does not imply whether data is corrected for decay or not.",
  " -String in data_history.descrip tells whether data is corrected for decay;",
  "  'Decay corrected.' or 'No decay correction.'",
  " ",
  "See also: ana2ecat, anabyteo, ana_lhdr, eframe, img2flat, ecat2nii",
  " ",
  "Keywords: image, format conversion, ECAT, Analyze",
  0};
/*****************************************************************************/

/*****************************************************************************/
/* Turn on the globbing of the command line, since it is disabled by default in
   mingw-w64 (_dowildcard=0); in MinGW32 define _CRT_glob instead, if necessary;
   In Unix&Linux wildcard command line processing is enabled by default. */
/*
#undef _CRT_glob
#define _CRT_glob -1
*/
int _dowildcard = -1;
/*****************************************************************************/

/*****************************************************************************/
/**
 *  Main
 */
int main(int argc, char **argv)
{
  int      ai, help=0, version=0, verbose=1;
  int      ret, errorNr=0, flipping, fi=0, ffi=0, fileNr=0;
  int      ana_order=1; /* 0=little (intel), 1=big endian (sparc) */
  int      separate_frames=0;
  int      file_format=0;
  int      save_sif=0; // SIF is made (1) or not made (0)
  char     ecatfile[FILENAME_MAX], dbname[FILENAME_MAX], *cptr;
  char     temp[FILENAME_MAX], outputdir[FILENAME_MAX], dbdir[FILENAME_MAX];
  IMG      img;
  float    fmin, fmax;
  SIF      sif;


  /*
   *  Get arguments
   */
  if(argc==1) {tpcPrintUsage(argv[0], info, stderr); return(1);}
  outputdir[0]=ecatfile[0]=dbname[0]=dbdir[0]=(char)0;
  flipping=anaFlipping();
  imgInit(&img); sifInit(&sif);
  /* Options */
  for(ai=1; ai<argc; ai++) if(*argv[ai]=='-') { /* options */
    cptr=argv[ai]+1; if(*cptr=='-') cptr++; if(cptr==NULL) continue;
    if(tpcProcessStdOptions(argv[ai], &help, &version, &verbose)==0) continue;
    if(strncasecmp(cptr, "FLIP=", 5)==0) {
      cptr+=5; if(*cptr=='n' || *cptr=='N') {flipping=0; continue;}
      else if(*cptr=='y' || *cptr=='Y') {flipping=1; continue;}
    } else if(strncasecmp(cptr, "FRAMES", 2)==0) {
      separate_frames=1; continue;
    } else if(strcasecmp(cptr, "LITTLE")==0) {
      ana_order=0; continue;
    } else if(strcasecmp(cptr, "PC")==0) {
      ana_order=0; continue;
    } else if(strcasecmp(cptr, "INTEL")==0) {
      ana_order=0; continue;
    } else if(strcasecmp(cptr, "BIG")==0) {
      ana_order=1; continue;
    } else if(strcasecmp(cptr, "SUN")==0) {
      ana_order=1; continue;
    } else if(strcasecmp(cptr, "SPARC")==0) {
      ana_order=1; continue;
    } else if(strncasecmp(cptr, "O=", 2)==0) {
      cptr+=2; 
      if(strlen(cptr)>0) strcpy(outputdir, cptr); else strcpy(outputdir, ".");
      continue;
    } else if(strcasecmp(cptr, "SIF")==0) {
      save_sif=1; continue;
    }
    fprintf(stderr, "Error: invalid option '%s'\n", argv[ai]);
    return(1);
  } else break;
  
  /* Print help or version? */
  if(help==2) {tpcHtmlUsage(argv[0], info, ""); return(0);}
  if(help) {tpcPrintUsage(argv[0], info, stdout); return(0);}
  if(version) {tpcPrintBuild(argv[0], stdout); return(0);}

  /* Process other arguments, starting from the first non-option */
  for(; ai<argc; ai++) {
    /* Other arguments are ECAT files which must exist */
    if(access(argv[ai], 0) == -1) {
      fprintf(stderr, "Error: file %s does not exist\n", argv[ai]);
      return(1);
    }
    if(ffi<1) ffi=ai;
    fileNr++;
  }

  /* Is something missing? */
  if(fileNr<1) {
    fprintf(stderr, "Error: no ECAT files were specified.\n");
    return(1);
  }


  /* In verbose mode print arguments and options */
  if(verbose>1) {
    printf("flipping := %d\n", flipping);
    printf("ana_order := %d\n", ana_order);
    printf("separate_frames := %d\n", separate_frames);
    printf("save_sif := %d\n", save_sif);
    printf("outputdir := %s\n", outputdir);
    printf("fileNr := %d\n", fileNr);
  }
  if(verbose>2) IMG_TEST=ANALYZE_TEST=verbose-2; else IMG_TEST=ANALYZE_TEST=0;
  if(verbose>0) {
    if(flipping==0) printf("image(s) will not be flipped in z-direction.\n");
    else printf("image(s) will be flipped in z-direction.\n");
  }


  /*
   *  Process each ECAT file separately
   */
  if(verbose>1) printf("processing...\n");
  fileNr=0;
  for(ai=ffi; ai<argc; ai++) {

    strcpy(ecatfile, argv[ai]);
    if(verbose>0) {fprintf(stdout, "%s : \n", ecatfile); fflush(stdout);}
    imgEmpty(&img);
    sifEmpty(&sif);

    /*
     *  Make Analyze database name
     */
    /*  Make output directory name, if it was not given */
    if(strlen(outputdir)<1) {
      /* Get input directory */
      strcpy(dbdir, ecatfile);
      cptr=strrchr(dbdir, '/'); if(cptr==NULL) cptr=strrchr(dbdir, '\\');
      if(cptr!=NULL) {cptr++; cptr[0]=(char)0;} else dbdir[0]=(char)0;
      /* Add default part */
      if(ana_order==1) strcat(dbdir, "ana_unix_files");
      else strcat(dbdir, "ana_pc_files");
    } else {
      strcpy(dbdir, outputdir);
      /* Remove trailing slashes */
      ret=strlen(dbdir)-1;
      if(dbdir[ret]=='/' || dbdir[ret]=='\\') dbdir[ret]=(char)0;
    }
    /* Create the subdirectory */
    if(access(dbdir, 0) == -1) {
      if(verbose>0) {
        fprintf(stdout, "  Creating subdirectory %s\n", dbdir);
        fflush(stdout);
      }
#ifdef WIN32
      ret=mkdir(dbdir);
#else
      ret=mkdir(dbdir, 00775);
#endif
      if(ret!=0) {
        fprintf(stderr, "  Error: cannot create subdirectory.\n");
        fflush(stderr); errorNr++; continue;
      }
    }
    /* Combine path and filename */
    cptr=strrchr(ecatfile, '/'); if(cptr==NULL) cptr=strrchr(ecatfile, '\\');
    if(cptr==NULL) cptr=ecatfile; else cptr++;
#ifdef WIN32
    snprintf(dbname, FILENAME_MAX, "%s\\%s", dbdir, cptr);
#else
    snprintf(dbname, FILENAME_MAX, "%s/%s", dbdir, cptr);
#endif
    cptr=strrchr(dbname, '.'); if(cptr!=NULL) cptr[0]=(char)0;
    if(verbose>1) printf("  Analyze db_name: '%s'\n", dbname);
    /* Check if file exists: Analyze database can be overwritten, but existing
       ECAT 6.3 image (.img without .hdr) will lead to an error */
    if(separate_frames!=1 && anaExists(dbname)==0) {
      strcpy(temp, dbname); strcat(temp, ".img");
      if(access(temp, 0) != -1) {
        fprintf(stderr, "  Error: %s would be overwritten.\n", temp);
        fprintf(stdout, "  No conversion is done for %s\n", ecatfile);
        fflush(stdout); fflush(stderr);
        errorNr++; continue;
      }
    }
    
    /*
     *  Remove possibly existing database
     */
    ret=anaRemove(dbname); if(ret) return STATUS_CANNOTERASE;
         
    /*
     *  Get the global min and max pixel values;
     *  those are needed for Analyze header, and for scaling pixels
     */
    if(verbose>1) fprintf(stdout, "  searching min and max in %s\n", ecatfile);
    ret=imgReadMinMax(ecatfile, &fmin, &fmax);
    if(ret) {
      fprintf(stderr, "Error: %s\n", imgStatus(ret)); 
      if(verbose>3) imgInfo(&img);
      errorNr++; fflush(stderr); continue;
    }
    if(verbose>2) 
      printf("    global_min := %g\n    global_max := %g\n", fmin, fmax);

    /*
     *  Get the frame number and allocate memory for SIF, if requested
     */
    if(save_sif) {
      /* Read information from input image for SIF */
      if(verbose>1) printf("  reading header information\n");
      ret=imgReadHeader(ecatfile, &img, img._fileFormat);
      if(ret) {
        fprintf(stderr, "Error: %s\n", imgStatus(ret));
        errorNr++; fflush(stderr); continue;
      }
      /* Allocate SIF to store frame times */
      ret=sifSetmem(&sif, img.dimt); if(ret!=0) {
        fprintf(stderr, "Error: out of memory.\n");
        errorNr++; fflush(stderr); continue;
      }
      sif.colNr=4; sif.version=1;
      imgEmpty(&img);
    }

    /*
     *  Conversion of ECAT file, one frame at a time
     */
    if(verbose>0) fprintf(stdout, "  processing %s\n", ecatfile);
    fi=0; imgEmpty(&img);
    while((ret=imgReadFrame(ecatfile, fi+1, &img, 0)) == 0) {
      if(verbose>1) printf("    frame %d\n", fi+1);
      /* reverse the flipping/no-flipping in imgWriteAnalyzeFrame, if required */
      if(flipping!=anaFlipping()) imgFlipPlanes(&img);
      /* Set the write file format including byte order */
      if(fi==0) file_format=img._fileFormat; // must be saved to enable reading 
                                             // the next frame
      if(ana_order) img._fileFormat=IMG_ANA; else img._fileFormat=IMG_ANA_L;
      /* Write the frame in Analyze format */
      if(separate_frames!=1) { // into one Analyze file
        ret=imgWriteAnalyzeFrame(dbname, fi+1, &img, 0, fmin, fmax);
        if(ret) anaRemove(dbname);  //printf("ret := %d\n", ret);
      } else { // each frame in a separate Analyze file
        snprintf(temp, FILENAME_MAX, "%s_fr%03d", dbname, fi+1); // make database name for frame
        ret=imgMinMax(&img, &fmin, &fmax);
        if(ret==0) ret=imgWriteAnalyzeFrame(temp, 1, &img, 0, fmin, fmax);
        if(ret) anaRemove(temp);
      }
      if(ret!=STATUS_OK) break;
      if(verbose>3) printf("    frame written.\n");
      /* Set SIF contents for this frame, if requested */
      if(save_sif) {
        sif.x1[fi]=img.start[0]; sif.x2[fi]=img.end[0];
        if(fi==0) {
          /* Set isotope */
          strcpy(sif.isotope_name, imgIsotope(&img) );
          /* Set studynumber */
          strcpy(sif.studynr, img.studyNr);
          /* Set scan start time */
          sif.scantime=img.scanStart;
        }
      }
      /* Prepare to the next frame */
      img._fileFormat=file_format;
      fi++;
    } // next frame
    if(verbose>0) {
      if(ret==STATUS_NOMATRIX)
        fprintf(stdout, "  %d frame(s) processed.\n", fi);
    }
    if(ret!=STATUS_OK && ret!=STATUS_NOMATRIX) {
      fprintf(stderr, "Error: %s\n", imgStatus(ret)); //if(TEST) imgInfo(&img);
      errorNr++; fflush(stderr); continue;
    }
    imgEmpty(&img);

    /* Save SIF, if requested */
    if(save_sif) {
      char temp[FILENAME_MAX+4];
      sprintf(temp, "%s.sif", dbname);
      ret=sifWrite(&sif, temp);
      if(ret!=0) {
        fprintf(stderr, "Error: cannot write %s\n", temp);
        errorNr++; fflush(stderr); continue;
      }
      sifEmpty(&sif);
    }

  } /* next file */

  if(errorNr>0) return(errorNr+10);
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/// @endcond
