#!/bin/bash
#: Title      : test_ana2ecat
#: Date       : 2023-06-30
#: Author     : "Vesa Oikonen" <vesa.oikonen@utu.fi>
#: Options    : None

# Set the name of executable to test
if [ -n "${OS+1}" ] && [ "$OS" = "Windows_NT" ]; then
EXT=.exe;
else
EXT=;
fi
PROGRAM=../../ana2ecat$EXT;
if [ ! -f $PROGRAM ]; then
  printf "Failed: executable does not exist.\n"
  exit 1
fi


# Make sure that environment variable for flipping is not set
ANALYZE_FLIP=;
export ANALYZE_FLIP
printf "\n ANALYZE_FLIP = %s\n\n", $ANALYZE_FLIP


printf "=====================================================================\n"
printf "creating test data for %s\n" $PROGRAM
printf "=====================================================================\n"


printf "\n verifying that required data exists \n"

if [ ! -f image1.dat ] || [ ! -f image1.inf ] || [ ! -f image1.fra ] ; then
  printf "Failed: required file does not exist.\n"
  exit 1
fi
if [ ! -f image2.dat ] || [ ! -f image2.inf ] || [ ! -f image2.fra ] ; then
  printf "Failed: required file does not exist.\n"
  exit 1
fi

if [ ! -f ana_prev/ecat6.hdr ] || [ ! -f ana_prev/ecat6.img ] || [ ! -f ana_prev/ecat6.sif ] ; then
  printf "Failed: required file does not exist.\n"
  exit 1
fi
if [ ! -f ana_prev/ecat7.hdr ] || [ ! -f ana_prev/ecat7.img ] || [ ! -f ana_prev/ecat7.sif ] ; then
  printf "Failed: required file does not exist.\n"
  exit 1
fi
if [ ! -f ecat_prev/ecat6.img ] || [ ! -f ecat_prev/ecat7.v ] ; then
  printf "Failed: required file does not exist.\n"
  exit 1
fi

mkdir orig

if [ ! -f image1.v ] || [ ! -f image1.img ]; then
  printf "\n making image1.v and image1.img\n"
  asc2flat image1.dat image1.bin
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  flat2img -scanner=Adv -inf=image1.inf image1.bin image1.v
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  eframe image1.v image1.fra
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  imgdecay -i=C-11 image1.v
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  esetstrt image1.v 2007-03-20 15:22:02
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  img2tif -rb image1.v image1.tif
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  e7to63 image1.v image1.img
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  printf "\n done. \n"
fi
if [ ! -f orig/image1.v ] || [ ! -f orig/image1.img ]; then
  cp -f image1.v orig/
  cp -f image1.img orig/
fi


if [ ! -f image2.v ] || [ ! -f image2.img ]; then
  printf "\n making image2.v and image2.img\n"
  rm -f image2.i
  asc2flat image2.dat image2.bin
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  flat2img -scanner=HR+ -inf=image2.inf image2.bin image2.v
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  eframe image2.v image2.fra
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  imgdecay -i=F-18 image2.v
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  esetstrt image2.v 2006-06-20 12:01:59
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  img2tif -rb image2.v image2.tif
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  e7to63 image2.v image2.img
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  printf "\n done. \n"
fi
if [ ! -f orig/image2.v ] || [ ! -f orig/image2.img ]; then
  cp -f image2.v orig/
  cp -f image2.img orig/
fi

printf "\n making ana_unix_files\n"
rm -f ana_unix_files/*
rmdir ana_unix_files
ecat2ana -sif -big -flip=y -o=ana_unix_files image1.v image2.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n done. \n"

printf "\n making ana_pc_files\n"
rm -f ana_pc_files/*
rmdir ana_pc_files
ecat2ana -sif -little -flip=y -o=ana_pc_files image1.v image2.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n done. \n"


printf "\n\n"
printf "=====================================================================\n"
printf "testing %s\n" $PROGRAM
printf "=====================================================================\n"

printf "\n Testing compatibility with previous a-lot-used software version \n\n"

printf "\n 1.0.0 \n"
printf " ECAT 6 \n\n"

$PROGRAM -o=temp -6 ana_prev/ecat6
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
imgmatch -abs=0.1 -rel=0.5 temp/ecat6.img ecat_prev/ecat6.img
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n Passed. \n\n"

printf "\n 1.0.1 \n"
printf " ECAT 7 \n\n"

rm -f temp/*
rmdir temp

$PROGRAM -o=temp -6 ana_prev/ecat6
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
imgmatch -abs=0.1 -rel=0.5 temp/ecat6.img ecat_prev/ecat6.img
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n Passed. \n\n"

rm -f temp/*
rmdir temp

$PROGRAM -o=temp -7 ana_prev/ecat7
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
imgmatch -abs=0.1 -rel=0.5 temp/ecat7.v ecat_prev/ecat7.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n Passed. \n\n"


printf "\n=================================================================\n\n"

printf "\n Analyze input \n"

printf "\n 2.0.0 \n"
printf "Database not given \n\n"

rm -f temp/*
rmdir temp
$PROGRAM -d2 -o=temp 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error "Analyze database was not given."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n Passed. \n\n"


printf "\n 2.0.1 \n"
printf "non-existing input path \n\n"

$PROGRAM -d2 -o=temp nonexisting 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error "cannot open Analyze image or directory nonexisting"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n Passed. \n\n"


printf "\n 2.0.2 \n"
printf "input path exists but contains no Analyze images \n\n"

rm -f temp/*
rmdir temp
mkdir temp
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi

$PROGRAM -7 -o=temp2 temp 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error "No Analyze files were found in temp"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n Passed. \n\n"


printf "\n 2.0.3 \n"
printf "input path exists but specified database does not \n\n"

rm -f temp/*
rmdir temp

$PROGRAM -7 -o=temp ana_unix_files/nonexisting 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error "cannot open Analyze image or directory ana_unix_files/nonexisting"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n Passed. \n\n"


printf "\n 2.1.0 \n"
printf "More than one database can be processed \n\n"

rm -f temp/*
rmdir temp

$PROGRAM -7 -o=temp ana_unix_files
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
if [ -f temp/image1.v ] ; then rm temp/image1.v; else printf "Failed!\n"; exit 1 ; fi
printf "\n ok \n"
if [ -f temp/image2.v ] ; then rm temp/image2.v; else printf "Failed!\n"; exit 1 ; fi
printf "\n Passed. \n\n"


printf "\n 2.1.1 \n"
printf "One datase cannot be processed, but others are processed \n\n"

rm -f temp/*
rmdir temp
cp -f image1.img ana_unix_files/stupid.hdr
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
cp -f image1.img ana_unix_files/stupid.img
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi

$PROGRAM -7 -o=temp ana_unix_files 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then
rm -f ana_unix_files/stupid.hdr ana_unix_files/stupid.img
printf "Failed!\n"; exit 1;
fi
printf "\n ok \n"
rm -f ana_unix_files/stupid.hdr ana_unix_files/stupid.img

iftisval stdout.txt Error "unsupported file type"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

if [ -f temp/image1.v ] ; then rm temp/image1.v; else printf "Failed!\n"; exit 1 ; fi
printf "\n ok \n"
if [ -f temp/image2.v ] ; then rm temp/image2.v; else printf "Failed!\n"; exit 1 ; fi
printf "\n Passed. \n\n"




printf "\n=================================================================\n\n"

printf "\n Output options \n"

printf "\n 3.0.0 \n"
printf " Output path not specified \n\n"

rm -f image1.v image2.v

$PROGRAM -7 -d2 ana_unix_files 1>stdout.txt 2>&1
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

iftisval stdout.txt ecatdir "."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

if [ ! -f image1.v ] ; then printf "Failed!\n"; exit 1 ; fi
printf "\n ok \n"
if [ ! -f image2.v ] ; then printf "Failed!\n"; exit 1 ; fi
printf "\n Passed. \n\n"

imgmatch -abs=0.1 -rel=0.5 orig/image1.v image1.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
imgmatch -abs=0.1 -rel=0.5 orig/image2.v image2.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n Passed. \n\n"


printf "\n 3.1.0 \n"
printf " Output path given with option -o= \n\n"

rm temp/*
rmdir temp

$PROGRAM -7 -d2 -o=temp ana_unix_files 1>stdout.txt 2>&1
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

iftisval stdout.txt ecatdir "temp"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

if [ ! -f temp/image1.v ] ; then printf "Failed!\n"; exit 1 ; fi
printf "\n ok \n"
if [ ! -f temp/image2.v ] ; then printf "Failed!\n"; exit 1 ; fi
printf "\n Passed. \n\n"

imgmatch -abs=0.1 -rel=0.5 orig/image1.v temp/image1.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
imgmatch -abs=0.1 -rel=0.5 orig/image2.v temp/image2.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n Passed. \n\n"


printf "\n 3.1.1 \n"
printf " Output path set to current dir with option -o= \n\n"

rm -f image1.v image2.v

$PROGRAM -7 -d2 -o= ana_unix_files 1>stdout.txt 2>&1
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

iftisval stdout.txt ecatdir "."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

if [ ! -f image1.v ] ; then printf "Failed!\n"; exit 1 ; fi
printf "\n ok \n"
if [ ! -f image2.v ] ; then printf "Failed!\n"; exit 1 ; fi
printf "\n Passed. \n\n"

imgmatch -abs=0.1 -rel=0.5 orig/image1.v image1.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
imgmatch -abs=0.1 -rel=0.5 orig/image2.v image2.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n Passed. \n\n"


printf "\n 3.2.0 \n"
printf " Input Analyze .img file cannot be overwritten by mistake \n\n"

rm -f temp/*
rmdir temp
mkdir temp
cp -f ana_unix_files/* temp
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi

rm -f stdout.txt
$PROGRAM -6 -o=temp temp 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

iftisval stdout.txt Error "temp/image1.img would be overwritten."
if [ $? -ne 0 ] ; then 
  iftisval stdout.txt Error "temp\image1.img would be overwritten."
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi
printf "\n ok \n"
iftisval stdout.txt Error "temp/image2.img would be overwritten."
if [ $? -ne 0 ] ; then 
  iftisval stdout.txt Error "temp\image2.img would be overwritten."
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi
printf "\n ok \n"

$PROGRAM -6 -o=temp temp/image2 1>stdout.txt 2>&1

iftisval stdout.txt Error "temp/image2.img would be overwritten."
if [ $? -ne 0 ] ; then 
  iftisval stdout.txt Error "temp\image2.img would be overwritten."
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi
printf "\n Passed. \n\n"


printf "\n 3.2.1 \n"
printf " But existing ECAT 6 .img file can be overwritten \n\n"

rm -f temp/*
rmdir temp
mkdir temp

cp -f orig/image2.img temp/image1.img
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
$PROGRAM -6 -o=temp ana_unix_files
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
imgmatch -abs=0.1 -rel=0.5 orig/image1.img temp/image1.img
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

cp -f orig/image1.v temp/image2.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
$PROGRAM -7 -o=temp ana_unix_files/image2
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
imgmatch -abs=0.1 -rel=0.5 orig/image2.v temp/image2.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"




printf "\n=================================================================\n\n"

printf "\n Z flipping \n"

printf "\n 4.0.0 \n"
printf " no option or environment variable \n\n"

rm -f temp/*
rmdir temp

ANALYZE_FLIP=;
export ANALYZE_FLIP
printf "\n ANALYZE_FLIP = %s\n\n", $ANALYZE_FLIP
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi

$PROGRAM -d2 -o=temp ana_unix_files/image1 1>stdout.txt 2>&1
iftisval stdout.txt flipping "1"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n Passed. \n\n"


printf "\n 4.0.1 \n"
printf " environment variable is y \n\n"

rm -f temp/*
rmdir temp

ANALYZE_FLIP=y;
export ANALYZE_FLIP
printf "\n ANALYZE_FLIP = %s\n\n", $ANALYZE_FLIP
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi

$PROGRAM -d2 -o=temp ana_unix_files/image2 1>stdout.txt 2>&1
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt flipping "1"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n Passed. \n\n"


printf "\n 4.0.2 \n"
printf " environment variable is n \n\n"

rm -f temp/*
rmdir temp

ANALYZE_FLIP=n;
export ANALYZE_FLIP
printf "\n ANALYZE_FLIP = %s\n\n", $ANALYZE_FLIP
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi

$PROGRAM -d2 -o=temp ana_unix_files/image2 1>stdout.txt 2>&1
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt flipping "0"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n Passed. \n\n"


printf "\n 4.1.0 \n"
printf " option -flip=y overrides environment variable \n\n"

rm -f temp/*
rmdir temp

ANALYZE_FLIP=n;
export ANALYZE_FLIP
printf "\n ANALYZE_FLIP = %s\n\n", $ANALYZE_FLIP
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi

$PROGRAM -d2 -flip=y -o=temp ana_unix_files/image2 1>stdout.txt 2>&1
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt flipping "1"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n Passed. \n\n"


printf "\n 4.1.1 \n"
printf " option -flip=n overrides environment variable \n\n"

rm -f temp/*
rmdir temp

ANALYZE_FLIP=y;
export ANALYZE_FLIP
printf "\n ANALYZE_FLIP = %s\n\n", $ANALYZE_FLIP
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi

$PROGRAM -d2 -flip=n -o=temp ana_unix_files/image2 1>stdout.txt 2>&1
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt flipping "0"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n Passed. \n\n"


# Remove environment variable
ANALYZE_FLIP=;
export ANALYZE_FLIP
printf "\n ANALYZE_FLIP = %s\n\n", $ANALYZE_FLIP


printf "\n 4.2.0 \n"
printf " Verify that images really are flipped \n\n"

rm -f temp/*
rmdir temp

$PROGRAM -7 -o=temp -flip=y ana_unix_files
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
imgmatch -abs=0.1 -rel=0.5 orig/image1.v temp/image1.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
imgmatch -abs=0.1 -rel=0.5 orig/image2.v temp/image2.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n Passed. \n\n"

printf "\n 4.2.1 \n"
printf " Verify that images really are NOT flipped \n\n"

rm -f temp/*
rmdir temp

$PROGRAM -7 -o=temp -flip=n ana_unix_files
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
imgflip -z temp/image1.v temp/image1.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
imgflip -z temp/image2.v temp/image2.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
imgmatch -abs=0.1 -rel=0.5 orig/image1.v temp/image1.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
imgmatch -abs=0.1 -rel=0.5 orig/image2.v temp/image2.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n Passed. \n\n"




printf "\n=================================================================\n\n"

printf "\n Option for ECAT format \n"

printf "\n 5.0.0 \n"
printf " no option \n\n"

rm -f temp/*
rmdir temp

$PROGRAM -d2 -o=temp ana_unix_files/image1 1>stdout.txt 2>&1
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt outFormat "7"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
imgmatch -abs=0.1 -rel=0.5 orig/image1.v temp/image1.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n Passed. \n\n"

printf "\n 5.0.1 \n"
printf " option -7 \n\n"

rm -f temp/*
rmdir temp

$PROGRAM -d2 -7 -o=temp ana_unix_files/image1 1>stdout.txt 2>&1
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt outFormat "7"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
imgmatch -abs=0.1 -rel=0.5 orig/image1.v temp/image1.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n Passed. \n\n"

printf "\n 5.0.2 \n"
printf " option -6 \n\n"

rm -f temp/*
rmdir temp

$PROGRAM -d2 -6 -o=temp ana_unix_files/image1 1>stdout.txt 2>&1
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt outFormat "6"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
imgmatch -abs=0.1 -rel=0.5 orig/image1.img temp/image1.img
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n Passed. \n\n"


printf "\n=================================================================\n\n"

printf "\n Byte order \n"
printf " big endian data used several times before, now little endian \n\n"

printf "\n 6.0.0 \n"
printf " ECAT 6 \n\n"

rm -f temp/*
rmdir temp

$PROGRAM -6 -o=temp ana_pc_files/image1
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
imgmatch -abs=0.1 -rel=0.5 orig/image1.img temp/image1.img
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n Passed. \n\n"

printf "\n 6.0.1 \n"
printf " ECAT 7 \n\n"

rm -f temp/*
rmdir temp

$PROGRAM -7 -o=temp ana_pc_files/image2
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
imgmatch -abs=0.1 -rel=0.5 orig/image2.v temp/image2.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n Passed. \n\n"




printf "\n=================================================================\n\n"

printf "\n Misc tests \n"

printf "\n 7.0.0 \n"
printf " Fails if no arguments or options \n\n"

$PROGRAM 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n Passed. \n\n"


printf "\n 7.0.1 \n"
printf " Fails if extra arguments \n\n"

rm -f temp/*
rmdir temp

$PROGRAM -o=temp ana_unix_files extra 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error "invalid argument 'extra'"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n Passed. \n\n"


printf "\n 7.1.0 \n"
printf " Fails if unknown option \n\n"

rm -f temp/*
rmdir temp

$PROGRAM -o=temp -stupidoption ana_unix_files 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error "invalid option '-stupidoption'"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n Passed. \n\n"




printf "\n===================================================================\n"
printf "   All passed! \n"
printf "===================================================================\n"
exit 0

