#!/bin/bash
#: Title      : test_fitk2
#: Date       : 2023-06-29
#: Author     : "Vesa Oikonen" <vesa.oikonen@utu.fi>
#: Options    : None

# Set the name of executable to test
if [ -n "${OS+1}" ] && [ "$OS" = "Windows_NT" ]; then
EXT=.exe;
else
EXT=;
fi

PROGRAM=../../fitk2$EXT;
if [ ! -f $PROGRAM ]; then
  printf "Failed: executable does not exist.\n"
  exit 1
fi

printf "=====================================================================\n"
printf "creating test data for %s\n" $PROGRAM
printf "=====================================================================\n"

printf "\n verifying that required data exists \n"


if [ ! -f plasma.dat ] || [ ! -f frames.dat ]; then
  printf "Failed: required data does not exist.\n"
  exit 1
fi
if [ ! -f frames_few.dat ] || [ ! -f frames_long.dat ]; then
  printf "Failed: required data does not exist.\n"
  exit 1
fi

printf "\n making data for testing \n"

if [ ! -f plasma.bld ] || [ ! -f blood.bld ]; then
  rm -f plasma_na.bld blood_na.bld k2tis.dft k2tis1.dft k2tis1_na.dft
  rm -f plasma_3.bld blood_3.bld k2tis1_3.dft
  printf "\n making plasma.bld \n"
  tacformat -f=PMOD plasma.dat plasma.bld
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacunit -xset=min -yset=kBq/ml plasma.bld
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacren plasma.bld 0 plasma
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  printf "\n making blood.bld \n"
  taccalc plasma.bld x 0.5 blood.bld
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacren blood.bld 0 blood
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tac2svg -mt="Input" -legend=y input.svg -l plasma.bld blood.bld 
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi

if [ ! -f plasma_na.bld ] || [ ! -f blood_na.bld ]; then
  printf "\n Make input with NaN \n"
  cp -f plasma.bld plasma_na.bld
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacsety plasma_na.bld 1 20 nan
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  cp -f blood.bld blood_na.bld
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacsety blood_na.bld 1 10 nan
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi

if [ ! -f plasma_3.bld ] || [ ! -f blood_3.bld ]; then
  printf "\n Make input with only 3 samples \n"
  interpol -x=1.2,8,28 plasma.bld plasma_3.bld
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  interpol -x=1.2,8,28 blood.bld blood_3.bld
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi


if [ ! -f k2tis.dft ] || [ ! -f k2tis_few.dft ] || [ ! -f k2tis_long.dft ]; then
  rm -f k2tis1_na.dft
  printf "\n simulate tissue TACs \n"
  p2t_v3c -nosub plasma.bld blood.bld 0.3 0.15   0.00 0.00 0 0 0 4 30 temp2.dft
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacren --silent temp2.dft 1 ref1_mean_Pl01
  tacadd --silent -ovr temp.dft temp2.dft
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  p2t_v3c -nosub plasma.bld blood.bld 0.3 0.10   0.00 0.00 0 0 0 4 30 temp2.dft
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacren --silent temp2.dft 1 tis1_mean_Pl01
  tacadd --silent temp.dft temp2.dft
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  p2t_v3c -nosub plasma.bld blood.bld 0.02 0.010 0.00 0.00 0 0 0 4 30 temp2.dft
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacren --silent temp2.dft 1 tis2_mean_Pl01
  tacadd --silent temp.dft temp2.dft
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  p2t_v3c -nosub plasma.bld blood.bld 0.02 0.001 0.00 0.00 0 0 0 4 30 temp2.dft
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacren --silent temp2.dft 1 tis3_mean_Pl01
  tacadd --silent temp.dft temp2.dft
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  p2t_v3c -nosub plasma.bld blood.bld 0.1 0.02   0.00 0.00 0 0 0 4 30 temp2.dft
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacren --silent temp2.dft 1 tis4_mean_Pl01
  tacadd --silent temp.dft temp2.dft
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  p2t_v3c -nosub plasma.bld blood.bld 0.1 0.002  0.00 0.00 0 0 0 4 30 temp2.dft
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacren --silent temp2.dft 1 tis5_mean_Pl01
  tacadd --silent temp.dft temp2.dft
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  p2t_v3c -nosub plasma.bld blood.bld 0.6 0.01   0.00 0.00 0 0 0 4 30 temp2.dft
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacren --silent temp2.dft 1 tis6_mean_Pl01
  tacadd --silent temp.dft temp2.dft
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  p2t_v3c -nosub plasma.bld blood.bld 0.6 0.005  0.00 0.00 0 0 0 4 30 temp2.dft
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacren --silent temp2.dft 1 tis7_mean_Pl01
  tacadd --silent temp.dft temp2.dft
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  p2t_v3c -nosub plasma.bld blood.bld 0.05  0.00 0.00 0.00 0 0 0 4 30 temp2.dft
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacren --silent temp2.dft 1 tis8_mean_Pl01
  tacadd --silent temp.dft temp2.dft
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi

  printf "\n simulate PET frames \n"
  simframe --silent temp.dft frames.dat k2tis.dft
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacstudy k2tis.dft test1
  tac2svg k2tis.svg k2tis.dft
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  simframe --silent temp.dft frames_few.dat k2tis_few.dft
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacstudy k2tis_few.dft test2
  simframe --silent temp.dft frames_long.dat k2tis_long.dft
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacstudy k2tis_long.dft test3
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi

if [ ! -f k2tis1.dft ] || [ ! -f k2tis1_long.dft ]; then
  rm -f k2tis1_na.dft
  printf "\n For faster testing, take just one region \n"
  tacadd -ovr --silent k2tis1.dft k2tis.dft tis1
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacstudy k2tis1.dft fast1
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacadd -ovr --silent k2tis1_long.dft k2tis_long.dft tis1
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacstudy k2tis1_long.dft fast2
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi

if [ ! -f k2tis1_na.dft ]; then
  printf "\n Make TAC with NaN \n"
  cp -f k2tis1.dft k2tis1_na.dft
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacsety k2tis1_na.dft 1 12 nan
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi


printf "\n\n"
printf "=====================================================================\n"
printf "testing %s\n" $PROGRAM
printf "=====================================================================\n"


printf "\n 1.0.0 \n"
printf " Test case: Fit simulated TACs with realistic range of parameters. \n"
printf " Expected result: Result file is created, and results are close to correct \n"
printf " Expected result 2: SVG plot is created. \n"
printf " Expected result 3: Saved fitted TAC can be used for plotting, too. \n\n"

rm -f output.res fit_1_0_0.dat fit_1_0_0.svg
$PROGRAM -fit=fit_1_0_0.dat -svg=fit_1_0_0.svg plasma.bld blood.bld k2tis.dft 999 output.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
parmatch -abs=0.001 -rel=2 -tacnames=y -parunits=y -par=1-3 correct_1_0_0.res output.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

if [ ! -f fit_1_0_0.svg ]; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tac2svg fit_1_0_0b.svg -s k2tis.dft -l fit_1_0_0.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 1.0.1 \n"
printf " Test case 1: Fit simulated TACs with realistic range of parameters. \n"
printf " Test case 2: Vb constrained to correct value with option -Vb. \n"
printf " Expected result 1: Result file is created, and results are close to correct \n"
printf " Expected result 2: Fixed Vb shown in result headers. \n\n"
printf " Expected result 3: Parameter SD and CL are not saved by default. \n\n"

rm -f output.res fit_1_0_1.svg
$PROGRAM -Vb=4 -svg=fit_1_0_1.svg plasma.bld blood.bld k2tis.dft 999 output.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
parmatch -abs=0.001 -rel=2 -tacnames=y -parunits=y -par=1-2 -sd=y -cl=y correct_1_0_1.res output.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
parmatch -abs=0.000001 -par=3 correct_1_0_1.res output.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval output.res Vb "4 %"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n --------------------------------------------------------------- \n"

printf "\n 1.1.0 \n"
printf " Test case: Bootstrapping with options -sd=y and -cl=y. \n"
printf " Expected result: Result file contains sd and cl for parameters \n\n"

rm -f output.res fit_1_1_0.svg
$PROGRAM -sd=y -cl=y -svg=fit_1_1_0.svg plasma.bld blood.bld k2tis1_long.dft 60 output.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
parmatch -abs=0.001 -rel=2 -tacnames=y -parunits=y -par=1-3 -sd=y -cl=y correct_1_1_0.res output.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n --------------------------------------------------------------- \n"


printf "\n 1.2.0 \n"
printf " Test case 1: Calculate DVR by giving reference region name. \n"
printf " Test case 2: Fit end time is shorter than duration of data. \n"
printf " Expected result 1: Result file contains close to correct results. \n\n"
printf " Expected result 2: Given fit time is applied. \n\n"

rm -f output.res fit_1_2_0.svg
$PROGRAM -dvr=ref1 -svg=fit_1_2_0.svg plasma.bld blood.bld k2tis.dft 30 output.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
parmatch -abs=0.001 -rel=2 -tacnames=y -parunits=y -par=1-4 -sd=y -cl=y correct_1_2_0.res output.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval output.res "Data range" "0 - 30 min"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval output.res "Reference region" "ref1 mean Pl01"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n 1.2.1 \n"
printf " Test case 1: Calculate DVR by giving reference region number. \n"
printf " Expected result 1: Result file contains close to correct results. \n\n"

rm -f output.res fit_1_2_1.svg
$PROGRAM -dvr=1 -svg=fit_1_2_1.svg plasma.bld blood.bld k2tis.dft 30 output.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
parmatch -abs=0.001 -rel=2 -tacnames=y -parunits=y -par=1-4 -sd=y -cl=y correct_1_2_0.res output.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval output.res "Reference region" "ref1 mean Pl01"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 1.2.2 \n"
printf " Test case: Reference region is given in separate file. \n"
printf " Expected result: Result file contains close to correct results. \n\n"

tacadd -ovr reftemp.tac k2tis.dft ref1
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
cp -f k2tis.dft tistemp.tac
tacdel tistemp.tac ref1
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
cp -f correct_1_2_0.res tempc.res
parsort tempc.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi

rm -f output.res fit_1_2_2.svg
$PROGRAM -dvr=reftemp.tac -svg=fit_1_2_2.svg plasma.bld blood.bld tistemp.tac 30 output.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
parsort output.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
parmatch -abs=0.001 -rel=2 -tacnames=y -parunits=y -par=1-4 -sd=y -cl=y tempc.res output.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval output.res "Reference region" "ref1 mean Pl01"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval output.res "Reference file" "reftemp.tac"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n --------------------------------------------------------------- \n"


printf "\n 1.3.0 \n"
printf " Test case: Calculate BPnd by giving reference region name. \n"
printf " Expected result: Result file contains close to correct results. \n\n"

rm -f output.res fit_1_3_0.svg
$PROGRAM -BPnd=ref1 -Vb=4 -svg=fit_1_3_0.svg plasma.bld blood.bld k2tis.dft 120 output.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
parmatch -abs=0.001 -rel=2 -tacnames=y -parunits=y -par=1-4 -sd=y -cl=y correct_1_3_0.res output.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 1.4.0 \n"
printf " Test case 1: Calculate BPp by giving reference region name. \n"
printf " Test case 2: result file exists. \n"
printf " Expected result 1: Result file contains close to correct results. \n"
printf " Expected result 2: Backup is made from prev result file. \n\n"

rm -f fit_1_4_0.svg
$PROGRAM -BPp=ref1 -Vb=4 -svg=fit_1_4_0.svg plasma.bld blood.bld k2tis.dft 120 output.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
parmatch -abs=0.001 -rel=2 -tacnames=y -parunits=y -par=1-4 -sd=y -cl=y correct_1_4_0.res output.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"
parmatch -abs=0.001 -rel=2 -tacnames=y -parunits=y -par=1-4 -sd=y -cl=y correct_1_3_0.res output.res.bak
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n===================================================================\n"
printf "   Basic usability tests passed! \n"
printf "   Now testing error handling. \n"
printf "===================================================================\n"



printf "\n 2.0.0 \n"
printf " Test case: Unknwon option. \n"
printf " Expected result: Error. \n\n"

$PROGRAM -nonexistingoption plasma.bld blood.bld k2tis1.dft 30 output.res 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error "invalid option '-nonexistingoption'."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 2.0.1 \n"
printf " Test case: Missing filename. \n"
printf " Expected result: Error. \n\n"

$PROGRAM  blood.bld k2tis1.dft 30 output.res 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

$PROGRAM plasma.bld blood.bld k2tis1.dft 30  1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 2.1.0 \n"
printf " Test case: Nonexisting data file. \n"
printf " Expected result: Error. \n\n"

$PROGRAM nonexistingfile.dat blood.bld k2tis1.dft 30 output.res 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error "cannot read 'nonexistingfile.dat': cannot open file"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

$PROGRAM plasma.bld nonexistingfile.dat k2tis1.dft 30 output.res 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error "cannot read 'nonexistingfile.dat': cannot open file"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

$PROGRAM plasma.bld blood.bld nonexistingfile.dat 30 output.res 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error "cannot read 'nonexistingfile.dat': cannot open file"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 2.1.1 \n"
printf " Test case: Nonexisting blood file. \n"
printf " Expected result: No error if Vb=0. \n\n"

$PROGRAM -Vb=0 plasma.bld nonexistingfile.dat k2tis1.dft 30 output.res 1>stdout.txt 2>&1
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 2.1.2 \n"
printf " Test case: Nonexisting ref file. \n"
printf " Expected result: Error. \n\n"

$PROGRAM -DVR=nonexistingfile.dat plasma.bld blood.bld k2tis1.dft 30 output.res 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Error in reading 'nonexistingfile.dat'" "cannot find region"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 2.1.3 \n"
printf " Test case: Given reference region name does not exist. \n"
printf " Expected result: Error. \n\n"

$PROGRAM -DVR=stupid plasma.bld blood.bld k2tis1.dft 30 output.res 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Error in reading 'stupid'" "cannot find region"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 2.1.4 \n"
printf " Test case: Given reference region nr does not exist. \n"
printf " Expected result: Error. \n\n"

$PROGRAM -DVR=16 plasma.bld blood.bld k2tis1.dft 30 output.res 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Error in reading '16'" "cannot find region"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

$PROGRAM -DVR=0 plasma.bld blood.bld k2tis1.dft 30 output.res 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Error in reading '0'" "cannot find region"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n 2.2.0 \n"
printf " Test case: Invalid data file contents. \n"
printf " Expected result: Error. \n\n"

printf "# badfile\n" > badfile.dat
printf "ugh banzai\n" >> badfile.dat
printf "2 meh\n" >> badfile.dat

$PROGRAM badfile.dat blood.bld k2tis1.dft 30 output.res 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error "cannot read 'badfile.dat': wrong format"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

$PROGRAM plasma.bld badfile.dat k2tis1.dft 30 output.res 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error "cannot read 'badfile.dat': wrong format"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

$PROGRAM plasma.bld blood.bld badfile.dat 30 output.res 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error "cannot read 'badfile.dat': wrong format"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 2.2.1 \n"
printf " Test case: Invalid blood file contents. \n"
printf " Expected result: No error if Vb=0. \n\n"

$PROGRAM -Vb=0 plasma.bld badfile.dat k2tis1.dft 30 output.res 1>stdout.txt 2>&1
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 2.2.2 \n"
printf " Test case: Invalid ref file contents. \n"
printf " Expected result: Error. \n\n"

$PROGRAM -DVR=badfile.dat plasma.bld blood.bld k2tis1.dft 30 output.res 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Error in reading 'badfile.dat'"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n 2.3.0 \n"
printf " Test case: Input file contains more than one TAC. \n"
printf " Expected result 1: Warning. \n"
printf " Expected result 2: First TAC is used. \n\n"

taccalc plasma.bld . 0.001 temp.bld
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
tacadd -ovr plasma_2_3_0.bld plasma.bld
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
tacadd plasma_2_3_0.bld temp.bld
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi

taccalc blood.bld . 0.001 temp.bld
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
tacadd -ovr blood_2_3_0.bld blood.bld
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
tacadd blood_2_3_0.bld temp.bld
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi

$PROGRAM plasma_2_3_0.bld blood_2_3_0.bld k2tis1.dft 60 output.res 1>stdout.txt 2>&1
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Warning "using only first TAC in plasma_2_3_0.bld"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"
iftisval stdout.txt Warning "using only first TAC in blood_2_3_0.bld"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"
parmatch -abs=0.01 -rel=1 -par=1-3 correct_2_3_0.res output.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 2.3.1 \n"
printf " Test case: Ref file contains more than one TAC. \n"
printf " Expected result 1: Warning. \n\n"
printf " Expected result 2: First TAC is used. \n\n"

tacadd -ovr reftemp.tac k2tis.dft ref1
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
tacadd reftemp.tac k2tis.dft tis8
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi

rm -f output.res
$PROGRAM -dvr=reftemp.tac plasma.bld blood.bld k2tis.dft 60 output.res 1>stdout.txt 2>&1
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Warning "ref1 mean Pl01 selected of 2 reference regions."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"
iftisval output.res "Reference region" "ref1 mean Pl01"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval output.res "Reference file" "reftemp.tac"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 2.3.2 \n"
printf " Test case: Ref file contains different sample nr. \n"
printf " Expected result: Error. \n\n"

$PROGRAM -dvr=k2tis_long.dft plasma.bld blood.bld k2tis.dft 30 output.res 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Error in reading 'k2tis_long.dft'"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 2.4.0 \n"
printf " Test case: Data file contains NaN. \n"
printf " Expected result: Error. \n\n"

$PROGRAM plasma_na.bld blood.bld k2tis1.dft 60 output.res 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error "missing sample(s) in data"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

$PROGRAM plasma.bld blood_na.bld k2tis1.dft 60 output.res 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error "missing sample(s)"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

$PROGRAM plasma.bld blood.bld k2tis1_na.dft 60 output.res 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error "missing sample(s) in k2tis1_na.dft"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 2.5.0 \n"
printf " Test case: Data file contains only 3 samples. \n"
printf " Expected result: Error. \n\n"

$PROGRAM plasma_3.bld blood.bld k2tis1.dft 60 output.res 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error "plasma_3.bld contains too few samples"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

$PROGRAM plasma.bld blood_3.bld k2tis1.dft 60 output.res 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error "blood_3.bld contains too few samples"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

$PROGRAM plasma.bld blood.bld k2tis_few.dft 60 output.res 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error "too few samples in specified fit duration."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 2.5.1 \n"
printf " Test case: Blood file contains only 3 samples. \n"
printf " Expected result: No error if Vb=0. \n\n"

$PROGRAM -Vb=0 plasma.bld blood_3.bld k2tis1.dft 60 output.res 1>stdout.txt 2>&1
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n===================================================================\n"
printf "   Constraints \n"
printf "===================================================================\n"

printf "\n 3.0.0 \n"
printf " Test case: constraint file is created with option -lim=fname.\n"
printf " Expected result: File is created with correct contents. \n\n"

rm -f output.par
$PROGRAM -lim=output.par
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

iftisval output.par K1_lower 0
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
iftisval output.par K1_upper 10
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
iftisval output.par K1k2_lower 0
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
iftisval output.par K1k2_upper 500
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
iftisval output.par Vb_lower 0
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
iftisval output.par Vb_upper 0.08
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n 3.0.1 \n"
printf " Test case: existing constraint file and option -lim=fname.\n"
printf " Expected result: Error. \n\n"

$PROGRAM -lim=output.par 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Error" "parameter constraint file output.par exists."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n 3.0.2 \n"
printf " Test case: trying to use nonexisting constraint file.\n"
printf " Expected result: Error. \n\n"

$PROGRAM -lim=nonexisting.par plasma.bld blood.bld k2tis1.dft 60 output.res 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Error in reading 'nonexisting.par'" "cannot open file"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n 3.0.3 \n"
printf " Test case: trying to use constraint file with bad contents.\n"
printf " Expected result: Error. \n\n"

$PROGRAM -lim=badfile.dat plasma.bld blood.bld k2tis1.dft 60 output.res 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Error in reading 'badfile.dat'" "file contains no data"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n 3.1.0 \n"
printf " Test case: set constraints for one parameter with -lim.\n"
printf " Expected result: Limits are set correctly. \n\n"

printf "K1_lower := 0.1\n"  > limit.par
printf "K1_upper := 1.2\n" >> limit.par

$PROGRAM -d4 -lim=limit.par plasma.bld blood.bld k2tis1.dft 20 output.res 1>stdout.txt 2>&1
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt constraints "[0.1,1.2] [0,500] [0,0.08]"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 3.1.1 \n"
printf " Test case: set constraints for all parameters with -lim.\n"
printf " Expected result: Limits are set correctly. \n\n"

printf "K1k2_lower := 0.6\n" >> limit.par
printf "K1k2_upper := 1.0\n" >> limit.par
printf "Vb_lower := 0.02\n"  >> limit.par
printf "Vb_upper := 0.05\n"  >> limit.par

$PROGRAM -d4 -lim=limit.par plasma.bld blood.bld k2tis1.dft 20 output.res 1>stdout.txt 2>&1
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt constraints "[0.1,1.2] [0.6,1] [0.02,0.05]"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 3.2.0 \n"
printf " Test case: option -Vb=0\n"
printf " Expected result: Limits are set correctly. \n\n"

$PROGRAM -d4 -Vb=0 plasma.bld blood.bld k2tis1.dft 20 output.res 1>stdout.txt 2>&1
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt constraints "[0,10] [0,500] [0,0]"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 3.2.1 \n"
printf " Test case: option -Vb=4.5\n"
printf " Expected result: Limits are set correctly. \n\n"

$PROGRAM -d4 -Vb=4.5 plasma.bld blood.bld k2tis1.dft 20 output.res 1>stdout.txt 2>&1
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt constraints "[0,10] [0,500] [0.045,0.045]"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 3.2.2 \n"
printf " Test case: option -Vb=0.045 by mistake\n"
printf " Expected result: Warning. \n\n"

$PROGRAM -d4 -Vb=0.045 plasma.bld blood.bld k2tis1.dft 20 output.res 1>stdout.txt 2>&1
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Warning "Vb was set to 0.045%"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n===================================================================\n"
printf "   Weights \n"
printf "===================================================================\n"


printf "\n 4.0.0 \n"
printf " Test case: by default weights are 1\n"
printf " Expected result: correct weights are printed in test mode. \n\n"

$PROGRAM -d4 -Vb=4 plasma.bld blood.bld k2tis_long.dft 999 output.res 1>stdout.txt 2>&1
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt common_data_weights "1, 1, 1, 1, 1, 1, 1, 1"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n 4.0.1 \n"
printf " Test case: Datafile contains weights.\n"
printf " Expected result: correct weights are printed in test mode. \n\n"

cp -f k2tis_long.dft temp.tac
tacweigh -wf temp.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi

$PROGRAM -d4 -Vb=4 plasma.bld blood.bld temp.tac 999 output.res 1>stdout.txt 2>&1
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt common_data_weights "0.0666667, 0.0666667, 0.266667, 0.266667, 0.666667, 1.33333, 1.33333, 4"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n===================================================================\n"
printf "   All passed! \n"
printf "===================================================================\n"
exit 0

