#!/bin/bash
#: Title      : test_fitk2di
#: Date       : 2023-06-30
#: Author     : "Vesa Oikonen" <vesa.oikonen@utu.fi>
#: Options    : None

# Set the name of executable to test
if [ -n "${OS+1}" ] && [ "$OS" = "Windows_NT" ]; then
EXT=.exe;
else
EXT=;
fi

PROGRAM=../../fitk2di$EXT;
if [ ! -f $PROGRAM ]; then
  printf "Failed: executable does not exist.\n"
  exit 1
fi


printf "=====================================================================\n"
printf "creating test data for %s\n" $PROGRAM
printf "=====================================================================\n"


printf "\n verifying that required data exists \n"

if [ ! -f frames.dat ] || [ ! -f frames_few.dat ] || [ ! -f frames_long.dat ]; then
  printf "Failed: required data does not exist.\n"
  exit 1
fi
if [ ! -f parent.fit ] || [ ! -f km.lim ]; then
  printf "Failed: required data does not exist.\n"
  exit 1
fi
if [ ! -f correct_test0.res ] || [ ! -f correct_test1.res ]; then
  printf "Failed: required data does not exist.\n"
  exit 1
fi
if [ ! -f correct_test2.res ] || [ ! -f correct_test3.res ]; then
  printf "Failed: required data does not exist.\n"
  exit 1
fi
if [ ! -f correct_test2bpnd.res ] || [ ! -f correct_test2bpp.res ]; then
  printf "Failed: required data does not exist.\n"
  exit 1
fi
if [ ! -f correct_test2dvr.res ]; then
  printf "Failed: required data does not exist.\n"
  exit 1
fi
if [ ! -f test0.lim ] || [ ! -f test1.lim ] || [ ! -f test2.lim ]; then
  printf "Failed: required data does not exist.\n"
  exit 1
fi


printf "\n making data for testing \n"

if [ ! -f parent.dat ] || [ ! -f parent_i.dat ]; then
  printf "\n Making parent plasma TAC for simulation\n"
  rm -f metabolite1.dat metabolite2.dat plasma.dat metabolites.dat
  rm -f parent.rat blood.dat test0.dat test1.dat test2.dat test3.dat
  fit2dat -c=0,10,0.05 parent.fit temp1.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  fit2dat -c=10,120,0.25 parent.fit temp2.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  dftcat temp1.dat temp2.dat parent.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacstudy --force parent.dat parent
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  printf "\n Making parent plasma TAC for fitting\n"
  fit2dat -c=10,20,0.25 parent.fit temp2.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  dftcat temp1.dat temp2.dat parent_i.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  fit2dat -c=20,30,0.50 parent.fit temp2.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  dftcat parent_i.dat temp2.dat 
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  fit2dat -c=30,60,1.0 parent.fit temp2.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  dftcat parent_i.dat temp2.dat 
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  fit2dat -c=60,90,2.0 parent.fit temp2.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  dftcat parent_i.dat temp2.dat 
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  fit2dat -c=90,120,5.0 parent.fit temp2.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  dftcat parent_i.dat temp2.dat 
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacstudy --force parent_i.dat parent
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi

if [ ! -f metabolite1.dat ] || [ ! -f metabolite1_i.dat ]; then
  printf "\n Making metabolite1 plasma TAC \n"
  rm -f metabolite2.dat plasma.dat metabolites.dat blood.dat
  tactime -nogap parent.dat +1 temp1.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  p2t_3c -nosub temp1.dat 0.1 0.05 0 0 0 0 metabolite1.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacstudy --force metabolite1.dat metab1
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  interpol -f=parent_i.dat metabolite1.dat metabolite1_i.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi

if [ ! -f metabolite2.dat ] || [ ! -f metabolite2_i.dat ]; then
  printf "\n Making metabolite2 plasma TAC \n"
  rm -f plasma.dat metabolites.dat blood.dat
  tactime -nogap metabolite1.dat +1 temp2.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  p2t_3c -nosub temp2.dat 0.025 0.04 0 0 0 0 metabolite2.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacstudy --force metabolite2.dat metab2
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  interpol -f=parent_i.dat metabolite2.dat metabolite2_i.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi

if [ ! -f plasma.dat ] || [ ! -f metabolites.dat ] || [ ! -f plasma_i.dat]; then
  printf "\n Making total plasma TAC \n"
  rm -f parent.rat
  taccalc metabolite1.dat + metabolite2.dat metabolites.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacstudy --force metabolites.dat metab
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  taccalc parent.dat + metabolites.dat plasma.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacstudy --force plasma.dat plasma
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  interpol -f=parent_i.dat metabolites.dat metabolites_i.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  interpol -f=parent_i.dat plasma.dat plasma_i.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi

if [ ! -f parent.rat ]; then
  printf "\n Making parent.rat \n"
  taccalc parent.dat : plasma.dat parent.rat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacstudy --force parent.rat parent
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tac2svg parentrat.svg -l parent.rat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi

if [ ! -f blood.dat ] || [ ! -f blood0.dat ] || [ ! -f blood_i.dat ]; then
  printf "\n Simulating blood TAC, assuming that metab1 goes to RBC \n"
  p2blood norbc parent.dat 0.40 temp1.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  p2blood inwater metabolite1.dat 0.40 temp2.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  p2blood norbc metabolite2.dat 0.40 temp3.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  taccalc temp1.dat + temp2.dat temp4.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  taccalc temp4.dat + temp3.dat blood.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacstudy --force blood.dat blood
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  taccalc blood.dat : plasma.dat bp.rat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacstudy --force bp.rat bl-to-pl
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tac2svg bprat.svg -l bp.rat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  taccalc blood.dat x 0 blood0.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi

  interpol -f=parent_i.dat blood.dat blood_i.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  interpol -f=parent_i.dat blood0.dat blood0_i.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi

printf "\n plotting input TACs \n"
tac2svg input.svg -l plasma.dat blood.dat parent.dat metabolite1.dat metabolite2.dat metabolites.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi

if [ ! -f test0.dat ] || [ ! -f test0l.dat ]; then
  printf "\n Simulating tissue data with metab1 input and assumptions \n"
  printf " K1p=K1m and Vb=0 \n"
  p2t_3c -nosub parent.dat 0.6 0.2 0 0 0 0 temp1.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  p2t_3c -nosub metabolite1.dat 0.6 0.6 0 0 0 0 temp2.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  taccalc temp1.dat + temp2.dat tac1.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacren tac1.dat 0 tac1
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  p2t_3c -nosub parent.dat 0.12 0.04 0 0 0 0 temp1.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  p2t_3c -nosub metabolite1.dat 0.12 0.12 0 0 0 0 temp2.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  taccalc temp1.dat + temp2.dat tac2.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacren tac2.dat 0 tac2
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacadd -ovr temp3.dat tac1.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacadd temp3.dat tac2.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  simframe temp3.dat frames.dat test0.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacstudy --force test0.dat test0
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tac2svg test0.svg test0.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  simframe temp3.dat frames_long.dat test0l.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacstudy --force test0l.dat test0
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tac2svg test0l.svg test0l.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi

if [ ! -f test1.dat ] || [ ! -f test1l.dat ]; then
  printf "\n Simulating tissue data with metab1 input and assumption \n"
  printf " K1p=K1m \n"
  p2t_v3c -nosub parent.dat blood.dat 0.6 0.2 0 0 0 0 0 8 100 temp1.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  p2t_v3c -nosub metabolite1.dat blood0.dat 0.6 0.6 0 0 0 0 0 8 100 temp2.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  taccalc temp1.dat + temp2.dat tac1.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacren tac1.dat 0 tac1
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  p2t_v3c -nosub parent.dat blood.dat 0.12 0.04 0 0 0 0 0 8 100 temp1.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  p2t_v3c -nosub metabolite1.dat blood0.dat 0.12 0.12 0 0 0 0 0 8 100 temp2.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  taccalc temp1.dat + temp2.dat tac2.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacren tac2.dat 0 tac2
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacadd -ovr temp3.dat tac1.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacadd temp3.dat tac2.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  simframe temp3.dat frames.dat test1.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacstudy --force test1.dat test1
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tac2svg test1.svg test1.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  simframe temp3.dat frames_long.dat test1l.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacstudy --force test1l.dat test1
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tac2svg test1l.svg test1l.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi

if [ ! -f test2.dat ]; then
  printf "\n Simulating tissue data with metab2 input \n"
  p2t_v3c -nosub parent.dat blood.dat 0.2 0.1 0 0 0 0 0 8 100 temp1.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  p2t_v3c -nosub metabolite2.dat blood0.dat 0.4 0.5 0 0 0 0 0 8 100 temp2.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  taccalc temp1.dat + temp2.dat tac1.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacren tac1.dat 0 tac1
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  p2t_v3c -nosub parent.dat blood.dat 0.1 0.2 0 0 0 0 0 8 100 temp1.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  p2t_v3c -nosub metabolite2.dat blood0.dat 0.2 0.25 0 0 0 0 0 8 100 temp2.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  taccalc temp1.dat + temp2.dat tac2.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacren tac2.dat 0 tac2
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacadd -ovr temp4.dat tac1.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacadd temp4.dat tac2.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  simframe temp4.dat frames.dat test2.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacstudy --force test2.dat test2
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tac2svg test2.svg test2.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi

if [ ! -f test3.dat ] || [ ! -f test3p.dat ]; then
  printf "\n Simulating tissue data with km larger than 0 \n"
  p2t_di -parallel -sub parent.dat metabolite1.dat blood.dat 0.6 0.05 0 0 0 0 0 0.1 0.2 0.2 8 temp.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacadd -ovr tac3.dat temp.dat 1  
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacren tac3.dat 0 tac3
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  simframe tac3.dat frames.dat test3.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacstudy --force test3.dat test3
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi

  printf "\n Saving also tissue data without metabolite \n"
  tacadd -ovr temp1.dat temp.dat 2
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  taccbv -tv -sim temp1.dat blood.dat 0.08 tac3p.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacren tac3p.dat 0 tac3
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  simframe tac3p.dat frames.dat test3p.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacstudy --force test3p.dat test3
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi

  tac2svg test3.svg test3.dat test3p.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi


printf "\n\n"
printf "=====================================================================\n"
printf "testing %s\n" $PROGRAM
printf "=====================================================================\n"

printf "\n 0.1.0 \n"
printf " Test case: Default values of variables that user can set are correct. \n"
printf " Expected result 1: Test output contains correct variable values. \n"
printf " Expected result 2: Result file is created. \n\n"
printf " Notice: Fitting may take a long time. \n\n"

rm -f test1l.res
$PROGRAM -d3 -lim=test1.lim parent_i.dat metabolite1_i.dat blood_i.dat test1l.dat 90 test1l.res 1>stdout.txt 2>&1
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt pfile "parent_i.dat"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt mfile "metabolite1_i.dat"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt bfile "blood_i.dat"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt dfile "test1l.dat"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt rfile "test1l.res"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt fitdur "90"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt doBootstrap 0
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt bp_type "0"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
if [ ! -f test1l.res ]; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n===================================================================\n"
printf "   General usability \n"
printf "===================================================================\n"


printf "\n 1.0.0 \n"
printf " Test case: test0.dat. \n"
printf " Expected result: Correct results are saved. \n\n"
printf " Notice: Fitting may take a long time. \n\n"

rm -f 1_0_0.svg test0.res
$PROGRAM -svg=1_0_0.svg -lim=test0.lim parent_i.dat metabolite1_i.dat none test0.dat 999 test0.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
parmatch -abs=0.01 -rel=2 -tacnames=y -parunits=y -par=1-6 correct_test0.res test0.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
if [ ! -f 1_0_0.svg ]; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n 1.1.0 \n"
printf " Test case: test1.dat. \n"
printf " Expected result: Correct results are saved. \n\n"
printf " Notice: Fitting may take a long time. \n\n"

rm -f 1_1_0.svg test1.res
$PROGRAM -svg=1_1_0.svg -lim=test1.lim parent_i.dat metabolite1_i.dat blood_i.dat test1.dat 999 test1.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
parmatch -abs=0.01 -rel=2 -tacnames=y -parunits=y -par=1-6 correct_test1.res test1.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
if [ ! -f 1_1_0.svg ]; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n 1.2.0 \n"
printf " Test case: test2.dat. \n"
printf " Expected result: Correct results are saved. \n\n"
printf " Notice: Fitting may take a long time. \n\n"

rm -f 1_2_0.svg test2.res
$PROGRAM -svg=1_2_0.svg -lim=test2.lim parent_i.dat metabolite2_i.dat blood_i.dat test2.dat 999 test2.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
parmatch -abs=0.01 -rel=2 -tacnames=y -parunits=y -par=1-6 correct_test2.res test2.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
if [ ! -f 1_2_0.svg ]; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n 1.3.0 \n"
printf " Test case 1: test3.dat. \n"
printf " Test case 2: fitting km \n"
printf " Test case 3: using limits file \n"
printf " Expected result: Correct results are saved. \n\n"
printf " Notice: Fitting may take a long time. \n\n"

rm -f 1_3_0.svg test3.res
$PROGRAM -lim=km.lim -vb=8 -svg=1_3_0.svg parent_i.dat metabolite1_i.dat blood_i.dat test3.dat 999 test3.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
parmatch -abs=0.01 -rel=2 -tacnames=y -parunits=y -par=1-6 correct_test3.res test3.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
if [ ! -f 1_3_0.svg ]; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n===================================================================\n"
printf "   Reference region \n"
printf "===================================================================\n"


printf "\n 2.0.0 \n"
printf " Test case: DVR calculation with reference region. \n"
printf " Expected result: Correct results are saved. \n\n"
printf " Notice: Fitting may take a long time. \n\n"

rm -f 2_0_0.svg test2dvr.res
$PROGRAM -dvr=tac2 -vb=8 -svg=2_0_0.svg parent_i.dat metabolite2_i.dat blood_i.dat test2.dat 999 test2dvr.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
parmatch -abs=0.01 -rel=2 -tacnames=y -parunits=y -par=1-7 correct_test2dvr.res test2dvr.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
if [ ! -f 2_0_0.svg ]; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 2.1.0 \n"
printf " Test case: BPnd calculation with reference region. \n"
printf " Expected result: Correct results are saved. \n\n"
printf " Notice: Fitting may take a long time. \n\n"

rm -f 2_1_0.svg test2bpnd.res
$PROGRAM -BPnd=tac2 -vb=8 -svg=2_1_0.svg parent_i.dat metabolite2_i.dat blood_i.dat test2.dat 999 test2bpnd.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
parmatch -abs=0.01 -rel=2 -tacnames=y -parunits=y -par=1-7 correct_test2bpnd.res test2bpnd.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
if [ ! -f 2_1_0.svg ]; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 2.2.0 \n"
printf " Test case: BPp calculation with reference region. \n"
printf " Expected result: Correct results are saved. \n\n"
printf " Notice: Fitting may take a long time. \n\n"

rm -f 2_2_0.svg test2bpp.res
$PROGRAM -BPp=tac2 -vb=8 -svg=2_2_0.svg parent_i.dat metabolite2_i.dat blood_i.dat test2.dat 999 test2bpp.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
parmatch -abs=0.01 -rel=2 -tacnames=y -parunits=y -par=1-7 correct_test2bpp.res test2bpp.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
if [ ! -f 2_2_0.svg ]; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n===================================================================\n"
printf "   Parameter constraints \n"
printf "===================================================================\n"


printf "\n 3.0.0 \n"
printf " Test case: Own constraints for reference region \n"
printf " Expected result: without option -ref the ref constraints have no effect. \n\n"
printf " Notice: Fitting may take a long time. \n\n"

printf "ref_Vfp_lower := 1.5\nref_Vfp_upper := 1.5\n" > temp.lim
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi

rm -f test2b.res
$PROGRAM -lim=temp.lim -Vb=8 parent_i.dat metabolite2_i.dat blood_i.dat test2.dat 999 test2.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
parmatch -abs=0.01 -rel=2 -tacnames=y -parunits=y -par=1-6 correct_test2.res test2.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n 3.0.1 \n"
printf " Test case: Own constraints for reference region with option -ref\n"
printf " Expected result: with option -ref the ref constraints do have an effect\n"
printf "                  but only on ref region. \n\n"
printf " Notice: Fitting may take a long time. \n\n"

rm -f test2ref.res test2ref.ift
$PROGRAM -lim=temp.lim -ref=tac2 -Vb=8 parent_i.dat metabolite2_i.dat blood_i.dat test2.dat 999 test2ref.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
reslist -n -ift test2ref.res > test2res.ift
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
parmatch -abs=0.01 -rel=2 -tacnames=y -parunits=y -par=1-6 correct_test2.res test2ref.res
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval test2res.ift tac1_K1p/k2p "1.5 ml/ml"
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval test2res.ift tac2_K1p/k2p "1.5 ml/ml"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n 3.1.0 \n"
printf " Test case: Option for fixing reference Vfm to Vfp \n"
printf " Expected result: Option -refVfm=refVfp has no effect if reference \n"
printf "                  region is not defined \n\n"
printf " Notice: Fitting may take a long time. \n\n"

rm -f test2.res
$PROGRAM -refVfm=refVfp -Vb=8 parent_i.dat metabolite2_i.dat blood_i.dat test2.dat 999 test2.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
parmatch -abs=0.01 -rel=2 -tacnames=y -parunits=y -par=1-6 correct_test2.res test2.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n 3.1.1 \n"
printf " Test case: Option for fixing reference Vfm to Vfp \n"
printf " Expected result: Option -refVfm=refVfp effects reference region. \n\n"
printf " Notice: Fitting may take a long time. \n\n"

rm -f test2ref.res test2ref.ift
$PROGRAM --debug -refVfm=refVfp -ref=tac2 -Vb=8 parent_i.dat metabolite2_i.dat blood_i.dat test2.dat 999 test2ref.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

# ref region tac2 should be changed
parmatch -tac=2 -abs=0.015 -par=1-6 correct_test2.res test2ref.res
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
# tac1 should not be changed
parmatch -tac=1 -abs=0.015 -par=1-6 correct_test2.res test2ref.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

# check that Vfp equals Vfm in ref region
reslist -ift test2ref.res > test2ref.ift
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftmatch -abs=0.000001 test2ref.ift test2ref.ift tac2_K1p/k2p tac2_K1m/k2m
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"





printf "\n===================================================================\n"
printf "   All passed! \n"
printf "===================================================================\n"
exit 0

