#!/bin/bash
#: Title      : test_fitk4
#: Date       : 2023-06-29
#: Author     : "Vesa Oikonen" <vesa.oikonen@utu.fi>
#: Options    : None

# Set the name of executable to test
if [ -n "${OS+1}" ] && [ "$OS" = "Windows_NT" ]; then
EXT=.exe;
else
EXT=;
fi

PROGRAM=../../fitk4$EXT;
if [ ! -f $PROGRAM ]; then
  printf "Failed: executable does not exist.\n"
  exit 1
fi

printf "=====================================================================\n"
printf "creating test data for %s\n" $PROGRAM
printf "=====================================================================\n"


printf "\n verifying that required data exists \n"


if [ ! -f plasma.dat ] || [ ! -f frames.dat ]; then
  printf "Failed: required data does not exist.\n"
  exit 1
fi
if [ ! -f frames_few.dat ] || [ ! -f frames_long.dat ]; then
  printf "Failed: required data does not exist.\n"
  exit 1
fi
if [ ! -f correct_1_0_0.res ] || [ ! -f correct_1_1_0.res ]; then
  printf "Failed: required data does not exist.\n"
  exit 1
fi
if [ ! -f correct_1_1_1.res ] || [ ! -f correct_1_3_0.res ]; then
  printf "Failed: required data does not exist.\n"
  exit 1
fi

printf "\n making data for testing \n"


if [ ! -f plasma.bld ] || [ ! -f blood.bld ]; then
  rm -f plasma_na.bld blood_na.bld
  rm -f plasma_3.bld blood_3.bld
  rm -f sim.tac pet.tac
  printf "\n making plasma.bld \n"
  tacformat -f=PMOD plasma.dat plasma.bld
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacunit -xset=min -yset=kBq/ml plasma.bld
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacren plasma.bld 0 plasma
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  printf "\n making blood.bld \n"
  taccalc plasma.bld x 0.5 blood.bld
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacren blood.bld 0 blood
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tac2svg -mt="Input" -legend=y input.svg -l plasma.bld blood.bld 
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi

if [ ! -f plasma_na.bld ] || [ ! -f blood_na.bld ]; then
  printf "\n Make input with NaN \n"
  cp -f plasma.bld plasma_na.bld
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacsety plasma_na.bld 1 20 nan
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  cp -f blood.bld blood_na.bld
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacsety blood_na.bld 1 10 nan
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi

if [ ! -f plasma_3.bld ] || [ ! -f blood_3.bld ]; then
  printf "\n Make input with only 3 samples \n"
  interpol -x=1.2,8,28 plasma.bld plasma_3.bld
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  interpol -x=1.2,8,28 blood.bld blood_3.bld
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi


# Region   K1      K1/k2    k3       k3/k4   Vb     Vt
# Ref  1   0.3     2        0        0       4      2
# Ref  2   0.3     2        0.10     0.5     4      3
# Tis  1   0.3     2        0.20     2       4      6
# Tis  2   0.3     2        0.80     8       4      18
# Tis  3   0.3     2        0.40     1       4      4
# Tis  4   0.1     2        0.20     2       4      6
# Tis  5   0.1     2        0.80     8       4      18
# Tis  6   0.1     2        0.40     1       4      4

if [ ! -f sim.tac ]; then
  rm -f pet.tac pet_few.tac pet_long.tac
  printf "\n simulate tissue TACs \n"

  p2t_v3c -nosub plasma.bld blood.bld 0.3 0.15 0.00 0.00 0 0 0 4 30 temp.tac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacren --silent temp.tac 1 ref_1
  tacadd --silent -ovr sim.tac temp.tac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi

  p2t_v3c -nosub plasma.bld blood.bld 0.3 0.15 0.10 0.20 0 0 0 4 30 temp.tac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacren --silent temp.tac 1 ref_2
  tacadd --silent sim.tac temp.tac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi

  p2t_v3c -nosub plasma.bld blood.bld 0.3 0.15 0.20 0.10 0 0 0 4 30 temp.tac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacren --silent temp.tac 1 tis_1
  tacadd --silent sim.tac temp.tac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi

  p2t_v3c -nosub plasma.bld blood.bld 0.3 0.15 0.80 0.10 0 0 0 4 30 temp.tac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacren --silent temp.tac 1 tis_2
  tacadd --silent sim.tac temp.tac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi

  p2t_v3c -nosub plasma.bld blood.bld 0.3 0.15 0.40 0.40 0 0 0 4 30 temp.tac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacren --silent temp.tac 1 tis_3
  tacadd --silent sim.tac temp.tac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi

  p2t_v3c -nosub plasma.bld blood.bld 0.1 0.05 0.20 0.10 0 0 0 4 30 temp.tac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacren --silent temp.tac 1 tis_4
  tacadd --silent sim.tac temp.tac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi

  p2t_v3c -nosub plasma.bld blood.bld 0.1 0.05 0.80 0.10 0 0 0 4 30 temp.tac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacren --silent temp.tac 1 tis_5
  tacadd --silent sim.tac temp.tac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi

  p2t_v3c -nosub plasma.bld blood.bld 0.1 0.05 0.40 0.40 0 0 0 4 30 temp.tac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacren --silent temp.tac 1 tis_6
  tacadd --silent sim.tac temp.tac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi

fi

if [ ! -f pet.tac ] || [ ! -f pet_few.tac ] || [ ! -f pet_long.tac ]; then
  printf "\n simulate PET frames \n"
  rm -f pet_t2.tac pet_t2_long.tac pet_na.tac

  simframe --silent sim.tac frames.dat pet.tac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacstudy pet.tac test1
  tac2svg -legend=y -mt="Simulated TTACs" pet.svg pet.tac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi

  simframe --silent sim.tac frames_few.dat pet_few.tac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacstudy pet_few.tac test2

  simframe --silent sim.tac frames_long.dat pet_long.tac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacstudy pet_long.tac test3
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi

fi

if [ ! -f pet_t2.tac ] || [ ! -f pet_t2_long.tac ]; then
  printf "\n extract just one TAC for quick testing \n"
  tacadd -ovr pet_t2.tac pet.tac tis_2
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacadd -ovr pet_t2_long.tac pet_long.tac tis_2
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi

if [ ! -f pet_na.tac ]; then
  printf "\n Make TAC with NaN \n"
  cp -f pet.tac pet_na.tac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacsety pet_na.tac 1 12 nan
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi


printf "\n\n"
printf "=====================================================================\n"
printf "testing %s\n" $PROGRAM
printf "=====================================================================\n"

printf "\n 1.0.0 \n"
printf " Test case: Fit simulated TACs with realistic range of parameters. \n"
printf " Expected result: Result file is created, and results are close to correct \n"
printf " Expected result 2: SVG plot is created. \n"
printf " Expected result 3: Fitted TACs should be close to original TACs.\n"
printf " Expected result 4: Saved fitted TAC can be used for plotting, too. \n\n"

rm -f output_1_0_0.res fit_1_0_0.tac fit_1_0_0.svg
$PROGRAM -fit=fit_1_0_0.tac -svg=fit_1_0_0.svg plasma.bld blood.bld pet.tac 999 output_1_0_0.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

if [ ! -f fit_1_0_0.svg ]; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

# Parameter values from TAC1 are not verified because it is simulated with
# 1TCM and therefore k3 and k4 can get arbitrary values
parmatch -tac=2-8 -par=1-6 -abs=0.001 -rel=2 -tacnames=y -parunits=y correct_1_0_0.res output_1_0_0.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
# ... except that lets try it anyway, for now
parmatch -tac=1 -par=1,5-6 -abs=0.001 -rel=2 -tacnames=y -parunits=y correct_1_0_0.res output_1_0_0.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

tacmatch -abs=0.1 -rel=2 pet.tac fit_1_0_0.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

tac2svg -legend=y fit_1_0_0b.svg -s pet.tac -l fit_1_0_0.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n 1.0.1 \n"
printf " Test case: Constrain K1/k2 based on previously fitted median. \n"
printf " Expected result: Results are close to correct \n\n"

rm -f output_1_0_1.res fit_1_0_1.svg
$PROGRAM -fk1k2=output_1_0_0.res -svg=fit_1_0_1.svg plasma.bld blood.bld pet.tac 999 output_1_0_1.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

parmatch -tac=2-8 -par=1-6 -abs=0.001 -rel=2 -tacnames=y -parunits=y correct_1_0_0.res output_1_0_1.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
parmatch -tac=1 -par=1-2 -abs=0.001 -rel=2 -tacnames=y -parunits=y correct_1_0_0.res output_1_0_1.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 1.1.0 \n"
printf " Test case: Constrain K1/k2 based on reference region fitted with 1TCM. \n"
printf " Test case 2: BPnd is calculated with option -bpnd.\n"
printf " Expected result: Results are close to correct \n\n"

rm -f output_1_1_0.res fit_1_1_0.svg
$PROGRAM -r="ref 1" -rmod=2 -BPnd -svg=fit_1_1_0.svg plasma.bld blood.bld pet.tac 999 output_1_1_0.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

parmatch -tac=1-8 -par=1-7 -abs=0.001 -rel=2 -tacnames=y -parunits=y correct_1_1_0.res output_1_1_0.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 1.1.1 \n"
printf " Test case: Constrain K1/k2 based on reference region fitted with 2TCM. \n"
printf " Test case 2: BPnd is calculated with option -bpnd.\n"
printf " Expected result: Results are close to correct \n\n"

rm -f output_1_1_1.res
$PROGRAM -r="ref 2" -rmod=3 -BPnd -svg=fit_1_1_1.svg plasma.bld blood.bld pet.tac 999 output_1_1_1.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

parmatch -tac=2-8 -par=1-7 -abs=0.001 -rel=2 -tacnames=y -parunits=y correct_1_1_1.res output_1_1_1.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
parmatch -tac=1 -par=1-2 -abs=0.001 -rel=2 -tacnames=y -parunits=y correct_1_1_1.res output_1_1_1.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n 1.1.2 \n"
printf " Test case: Reference region is given in separate file. \n"
printf " Expected result: Result file contains the same results as before. \n\n"

rm -f output_1_1_2.res
tacadd -ovr reftemp.tac pet.tac "ref 2"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
cp -f pet.tac tistemp.tac
tacdel tistemp.tac "ref 2"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
cp -f correct_1_1_1.res tempc.res
parsort tempc.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

$PROGRAM -r=reftemp.tac -rmod=3 -BPnd -svg=fit_1_1_2.svg plasma.bld blood.bld tistemp.tac 999 output_1_1_2.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
parsort output_1_1_2.res
parmatch -tac=2-8 -par=1-7 -abs=0.001 -rel=2 -tacnames=y -parunits=y tempc.res output_1_1_2.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n --------------------------------------------------------------- \n"


printf "\n 1.2.0 \n"
printf " Test case: Constrain Vb with command-line option. \n"
printf " Expected result: Results and header contain correct Vb. \n\n"

rm -f output.res output.ift
$PROGRAM -Vb=2 plasma.bld blood.bld pet_long.tac 999 output.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

iftisval output.res Vb "2 %"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

parformat -f=IFT -hdr=No output.res output.ift
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval output.ift Vb "2 %"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n --------------------------------------------------------------- \n"


printf "\n 1.3.0 \n"
printf " Test case: Bootstrapping with options -sd=y and -cl=y. \n"
printf " Expected result: Result file contains sd and cl for parameters \n\n"

rm -f output.res
$PROGRAM -sd=y -cl=y -Vb=4 plasma.bld blood.bld pet_t2_long.tac 90 output.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
parmatch -abs=0.001 -rel=2 -tacnames=y -parunits=y -par=1-4 -sd=y -cl=y correct_1_3_0.res output.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n --------------------------------------------------------------- \n"


printf "\n 1.4.0 \n"
printf " Test case: Fit end time is shorter than duration of data. \n"
printf " Expected result: Given fit time is applied. \n\n"

rm -f output.res
$PROGRAM -Vb=4 plasma.bld blood.bld pet_t2_long.tac 70 output.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval output.res "Data range" "0 - 60 min"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
parmatch -abs=0.001 -rel=2 -tacnames=y -parunits=y -par=1-4 -sd=n -cl=n correct_1_3_0.res output.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n===================================================================\n"
printf "   Basic usability tests passed! \n"
printf "   Now testing error handling. \n"
printf "===================================================================\n"


printf "\n 2.0.0 \n"
printf " Test case: Unknown option. \n"
printf " Expected result: Error. \n\n"

$PROGRAM -nonexistingoption plasma.bld blood.bld pet_t2_long.tac 30 output.res 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error "invalid option '-nonexistingoption'."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n 2.0.1 \n"
printf " Test case: Missing filename. \n"
printf " Expected result: Error. \n\n"

$PROGRAM  blood.bld pet_t2_long.tac 30 output.res 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

$PROGRAM  plasma.bld pet_t2_long.tac 30 output.res 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

$PROGRAM plasma.bld blood.bld  30 output.res 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

$PROGRAM plasma.bld blood.bld pet_t2_long.tac 30  1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n --------------------------------------------------------------- \n"


printf "\n 2.1.0 \n"
printf " Test case: Nonexisting data file. \n"
printf " Expected result: Error. \n\n"

$PROGRAM nonexistingfile.dat blood.bld pet.tac 30 output.res 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error "cannot read 'nonexistingfile.dat': cannot open file"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

$PROGRAM plasma.bld nonexistingfile.dat pet.tac 30 output.res 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error "cannot read 'nonexistingfile.dat': cannot open file"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

$PROGRAM plasma.bld blood.bld nonexistingfile.dat 30 output.res 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error "cannot read 'nonexistingfile.dat': cannot open file"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n 2.1.1 \n"
printf " Test case: Nonexisting blood file. \n"
printf " Expected result: No error if Vb=0. \n\n"

$PROGRAM -Vb=0 plasma.bld nonexistingfile.dat pet_t2_long.tac 30 output.res 1>stdout.txt 2>&1
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 2.1.2 \n"
printf " Test case: Nonexisting ref file. \n"
printf " Expected result: Error. \n\n"

$PROGRAM -R=nonexistingfile.dat plasma.bld blood.bld pet.tac 30 output.res 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Error in reading 'nonexistingfile.dat'" "cannot find region"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 2.1.3 \n"
printf " Test case: Given reference region name does not exist. \n"
printf " Expected result: Error. \n\n"

$PROGRAM -r=stupid plasma.bld blood.bld pet.tac 30 output.res 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Error in reading 'stupid'" "cannot find region"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

$PROGRAM -BPnd=stupid plasma.bld blood.bld pet.tac 30 output.res 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Error" "reference region stupid not found"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 2.1.4 \n"
printf " Test case: Given reference region nr does not exist. \n"
printf " Expected result: Error. \n\n"

$PROGRAM -R=16 plasma.bld blood.bld pet.tac 30 output.res 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Error in reading '16'" "cannot find region"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

$PROGRAM -R=0 plasma.bld blood.bld pet.tac 30 output.res 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Error in reading '0'" "cannot find region"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n --------------------------------------------------------------- \n"

printf "\n 2.2.0 \n"
printf " Test case: Invalid data file contents. \n"
printf " Expected result: Error. \n\n"

rm -f badfile.dat
printf "# badfile\n" > badfile.dat
printf "ugh banzai\n" >> badfile.dat
printf "2 meh\n" >> badfile.dat

$PROGRAM badfile.dat blood.bld pet.tac 30 output.res 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error "cannot read 'badfile.dat': wrong format"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

$PROGRAM plasma.bld badfile.dat pet.tac 30 output.res 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error "cannot read 'badfile.dat': wrong format"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

$PROGRAM plasma.bld blood.bld badfile.dat 30 output.res 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error "cannot read 'badfile.dat': wrong format"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n 2.2.1 \n"
printf " Test case: Invalid blood file contents. \n"
printf " Expected result: No error if Vb=0. \n\n"

$PROGRAM -Vb=0 plasma.bld badfile.dat pet_t2_long.tac 30 output.res 1>stdout.txt 2>&1
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 2.2.2 \n"
printf " Test case: Invalid ref file contents. \n"
printf " Expected result: Error. \n\n"

$PROGRAM -r=badfile.dat plasma.bld blood.bld pet.tac 30 output.res 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Error in reading 'badfile.dat'"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n --------------------------------------------------------------- \n"

printf "\n 2.3.0 \n"
printf " Test case: Input file contains more than one TAC. \n"
printf " Expected result 1: Warning. \n"
printf " Expected result 2: First TAC is used. \n\n"

taccalc plasma.bld . 0.001 temp.bld
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
tacadd -ovr plasma_2_3_0.bld plasma.bld
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
tacadd plasma_2_3_0.bld temp.bld
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi

taccalc blood.bld . 0.001 temp.bld
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
tacadd -ovr blood_2_3_0.bld blood.bld
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
tacadd blood_2_3_0.bld temp.bld
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi

rm -f output.res
$PROGRAM -Vb=4 plasma_2_3_0.bld blood_2_3_0.bld pet_t2_long.tac 60 output.res 1>stdout.txt 2>&1
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Warning "using only first TAC in plasma_2_3_0.bld"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"
iftisval stdout.txt Warning "using only first TAC in blood_2_3_0.bld"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"
parmatch -abs=0.01 -rel=1 -par=1-4 -sd=n -cl=n correct_1_3_0.res output.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n 2.3.1 \n"
printf " Test case: Ref file contains more than one TAC. \n"
printf " Expected result 1: Warning. \n\n"
printf " Expected result 2: First TAC is used. \n\n"

tacadd -ovr reftemp.tac pet_long.tac "ref 1"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
tacadd reftemp.tac pet_long.tac "tis 4"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi

rm -f output.res
$PROGRAM -r=reftemp.tac -rmod=2 plasma.bld blood.bld pet_t2_long.tac 30 output.res 1>stdout.txt 2>&1
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Warning "ref 1 selected of 2 reference regions."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"
iftisval output.res "Reference region" "ref 1"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval output.res "Reference file" "reftemp.tac"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
resdel output.res "ref 1"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
resdel output.res "tis 4"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
parmatch -abs=0.01 -rel=1 -par=1-4 -sd=n -cl=n correct_1_3_0.res output.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n --------------------------------------------------------------- \n"

printf "\n 2.4.0 \n"
printf " Test case: Data file contains NaN. \n"
printf " Expected result: Error. \n\n"

$PROGRAM plasma_na.bld blood.bld pet.tac 60 output.res 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error "missing sample(s) in data"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

$PROGRAM plasma.bld blood_na.bld pet.tac 60 output.res 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error "missing sample(s)"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

$PROGRAM plasma.bld blood.bld pet_na.tac 60 output.res 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error "missing sample(s) in pet_na.tac"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n --------------------------------------------------------------- \n"


printf "\n 2.5.0 \n"
printf " Test case: Data file contains only 3 samples. \n"
printf " Expected result: Error. \n\n"

$PROGRAM plasma_3.bld blood.bld pet.tac 60 output.res 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error "plasma_3.bld contains too few samples"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

$PROGRAM plasma.bld blood_3.bld pet.tac 60 output.res 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error "blood_3.bld contains too few samples"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

$PROGRAM plasma.bld blood.bld pet_few.tac 60 output.res 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error "too few samples in specified fit duration."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n 2.5.1 \n"
printf " Test case: Blood file contains only 3 samples. \n"
printf " Expected result: No error if Vb=0. \n\n"

$PROGRAM -Vb=0 plasma.bld blood_3.bld pet_t2_long.tac 60 output.res 1>stdout.txt 2>&1
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"




printf "\n===================================================================\n"
printf "   Constraints \n"
printf "===================================================================\n"

printf "\n 3.0.0 \n"
printf " Test case: constraint file is created with option -lim=fname.\n"
printf " Expected result: File is created with correct contents. \n\n"

rm -f output.par
$PROGRAM -lim=output.par
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

iftisval output.par K1_lower 0
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
iftisval output.par K1_upper 5
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
iftisval -abs=0.0000001 output.par K1k2_lower 0.00001
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
iftisval output.par K1k2_upper 10
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
iftisval output.par k3_lower 0
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
iftisval output.par k3_upper 2
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
iftisval output.par k3k4_lower 0
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
iftisval output.par k3k4_upper 10
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
iftisval output.par Vb_lower 0
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
iftisval output.par Vb_upper 0.08
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n 3.0.1 \n"
printf " Test case: existing constraint file and option -lim=fname.\n"
printf " Expected result: Error. \n\n"

$PROGRAM -lim=output.par 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Error" "parameter constraint file output.par exists."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 3.0.2 \n"
printf " Test case: trying to use nonexisting constraint file.\n"
printf " Expected result: Error. \n\n"

$PROGRAM -lim=nonexisting.par plasma.bld blood.bld pet_long.tac 60 output.res 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Error in reading 'nonexisting.par'" "cannot open file"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 3.0.3 \n"
printf " Test case: trying to use constraint file with bad contents.\n"
printf " Expected result: Error. \n\n"

$PROGRAM -lim=badfile.dat plasma.bld blood.bld pet_long.tac 60 output.res 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Error in reading 'badfile.dat'" "file contains no data"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"




printf "\n 3.1.0 \n"
printf " Test case: set constraints for one parameter with -lim.\n"
printf " Expected result: Limits are set correctly. \n\n"

printf "K1_lower := 0.05\n"  > limit.par
printf "K1_upper := 0.8\n" >> limit.par

$PROGRAM -d4 -lim=limit.par plasma.bld blood.bld pet_t2_long.tac 20 output.res 1>stdout.txt 2>&1
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt constraints "[0.05,0.8] [1e-05,10] [0,2] [0,10] [0,0.08]"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 3.1.1 \n"
printf " Test case: set constraints for all parameters with -lim.\n"
printf " Expected result: Limits are set correctly. \n\n"

printf "K1k2_lower := 0.1\n" >> limit.par
printf "K1k2_upper := 2.0\n" >> limit.par
printf "k3_lower := 0\n" >> limit.par
printf "k3_upper := 3.0\n" >> limit.par
printf "k3k4_lower := 0\n" >> limit.par
printf "k3k4_upper := 30.0\n" >> limit.par
printf "Vb_lower := 0.02\n"  >> limit.par
printf "Vb_upper := 0.06\n"  >> limit.par

$PROGRAM -d4 -lim=limit.par plasma.bld blood.bld pet_t2_long.tac 20 output.res 1>stdout.txt 2>&1
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt constraints "[0.05,0.8] [0.1,2] [0,3] [0,30] [0.02,0.06]"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 3.2.0 \n"
printf " Test case: option -Vb=0\n"
printf " Expected result: Limits are set correctly. \n\n"

$PROGRAM -d4 -Vb=0 plasma.bld blood.bld pet_t2_long.tac 20 output.res 1>stdout.txt 2>&1
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt constraints "[0,5] [1e-05,10] [0,2] [0,10] [0,0]"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 3.2.1 \n"
printf " Test case: option -Vb=4.5\n"
printf " Expected result: Limits are set correctly. \n\n"

$PROGRAM -d4 -Vb=4.5 plasma.bld blood.bld pet_t2_long.tac 20 output.res 1>stdout.txt 2>&1
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt constraints "[0,5] [1e-05,10] [0,2] [0,10] [0.045,0.045]"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 3.2.2 \n"
printf " Test case: option -Vb=0.045 by mistake\n"
printf " Expected result: Warning. \n\n"

$PROGRAM -d4 -Vb=0.045 plasma.bld blood.bld pet_t2_long.tac 20 output.res 1>stdout.txt 2>&1
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Warning "Vb was set to 0.045%"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"




printf "\n===================================================================\n"
printf "   Weights \n"
printf "===================================================================\n"



printf "\n 4.0.0 \n"
printf " Test case: by default weights are 1\n"
printf " Expected result: correct weights are printed in test mode. \n\n"

$PROGRAM -d4 -Vb=4 plasma.bld blood.bld pet_t2_long.tac 999 output.res 1>stdout.txt 2>&1
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt common_data_weights "1, 1, 1, 1, 1, 1, 1, 1, 1"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n 4.0.1 \n"
printf " Test case: Datafile contains weights.\n"
printf " Expected result: correct weights are printed in test mode. \n\n"

cp -f pet_t2_long.tac temp.tac
tacweigh -wf temp.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi

$PROGRAM -d4 -Vb=4 plasma.bld blood.bld temp.tac 999 output.res 1>stdout.txt 2>&1
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt common_data_weights "0.05, 0.05, 0.2, 0.2, 0.5, 1, 1, 3, 3"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"




printf "\n===================================================================\n"
printf "   All passed! \n"
printf "===================================================================\n"
exit 0

