#!/bin/bash
#: Title      : test_fitk4di
#: Date       : 2024-06-19
#: Author     : "Vesa Oikonen" <vesa.oikonen@utu.fi>
#: Options    : None

# Set the name of executable to test
if [ -n "${OS+1}" ] && [ "$OS" = "Windows_NT" ]; then
EXT=.exe;
else
EXT=;
fi

PROGRAM=../../fitk4di$EXT;
if [ ! -f $PROGRAM ]; then
  printf "Failed: executable does not exist.\n"
  exit 1
fi


printf "=====================================================================\n"
printf "creating test data for %s\n" $PROGRAM
printf "=====================================================================\n"


printf "\n verifying that required data exists \n"

if [ ! -f frames.dat ] || [ ! -f parent.fit ]; then
  printf "Failed: required data does not exist.\n"
  exit 1
fi


printf "\n making data for testing \n"

if [ ! -f parent.dat ] || [ ! -f parent_i.dat ] || [ ! -f plasma.dat ]; then

  printf "\n Making parent plasma TAC for simulation\n"
  rm -f metabolite.dat plasma.dat blood.dat test?.dat
  fit2dat -c=0,10,0.05 parent.fit temp1.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  fit2dat -c=10,120,0.25 parent.fit temp2.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  taccat -first temp1.dat temp2.dat parent.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacstudy --force parent.dat parent
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi

  printf "\n Making parent plasma TAC for fitting with less samples \n"
  fit2dat -c=10,20,0.25 parent.fit temp2.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  taccat -first temp1.dat temp2.dat parent_i.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  fit2dat -c=20,30,0.50 parent.fit temp2.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  taccat -first parent_i.dat temp2.dat 
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  fit2dat -c=30,60,1.0 parent.fit temp2.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  taccat -first parent_i.dat temp2.dat 
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  fit2dat -c=60,90,2.0 parent.fit temp2.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  taccat -first parent_i.dat temp2.dat 
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  fit2dat -c=90,120,5.0 parent.fit temp2.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  taccat -first parent_i.dat temp2.dat 
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacstudy --force parent_i.dat parent
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi

if [ ! -f metabolite.dat ] || [ ! -f metabolite_i.dat ]; then
  printf "\n Making delayed metabolite plasma TAC \n"
  rm -f metabolite.dat plasma.dat blood.dat test?.dat
  tactime -nogap parent.dat +1 temp1.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  p2t_3c -nosub temp1.dat 0.1 0.05 0 0 0 0 metabolite.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacstudy --force metabolite.dat metab
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  interpol -f=parent_i.dat metabolite.dat metabolite_i.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi

if [ ! -f plasma.dat ] || [ ! -f plasma_i.dat]; then
  printf "\n Making total plasma TAC \n"
  rm -f parent.rat blood.dat test?.dat
  taccalc parent.dat + metabolite.dat plasma.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacstudy --force plasma.dat plasma
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  interpol -f=parent_i.dat plasma.dat plasma_i.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi

if [ ! -f parent.rat ]; then
  printf "\n Making parent.rat \n"
  taccalc parent.dat : plasma.dat parent.rat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacstudy --force parent.rat parent
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tac2svg parentrat.svg -l parent.rat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi

if [ ! -f blood.dat ] || [ ! -f blood_i.dat ]; then
  rm -f test1.dat test2.dat
  printf "\n Simulating blood TAC, assuming that metabolite goes to RBC \n"
  p2blood norbc parent.dat 0.40 temp1.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  p2blood inwater metabolite.dat 0.40 temp2.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  taccalc temp1.dat + temp2.dat blood.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacstudy --force blood.dat blood
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  taccalc blood.dat : plasma.dat bp.rat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacstudy --force bp.rat bl-to-pl
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tac2svg -legend=yes bprat.svg -l bp.rat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi

  interpol -f=parent_i.dat blood.dat blood_i.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi

printf "\n plotting input TACs \n"
tac2svg input.svg -l plasma.dat blood.dat parent.dat metabolite.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi



if [ ! -f test1.dat ] || [ ! -f test1v.dat ]; then
  printf "\n Simulating tissue data with reversible 2TCM with and without Vb \n"
  p2t_3c -nosub parent.dat 0.4 0.8 0.3 0.1 0 0 temp1.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  p2t_3c -nosub metabolite.dat 0.6 0.6 0 0 0 0 temp2.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  taccalc temp1.dat + temp2.dat tacs.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacren tacs.dat 0 ttac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  simframe tacs.dat frames.dat test1.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacstudy --force test1.dat test1
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tac2svg -legend=yes test1.svg test1.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi

  printf "Simulate tissue data with Vb=0.05 \n"
  taccbv -sim -tv test1.dat blood.dat 0.05 test1v.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tac2svg -legend=no test1v.svg test1v.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi



if [ ! -f test2.dat ]; then
  printf "\n Simulating tissue data with irreversible 2TCM model \n"
  p2t_3c -nosub parent.dat 0.4 0.8 0.06 0.0 0 0 temp1.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  p2t_3c -nosub metabolite.dat 0.6 0.6 0 0 0 0 temp2.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  taccalc temp1.dat + temp2.dat tacs.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacren tacs.dat 0 ttac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  simframe tacs.dat frames.dat test2.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacstudy --force test2.dat test2
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tac2svg -legend=no test2.svg test2.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi

if [ ! -f test3.dat ]; then
  printf "\n Simulating tissue data with reversible 1TCM model \n"
  p2t_3c -nosub parent.dat 0.4 0.8 0.0 0.0 0 0 temp1.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  p2t_3c -nosub metabolite.dat 0.6 0.6 0 0 0 0 temp2.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  taccalc temp1.dat + temp2.dat tacs.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacren tacs.dat 0 ttac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  simframe tacs.dat frames.dat test3.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacstudy --force test3.dat test3
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tac2svg -legend=no test3.svg test3.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi


if [ ! -f test4.dat ] || [ ! -f test4v.dat ]; then
  printf "\n Simulating tissue data with reversible 2TCM with and without Vb \n"
  printf " without brain-penetrating metabolite \n"
  p2t_3c -nosub parent.dat 0.4 0.8 0.3 0.1 0 0 tacs.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacren tacs.dat 0 ttac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  simframe tacs.dat frames.dat test4.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacstudy --force test1.dat test4
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tac2svg -legend=yes test4.svg test4.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi

  printf "Simulate tissue data with Vb=0.05 \n"
  taccbv -sim -tv test4.dat blood.dat 0.05 test4v.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tac2svg -legend=no test4v.svg test4v.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi


if [ ! -f test5.dat ] || [ ! -f test5v.dat ]; then
  printf "\n Simulating tissue data with only metabolite entering the brain \n"
  p2t_3c -nosub metabolite.dat 0.6 0.6 0 0 0 0 tacs.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacren tacs.dat 0 ttac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  simframe tacs.dat frames.dat test5.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacstudy --force test1.dat test5
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tac2svg -legend=yes test5.svg test5.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi

  printf "Simulate tissue data with Vb=0.05 \n"
  taccbv -sim -tv test5.dat blood.dat 0.05 test5v.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tac2svg -legend=no test5v.svg test5v.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi



printf "\n===================================================================\n"
printf "   General usability \n"
printf "===================================================================\n"


printf "\n 1.0.0 \n"
printf " Test case: reversible 2TCM with tight limits and no Vb. \n"
printf " Expected result: Results that are reasonably close to correct ones are saved. \n\n"

printf "K1p_upper := 1\n" > test1.lim
printf "V1p_upper := 2\n" >> test1.lim
printf "k3p_upper := 1\n" >> test1.lim
printf "k4p_upper := 0.5\n" >> test1.lim
printf "K1m_lower := 0.3\n" >> test1.lim
printf "K1m_upper := 1\n" >> test1.lim
printf "V1m_lower := 1\n" >> test1.lim
printf "V1m_upper := 1\n" >> test1.lim
printf "Vb_lower := 0\n" >> test1.lim
printf "Vb_upper := 0\n" >> test1.lim

rm -f 1_0_0.svg 1_0_0.res
$PROGRAM -svg=1_0_0.svg -lim=test1.lim parent_i.dat metabolite_i.dat none test1.dat 999 1_0_0.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

parformat -f=ift 1_0_0.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi

iftisval -abs=0.05 1_0_0.ift K1p 0.4
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval -abs=0.1 1_0_0.ift K1p/k2p 0.5
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval -abs=0.1 1_0_0.ift k3p 0.3
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval -abs=0.1 1_0_0.ift k4p 0.1
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval -abs=0.05 1_0_0.ift K1m 0.6
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval -abs=0.1 1_0_0.ift K1m/k2m 1.0
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval -abs=0.0 1_0_0.ift Vb 0
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval -abs=0.2 1_0_0.ift DV 2.0
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
printf "\n passed. \n\n"



printf "\n 1.0.1 \n"
printf " Test case: reversible 2TCM with tight limits and Vb. \n"
printf " Expected result: Results that are reasonably close to correct ones are saved. \n\n"

printf "K1p_upper := 1\n" > test1v.lim
printf "V1p_upper := 2\n" >> test1v.lim
printf "k3p_upper := 1\n" >> test1v.lim
printf "k4p_upper := 0.5\n" >> test1v.lim
printf "K1m_lower := 0.3\n" >> test1v.lim
printf "K1m_upper := 1\n" >> test1v.lim
printf "V1m_lower := 1\n" >> test1v.lim
printf "V1m_upper := 1\n" >> test1v.lim
printf "Vb_lower := 0.03\n" >> test1v.lim
printf "Vb_upper := 0.07\n" >> test1v.lim

rm -f 1_0_1.svg 1_0_1.res
$PROGRAM -svg=1_0_1.svg -lim=test1v.lim parent_i.dat metabolite_i.dat blood_i.dat test1v.dat 999 1_0_1.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

parformat -f=ift 1_0_1.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi

iftisval -abs=0.05 1_0_1.ift K1p 0.4
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval -abs=0.1 1_0_1.ift K1p/k2p 0.5
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval -abs=0.1 1_0_1.ift k3p 0.3
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval -abs=0.1 1_0_1.ift k4p 0.1
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval -abs=0.05 1_0_1.ift K1m 0.6
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval -abs=0.1 1_0_1.ift K1m/k2m 1.0
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval -abs=0.5 1_0_1.ift Vb 5
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval -abs=0.2 1_0_1.ift DV 2.0
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
printf "\n passed. \n\n"


printf "\n 1.1.0 \n"
printf " Test case: irreversible 2TCM data fitted with free k4. \n"
printf " Expected result: Results that are reasonably close to correct ones are saved. \n\n"

rm -f 1_1_0.svg 1_1_0.res
$PROGRAM -svg=1_1_0.svg -lim=test1.lim parent_i.dat metabolite_i.dat none test2.dat 999 1_1_0.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

parformat -f=ift 1_1_0.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi

iftisval -abs=0.05 1_1_0.ift K1p 0.4
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval -abs=0.1 1_1_0.ift K1p/k2p 0.5
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval -abs=0.01 1_1_0.ift k3p 0.06
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval -abs=0.01 1_1_0.ift k4p 0.0
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval -abs=0.05 1_1_0.ift K1m 0.6
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval -abs=0.1 1_1_0.ift K1m/k2m 1.0
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
printf "\n passed. \n\n"



printf "\n 1.1.1 \n"
printf " Test case: irreversible 2TCM data fitted with k4 fixed to zero. \n"
printf " Expected result: Results that are reasonably close to correct ones are saved. \n\n"

printf "K1p_upper := 1\n" > test2i.lim
printf "V1p_upper := 5\n" >> test2i.lim
printf "k3p_upper := 1\n" >> test2i.lim
printf "k4p_lower := 0.0\n" >> test2i.lim
printf "k4p_upper := 0.0\n" >> test2i.lim
printf "K1m_lower := 0\n" >> test2i.lim
printf "K1m_upper := 1\n" >> test2i.lim
printf "V1m_lower := 0\n" >> test2i.lim
printf "V1m_upper := 2\n" >> test2i.lim

rm -f 1_1_1.svg 1_1_1.res
$PROGRAM -svg=1_1_1.svg -lim=test2i.lim parent_i.dat metabolite_i.dat none test2.dat 999 1_1_1.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

parformat -f=ift 1_1_1.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi

iftisval -abs=0.05 1_1_1.ift K1p 0.4
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval -abs=0.1 1_1_1.ift K1p/k2p 0.5
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval -abs=0.01 1_1_1.ift k3p 0.06
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval -abs=0.01 1_1_1.ift k4p 0.0
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval -abs=0.05 1_1_1.ift K1m 0.6
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval -abs=0.1 1_1_1.ift K1m/k2m 1.0
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval -abs=0.002 1_1_1.ift Ki 0.027907
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 1.2.0 \n"
printf " Test case: reversible 1TCM data with free k3 and k4. \n"
printf " Expected result: Results that are reasonably close to correct ones are saved. \n\n"

rm -f 1_2_0.svg 1_2_0.res
$PROGRAM -svg=1_2_0.svg -lim=test1.lim parent_i.dat metabolite_i.dat none test3.dat 999 1_2_0.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

parformat -f=ift 1_2_0.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi

iftisval -abs=0.05 1_2_0.ift K1p 0.4
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval -abs=0.05 1_2_0.ift K1m 0.6
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval -abs=0.1 1_2_0.ift K1m/k2m 1.0
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval -abs=0.0 1_2_0.ift Vb 0
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval -abs=0.1 1_2_0.ift DV 0.5
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n 1.2.1 \n"
printf " Test case: reversible 1TCM data with k3 and k4 fixed to zero. \n"
printf " Expected result: Results that are reasonably close to correct ones are saved. \n\n"


printf "K1p_upper := 1\n" > test3.lim
printf "V1p_upper := 5\n" >> test3.lim
printf "k3p_lower := 0\n" >> test3.lim
printf "k3p_upper := 0\n" >> test3.lim
printf "K1m_lower := 0\n" >> test3.lim
printf "K1m_upper := 1\n" >> test3.lim
printf "V1m_lower := 0\n" >> test3.lim
printf "V1m_upper := 2\n" >> test3.lim

rm -f 1_2_1.svg 1_2_1.res
$PROGRAM -lim=test3.lim parent_i.dat metabolite_i.dat none test3.dat 999 1_2_1.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

parformat -f=ift 1_2_1.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi

iftisval -abs=0.05 1_2_1.ift K1p 0.4
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval -abs=0.1 1_2_1.ift K1p/k2p 0.5
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval -abs=0.01 1_2_1.ift k3p 0
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval -abs=0.01 1_2_1.ift k4p 0
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval -abs=0.05 1_2_1.ift K1m 0.6
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval -abs=0.1 1_2_1.ift K1m/k2m 1.0
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval -abs=0.1 1_2_1.ift DV 0.5
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n===================================================================\n"
printf "   Contribution of the metabolite \n"
printf "===================================================================\n"


printf "\n 2.0.0 \n"
printf " Test case: reversible 2TCM when metabolite does not reach the brain. \n"
printf " Expected result: Results that are reasonably close to correct ones are saved. \n\n"

printf "K1p_upper := 1\n" > test4.lim
printf "V1p_upper := 2\n" >> test4.lim
printf "k3p_upper := 1\n" >> test4.lim
printf "k4p_upper := 0.5\n" >> test4.lim
printf "K1m_lower := 0\n" >> test4.lim
printf "K1m_upper := 1\n" >> test4.lim
printf "V1m_lower := 0\n" >> test4.lim
printf "V1m_upper := 2\n" >> test4.lim
printf "Vb_lower := 0.03\n" >> test4.lim
printf "Vb_upper := 0.07\n" >> test4.lim

rm -f 2_0_0.svg 2_0_0.res
$PROGRAM -svg=2_0_0.svg -lim=test4.lim parent_i.dat metabolite_i.dat blood_i.dat test4v.dat 999 2_0_0.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

parformat -f=ift 2_0_0.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi

iftisval -abs=0.05 2_0_0.ift K1p 0.4
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval -abs=0.1 2_0_0.ift K1p/k2p 0.5
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval -abs=0.1 2_0_0.ift k3p 0.3
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval -abs=0.1 2_0_0.ift k4p 0.1
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval -abs=0.1 2_0_0.ift Vb 5
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval -abs=0.2 2_0_0.ift DV 2.0
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
printf "\n passed. \n\n"


printf "\n 2.0.1 \n"
printf " Test case 1: same but metabolite parameters fixed to zero. \n"
printf " Expected result 1: Results that are reasonably close to correct ones are saved. \n\n"

printf "K1p_upper := 1\n" > test4f.lim
printf "V1p_upper := 2\n" >> test4f.lim
printf "k3p_upper := 1\n" >> test4f.lim
printf "k4p_upper := 0.5\n" >> test4f.lim
printf "K1m_lower := 0\n" >> test4f.lim
printf "K1m_upper := 0\n" >> test4f.lim
printf "V1m_lower := 0\n" >> test4f.lim
printf "V1m_upper := 0\n" >> test4f.lim
printf "Vb_lower := 0\n" >> test4f.lim
printf "Vb_upper := 0.1\n" >> test4f.lim

rm -f 2_0_1.svg 2_0_1.res
$PROGRAM -svg=2_0_1.svg -lim=test4f.lim parent_i.dat metabolite_i.dat blood_i.dat test4v.dat 999 2_0_1.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

parformat -f=ift 2_0_1.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi

iftisval -abs=0.05 2_0_1.ift K1p 0.4
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval -abs=0.1 2_0_1.ift K1p/k2p 0.5
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval -abs=0.1 2_0_1.ift k3p 0.3
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval -abs=0.1 2_0_1.ift k4p 0.1
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval -abs=0.01 2_0_1.ift K1m 0
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval -abs=0.01 2_0_1.ift K1m/k2m 0
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval -abs=0.1 2_0_1.ift Vb 5
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval -abs=0.2 2_0_1.ift DV 2.0
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
printf "\n passed. \n\n"



printf "\n 2.1.0 \n"
printf " Test case: full model fitted when only metabolite reaches the brain. \n"
printf " Expected result: Results that are reasonably close to correct ones are saved. \n\n"

printf "K1p_upper := 1\n" > test5.lim
printf "V1p_upper := 2\n" >> test5.lim
printf "k3p_upper := 1\n" >> test5.lim
printf "k4p_upper := 0.5\n" >> test5.lim
printf "K1m_lower := 0\n" >> test5.lim
printf "K1m_upper := 1\n" >> test5.lim
printf "V1m_lower := 0\n" >> test5.lim
printf "V1m_upper := 2\n" >> test5.lim
printf "Vb_lower := 0.03\n" >> test5.lim
printf "Vb_upper := 0.07\n" >> test5.lim

rm -f 2_1_0.svg 2_1_0.res
$PROGRAM -svg=2_1_0.svg -lim=test5.lim parent_i.dat metabolite_i.dat blood_i.dat test5v.dat 999 2_1_0.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

parformat -f=ift 2_1_0.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi

iftisval -abs=0.01 2_1_0.ift K1m 0.6
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval -abs=0.01 2_1_0.ift K1m/k2m 1
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval -abs=0.1 2_1_0.ift Vb 5
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
printf "\n passed. \n\n"




printf "\n 2.1.1 \n"
printf " Test case: parent parameters fixed to zero when only metabolite reaches the brain. \n"
printf " Expected result: Results that are reasonably close to correct ones are saved. \n\n"

printf "K1p_upper := 0\n" > test5m.lim
printf "V1p_upper := 0\n" >> test5m.lim
printf "k3p_upper := 0\n" >> test5m.lim
printf "k4p_upper := 0\n" >> test5m.lim
printf "K1m_lower := 0\n" >> test5m.lim
printf "K1m_upper := 1\n" >> test5m.lim
printf "V1m_lower := 0\n" >> test5m.lim
printf "V1m_upper := 2\n" >> test5m.lim
printf "Vb_lower := 0.03\n" >> test5m.lim
printf "Vb_upper := 0.07\n" >> test5m.lim

rm -f 2_1_1.svg 2_1_1.res
$PROGRAM -svg=2_1_1.svg -lim=test5m.lim parent_i.dat metabolite_i.dat blood_i.dat test5v.dat 999 2_1_1.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

parformat -f=ift 2_1_1.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi

iftisval -abs=0.01 2_1_1.ift K1p 0
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval -abs=0.01 2_1_1.ift K1p/k2p 0
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval -abs=0.01 2_1_1.ift K1m 0.6
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval -abs=0.01 2_1_1.ift K1m/k2m 1
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval -abs=0.1 2_1_1.ift Vb 5
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval -abs=0.1 2_1_1.ift DV 0
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 2.2.0 \n"
printf " Test case: save metabolite corrected brain data. \n"
printf " Expected result: Data reasonably close to correct ones are saved. \n\n"

rm -f 2_2_0.res 2_2_0.dat 2_2_0.svg
$PROGRAM -mc=2_2_0.dat -lim=test1v.lim parent_i.dat metabolite_i.dat blood_i.dat test1v.dat 999 2_2_0.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

tac2svg -legend=no 2_2_0.svg -s test4v.dat -l 2_2_0.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

tacmatch -abs=0.5 -rel=1 test4v.dat 2_2_0.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"




printf "\n===================================================================\n"
printf "   All passed! \n"
printf "===================================================================\n"
exit 0

