/** @file esetstrt.c
 *  @brief Set the scan_start_time in ECAT file.
 *  @copyright (c) Turku PET Centre
 *  @author Vesa Oikonen
 */
/// @cond
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <string.h>
#include <unistd.h>
#include <time.h>
/*****************************************************************************/
#include "libtpcmisc.h"
#include "libtpcimgio.h"
/*****************************************************************************/

/*****************************************************************************/
static char *info[] = {
  "Set the scan_start_time in ECAT 6 or 7 file.",
  "New scan start time can ge given directly, or as time difference",
  "(positive or negative) as minutes.",
  " ",
  "Usage: @P [Options] ecatfile YYYY-MM-DD hh:mm:ss",
  "       or",
  "       @P [Options] ecatfile timedifference",
  " ",
  "Options:",
  " -stdoptions", // List standard options like --help, -v, etc
  " ",
  "Example 1:",
  "  @P a2345dy1.img 2003-12-25 13:15:03",
  "Example 2:",
  "  @P a2345dy1.img 59.8",
  " ",
  "See also: egetstrt, lmhdr, ecattime, ecatcat, edecay, injdifft",
  " ",
  "Keywords: image, ECAT, input, tool, scan start time",
  0};
/*****************************************************************************/

/*****************************************************************************/
/* Turn on the globbing of the command line, since it is disabled by default in
   mingw-w64 (_dowildcard=0); in MinGW32 define _CRT_glob instead, if necessary;
   In Unix&Linux wildcard command line processing is enabled by default. */
/*
#undef _CRT_glob
#define _CRT_glob -1
*/
int _dowildcard = -1;
/*****************************************************************************/

/*****************************************************************************/
/**
 *  Main
 */
int main(int argc, char **argv)
{
  int                 ai, help=0, version=0, verbose=1;
  int                 ret;
  int                 fformat=63;
  char                ecatfile[FILENAME_MAX], *cptr;
  char                datestr[256], timestr[256];
  double              diftime=nan("");
  char                tmp1[32], tmp2[32];
  FILE               *fp;
  ECAT63_mainheader   e63mhdr;
  ECAT7_mainheader    e7mhdr;
  struct tm           stn;


  /*
   *  Get arguments
   */
  if(argc==1) {tpcPrintUsage(argv[0], info, stderr); return(1);}
  ecatfile[0]=datestr[0]=timestr[0]=(char)0;
  /* Options */
  for(ai=1; ai<argc; ai++) if(*argv[ai]=='-') { /* options */
    cptr=argv[ai]+1; if(*cptr=='-') cptr++; if(cptr==NULL) continue;
    if(tpcProcessStdOptions(argv[ai], &help, &version, &verbose)==0) continue;
    fprintf(stderr, "Error: invalid option '%s'.\n", argv[ai]);
    return(1);
  } else break;
  
  /* Print help or version? */
  if(help==2) {tpcHtmlUsage(argv[0], info, ""); return(0);}
  if(help) {tpcPrintUsage(argv[0], info, stdout); return(0);}
  if(version) {tpcPrintBuild(argv[0], stdout); return(0);}

  /* Process other arguments, starting from the first non-option */
  if(ai<argc) { // first argument is the filename
    strlcpy(ecatfile, argv[ai++], FILENAME_MAX);
    /* check that file exists */
    if(access(ecatfile, 0) == -1) {
      fprintf(stderr, "Error: file '%s' does not exist.\n", ecatfile);
      return(1);
    }
  }
  if(ai==argc-1) { // if this is the last argument, then it must be time diff
    ret=atof_with_check(argv[ai], &diftime);
    if(ret==0) diftime*=60.0;
    if(ret!=0 || diftime==0) {
      fprintf(stderr, "Error: invalid time difference.\n"); 
      return(1);
    }
    ai++;
  }
  if((ai+1)<argc) { // if there are two arguments, those must be date and time
    strlcpy(datestr, argv[ai++], 256);
    strlcpy(timestr, argv[ai++], 256);
    /* Check the date and time, put in international format in timestr */
    strlcat(datestr, " ", 256); strlcat(datestr, timestr, 256);
    if(isdatetime(datestr, timestr)!=0) {
      fprintf(stderr, "Error: invalid date and time.\n");
      return(1);
    }
    if(verbose>4) printf("timestr='%s'\n", timestr);
    /* Convert it to struct tm */
    if(get_datetime(timestr, &stn, verbose-2)!=0) {
      fprintf(stderr, "Error: invalid date and time.\n");
      return(1);
    }
    if(verbose>3) printf("new_hour := %d\n", stn.tm_hour);
  }
  if(ai<argc) {
    fprintf(stderr, "Error: invalid argument %s\n", argv[ai]);
    return(1);
  }

  /* Is something missing? */
  if(!timestr[0] && isnan(diftime)) {
    fprintf(stderr, "Error: missing command-line argument; try %s --help\n",
      argv[0]);
    return(1);
  }


  /* In verbose mode print arguments and options */
  if(verbose>1) {
    printf("ecatfile := %s\n", ecatfile);
    if(timestr[0]) printf("date_and_time := %s\n", timestr);
    if(!isnan(diftime)) printf("diftime[s] := %g\n", diftime);
  }
  

  /*
   *  Read main header
   */
  if(verbose>1) printf("opening file %s\n", ecatfile);

  /* Open file for read and write */
  if((fp=fopen(ecatfile, "r+b")) == NULL) {
    fprintf(stderr, "Error: cannot open file %s\n", ecatfile);
    return(2);
  }

  /* Try to read ECAT 7.x main header */
  ret=ecat7ReadMainheader(fp, &e7mhdr);
  if(ret) {
    fprintf(stderr, "Error (%d): cannot read main header.\n", ret);
    fclose(fp); return(3);
  }

  /* If header could be read, check for magic number */
  if(strncmp(e7mhdr.magic_number, ECAT7V_MAGICNR, 7)==0) {
    /* This is ECAT 7.x file */
    fformat=7;
    /* All file types are supported */
    if(verbose>4) printf("  identified as an ECAT 7 file.\n");
    if(verbose>9) ecat7PrintMainheader(&e7mhdr, stdout);
  } else { /* Try to read as ECAT 6.3 file */
    if((ret=ecat63ReadMainheader(fp, &e63mhdr))) {
      fprintf(stderr, "Error (%d): cannot read main header.\n", ret);
      fclose(fp); return(3);
    }
    if(verbose>9) ecat63PrintMainheader(&e63mhdr, stdout);
    /* This is ECAT 6.3 file */
    fformat=63;
    if(verbose>4) printf("  identified as an ECAT 6.3 file.\n");
  }


  /*
   *  Set scan_start_time and rewrite the main header
   */
  if(verbose>1) printf("setting scan start time\n");
  if(fformat==7) {
  
    /* ECAT 7 main header contains scan_start_time as time_t */

    if(isnan(diftime)) { // new scan start time was given by user

      /* Get the original scan_start_time */
      time_t t;
      t=e7mhdr.scan_start_time;
      if(!ctime_r_int(&t, tmp1)) strcpy(tmp1, "1900-01-01 00:00:00");
      if(verbose>3) printf("tmp1 := '%s'\n", tmp1);

      /* Calculate new scan_start_time as time_t */
      if(strftime(tmp2, 32, "%Y-%m-%d %H:%M:%S", &stn)==0)
        strcpy(tmp2, "1900-01-01 00:00:00");
      if(verbose>3) printf("tmp2 := '%s'\n", tmp2);

      /* Put it in the header */
      e7mhdr.scan_start_time=timegm(&stn);

    } else { // time difference was given by user

      /* Put it in the header */
      e7mhdr.scan_start_time+=(int)diftime;
      if(verbose>3) {
        time_t t;
        t=e7mhdr.scan_start_time;
        if(!ctime_r_int(&t, tmp2)) strcpy(tmp2, "1900-01-01 00:00:00");
        if(verbose>3) printf("tmp2 := '%s'\n", tmp2);
      }

    }

    /* Write */
    if(verbose>1) fprintf(stdout, "writing main header\n");
    ret=ecat7WriteMainheader(fp, &e7mhdr);

  } else {
  
    /* ECAT 6 main header contains years, months etc as separate numbers */
    int YYYY, MM, DD, hh, mm, ss;

    /* Get the original scan_start_time as string */
    ecat63ScanstarttimeInt(&e63mhdr, tmp1);
    if(verbose>3) printf("tmp1 := '%s'\n", tmp1);

    if(isnan(diftime)) { // new scan start time was given by user

      /* Get the new scan_start_time as years, months etc */
      strlcpy(tmp2, timestr, 32);
      ret=sscanf(tmp2, "%d-%d-%d %d:%d:%d", &YYYY, &MM, &DD, &hh, &mm, &ss);
      if(ret!=6) {
        fprintf(stderr, "Error: invalid date and time.\n");
        fclose(fp); return(3);
      }

    } else { // time difference was given by user

      ret=get_datetime(tmp1, &stn, 0);
      if(ret==0) tmAdd((int)diftime, &stn);
      if(ret==0) {
        if(strftime(tmp2, 32, "%Y-%m-%d %H:%M:%S", &stn)==0) {
          strcpy(tmp2, "1900-01-01 00:00:00");
          ret=1;
        }
        if(verbose>3) printf("tmp2 := '%s'\n", tmp2);
      }
      if(ret==0) {
        if(sscanf(tmp2, "%d-%d-%d %d:%d:%d", &YYYY, &MM, &DD, &hh, &mm, &ss)!=6)
          ret=1;
      }
      if(ret) {
        fprintf(stderr, "Error: new scan start time is not valid.\n");
        fclose(fp); return(5);
      }

    }

    /* Put it into header */
    e63mhdr.scan_start_year=YYYY;
    e63mhdr.scan_start_month=MM;
    e63mhdr.scan_start_day=DD;
    e63mhdr.scan_start_hour=hh;
    e63mhdr.scan_start_minute=mm;
    e63mhdr.scan_start_second=ss;

    /* Write */
    if(verbose>1) fprintf(stdout, "writing main header\n");
    ret=ecat63WriteMainheader(fp, &e63mhdr);

  }
  
  if(verbose>1) printf("closing file\n");  
  fclose(fp);

  if(ret) {
    fprintf(stderr, "Error: cannot write the mainheader.\n");
    return(14);
  }
    
  if(verbose>0) fprintf(stdout, "Scan start: %s -> %s\n", tmp1, tmp2);
  if(verbose>1) fprintf(stdout, "done.\n");

  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/// @endcond
