#!/bin/bash
#: Title      : test_imgflow
#: Date       : 2023-03-26
#: Author     : "Vesa Oikonen" <vesa.oikonen@utu.fi>
#: Options    : None

# Set the name of executable to test
if [ -n "${OS+1}" ] && [ "$OS" = "Windows_NT" ]; then
EXT=.exe;
else
EXT=;
fi

PROGRAM=../../imgflowm$EXT;
if [ ! -f $PROGRAM ]; then
  printf "Failed: executable does not exist.\n"
  exit 1
fi

printf "=====================================================================\n"
printf "creating test data for %s\n" $PROGRAM
printf "=====================================================================\n"


printf "\n verifying that required data exists \n"


if [ ! -f blood.dat ] || [ ! -f frames.sif ]; then
  printf "Failed: required data does not exist.\n"
  exit 1
fi

if [ ! -f blood_delayed.dat ]; then
  rm -f dynamic.nii
  tactime -nogap blood.dat +10 temp.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  interpol -f=blood.dat temp.dat blood_delayed.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi

if [ ! -f dynamic.nii ] || [ ! -f correct_f.nii ]; then
  printf "\n Simulate dynamic 3D image with Vb=3 \n"
  b2t_h2o --silent      -voiname="f10" blood.dat  10 0.8 1 3 100 temp.dft
  b2t_h2o --silent -add -voiname="f20" blood.dat  20 0.8 1 3 100 temp.dft
  b2t_h2o --silent -add -voiname="f50" blood.dat  50 0.8 1 3 100 temp.dft
  b2t_h2o --silent -add -voiname="f100" blood.dat  100 0.8 1 3 100 temp.dft
  b2t_h2o --silent -add -voiname="f1" blood_delayed.dat  8 0.8 1 3 100 temp.dft
  b2t_h2o --silent -add -voiname="f2" blood_delayed.dat  15 0.8 1 3 100 temp.dft
  b2t_h2o --silent -add -voiname="f4" blood_delayed.dat  25 0.8 1 3 100 temp.dft
  b2t_h2o --silent -add -voiname="f8" blood_delayed.dat  60 0.8 1 3 100 temp.dft
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  simframe --silent temp.dft frames.sif dynamic.tac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacstudy --silent -force dynamic.tac sim1 
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tac2svg -legend=yes -mt="Dynamic" -x2=360 dynamic.svg -l blood.dat dynamic.tac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tac2nii dynamic.tac 10 10 10 dynamic.nii voimask.nii
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  printf "\n ok \n"

  printf "\n Make image with correct flow values \n"
  printf "0 0.1 0.2 0.5 1 0.08 0.15 0.25 0.60\n" > temp.dft
  printf "# timeunit := sec\n" >> temp.dft
  printf "# unit := mL/(mL*min)\n" >> temp.dft
  tac2nii temp.dft 10 10 10 correct_f.nii
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  printf "\n ok \n"
fi

if [ ! -f correct_va.nii ]; then
  printf "\n Make image with correct Va values \n"
  printf "0 0.03 0.03 0.03 0.03 0.03 0.03 0.03 0.03\n" > temp.dft
  printf "# timeunit := sec\n" >> temp.dft
  printf "# unit := mL/mL\n" >> temp.dft
  tac2nii temp.dft 10 10 10 correct_va.nii
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi

if [ ! -f correct_k2.nii ]; then
  printf "\n Make images with correct k2 values \n"
  printf "0 0.125 0.25 0.625 1.25 0.1 0.1875 0.3125 0.75\n" > temp.dft
  tac2nii temp.dft 10 10 10 temp.nii
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
# TACs were simulated with model Cpet = VbCb + (1-Vb)Ct
# but program to be tested currently uses model Cpet = VbCb + Ct
# Therefore correct k2 output is
  imgcalc temp.nii div 0.97 correct_k2.nii
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  printf "\n ok \n"
fi

if [ ! -f wholemask.nii ]; then
  printf "\n Make mask that covers all VOIs \n"
  printf "0 1 1 1 1 1 1 1 1\n" > temp.dft
  tac2nii temp.dft 10 10 10 wholemask.nii
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  printf "\n ok \n"
fi


if [ ! -f d1mask.nii ] || [ ! -f d2mask.nii ]; then
  printf "\n Make masks for the regions with different delays \n"
  printf "0 1 1 1 1 0 0 0 0\n" > temp.dft
  tac2nii temp.dft 10 10 10 d1mask.nii
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  printf "0 0 0 0 0 1 1 1 1\n" > temp.dft
  tac2nii temp.dft 10 10 10 d2mask.nii
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  printf "\n ok \n"
fi

if [ ! -f correct_f_m1.nii ] || [ ! -f correct_va_m1.nii ] || [ ! -f correct_k2_m1.nii ]; then
  printf "\n Make maps with correct values inside mask 1 and zeroes elsewhere\n"
  printf "0 0.1 0.2 0.5 1.0 0 0 0 0\n" > temp.dft
  tac2nii temp.dft 10 10 10 correct_f_m1.nii
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi

  printf "0 0.03 0.03 0.03 0.03 0 0 0 0\n" > temp.dft
  tac2nii temp.dft 10 10 10 correct_va_m1.nii
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi

  printf "0 0.125 0.25 0.625 1.25 0 0 0 0\n" > temp.dft
  tac2nii temp.dft 10 10 10 temp.nii
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  imgcalc temp.nii div 0.97 correct_k2_m1.nii
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  printf "\n ok \n"
fi


if [ ! -f dynamic_cbv.nii ]; then
  printf "\n Simulate dynamic 3D image with Vb=0 \n"
  b2t_h2o --silent      -voiname="f10" blood.dat  10 0.8 1 0 100 temp.dft
  b2t_h2o --silent -add -voiname="f20" blood.dat  20 0.8 1 0 100 temp.dft
  b2t_h2o --silent -add -voiname="f50" blood.dat  50 0.8 1 0 100 temp.dft
  b2t_h2o --silent -add -voiname="f100" blood.dat  100 0.8 1 0 100 temp.dft
  b2t_h2o --silent -add -voiname="f1" blood_delayed.dat  8 0.8 1 0 100 temp.dft
  b2t_h2o --silent -add -voiname="f2" blood_delayed.dat  15 0.8 1 0 100 temp.dft
  b2t_h2o --silent -add -voiname="f4" blood_delayed.dat  25 0.8 1 0 100 temp.dft
  b2t_h2o --silent -add -voiname="f8" blood_delayed.dat  60 0.8 1 0 100 temp.dft
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  simframe --silent temp.dft frames.sif dynamic_cbv.tac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacstudy --silent -force dynamic_cbv.tac sim2 
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tac2nii dynamic_cbv.tac 10 10 10 dynamic_cbv.nii voimask.nii
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  printf "\n ok \n"
fi


printf "\n\n"
printf "=====================================================================\n"
printf "testing %s\n" $PROGRAM
printf "=====================================================================\n"

printf "\n 1.0.0 \n"
printf " Test case 1: Fit model to all image regions. \n"
printf " Test case 2: Save optional result images. \n"
printf " Expected result 1: No errors; results can be bad because of different delays. \n"
printf " Expected result 2: Optional images created or overwritten. \n\n"

rm -f flow.nii va.nii k2.nii

$PROGRAM -va=va.nii -k2=k2.nii blood.dat dynamic.nii wholemask.nii flow.nii
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
if [ ! -f flow.nii ]; then printf "Failed!\n" ; exit 1 ; fi
if [ ! -f va.nii ]; then printf "Failed!\n" ; exit 1 ; fi
if [ ! -f k2.nii ]; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

# Now images pre-exist and will be rewritten
$PROGRAM -va=va.nii -k2=k2.nii blood.dat dynamic.nii wholemask.nii flow.nii
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
if [ ! -f flow.nii ]; then printf "Failed!\n" ; exit 1 ; fi
if [ ! -f va.nii ]; then printf "Failed!\n" ; exit 1 ; fi
if [ ! -f k2.nii ]; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

img2dft flow.nii voimask.nii flow.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
img2dft va.nii voimask.nii va.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
img2dft k2.nii voimask.nii k2.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

taclist va.tac
taclist k2.tac
taclist flow.tac


printf "\n 1.1.0 \n"
printf " Test case 1: Fit model to part of image. \n"
printf " Test case 2: Save optional result images. \n"
printf " Expected result: Correct results inside the mask, zero elsewhere. \n\n"

rm -f flow.nii va.nii k2.nii

$PROGRAM -va=va.nii -k2=k2.nii blood.dat dynamic.nii d1mask.nii flow.nii
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

img2dft flow.nii voimask.nii flow.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
img2dft va.nii voimask.nii va.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
img2dft k2.nii voimask.nii k2.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

printf "\nVa\n"
taclist va.tac
printf "\nk2\n"
taclist k2.tac
printf "\nFlow\n"
taclist flow.tac
printf "\n ok \n"


imgmatch -abs=0.001 -rel=1 -frames=n correct_f_m1.nii flow.nii
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
imgmatch -abs=0.001 -rel=1 -frames=n correct_k2_m1.nii k2.nii
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
imgmatch -abs=0.001 -rel=1 -frames=n correct_va_m1.nii va.nii
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n 1.1.1 \n"
printf " Test case 1: Fit model to the other part of image. \n"
printf " Test case 2: Save optional result images. \n"
printf " Expected result: Correct results inside the mask, conserved elsewhere. \n\n"

# Previously made parametric maps are NOT removed

$PROGRAM -va=va.nii -k2=k2.nii blood_delayed.dat dynamic.nii d2mask.nii flow.nii
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

img2dft flow.nii voimask.nii flow.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
img2dft va.nii voimask.nii va.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
img2dft k2.nii voimask.nii k2.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

printf "\nVa\n"
taclist va.tac
printf "\nk2\n"
taclist k2.tac
printf "\nFlow\n"
taclist flow.tac
printf "\n ok \n"

imgmatch -abs=0.001 -rel=1 -frames=n correct_f.nii flow.nii
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
imgmatch -abs=0.001 -rel=1 -frames=n correct_k2.nii k2.nii
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
imgmatch -abs=0.001 -rel=1 -frames=n correct_va.nii va.nii
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n 1.2.0 \n"
printf " Test case: Option -Va=0 with image pre-corrected for Va. \n"
printf " Expected result: Correct flow results. \n\n"

rm -f flow2.nii

$PROGRAM blood.dat dynamic_cbv.nii d1mask.nii flow2.nii
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
$PROGRAM blood_delayed.dat dynamic_cbv.nii d2mask.nii flow2.nii
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

img2dft flow2.nii voimask.nii flow2.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
taclist flow2.tac
printf "\n ok \n"

imgmatch -abs=0.001 -rel=1 -frames=n correct_f.nii flow2.nii
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"




printf "\n===================================================================\n"
printf "   All passed! \n"
printf "===================================================================\n"
exit 0

