#!/bin/bash
#: Title      : test_imgmtrap
#: Date       : 2023-03-15
#: Author     : "Vesa Oikonen" <vesa.oikonen@utu.fi>
#: Options    : None

# Set the name of executable to test
if [ -n "${OS+1}" ] && [ "$OS" = "Windows_NT" ]; then
EXT=.exe;
else
EXT=;
fi

PROGRAM=../../imgmtrap$EXT;
if [ ! -f $PROGRAM ]; then
  printf "Failed: executable does not exist.\n"
  exit 1
fi

printf "=====================================================================\n"
printf "creating test data for %s\n" $PROGRAM
printf "=====================================================================\n"


printf "\n verifying that required data exists \n"

if [ ! -f plasma.fit ] || [ ! -f frames.sif ] || [ ! -f p2rbc.par ]; then
  printf "Failed: required data does not exist.\n"
  exit 1
fi
if [ ! -f cpet.par ] || [ ! -f true_k1.tac ] || [ ! -f true_vb.tac ]; then
  printf "Failed: required data does not exist.\n"
  exit 1
fi



if [ ! -f plasma.bld ]; then
  rm -f blood.bld cpet.tac
  fit2dat -c=0,10,0.02 plasma.fit plasma.bld
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi

if [ ! -f plasma_1-HCT.bld ] || [ ! -f rbc_HCT.bld ] || [ ! -f blood.bld ]; then
  rm -f cpet.tac
# Making input function as (1-HCT)*Cp where HCT=0.39
  taccalc plasma.bld x 0.61 plasma_1-HCT.bld
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
# Simulating RBC concentration curve dCrbc=kc*(1-HCT)*Cp
# where kc=0.60 and thus kc*HCT=0.234
  sim_3tcm p2rbc.par plasma_1-HCT.bld plasma_1-HCT.bld rbc.bld
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
# Calculating the blood radioactivity residing in RBCs
  taccalc rbc.bld x 0.39 rbc_HCT.bld
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
# Calculating the blood curve as Cb=(1-HCT)*Cp+HCT*Crbc
  taccalc plasma_1-HCT.bld + rbc_HCT.bld blood.bld
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
# Setting file header information for plotting
  tacren blood.bld 1 Blood
  tacren plasma.bld 1 Plasma
  tacren plasma_1-HCT.bld 1 Plasma
  tacren rbc.bld 1 RBC
  tacren rbc_HCT.bld 1 RBC
  tacstudy --force blood.bld sim1
  tacstudy --force plasma.bld sim1
  tacstudy --force plasma_1-HCT.bld sim1
  tacstudy --force rbc.bld sim1
  tacstudy --force rbc_HCT.bld sim1
# Plotting blood components
  tac2svg -legend=y -yt="SUV" -mt="" -x2=10.2 blood.svg -l blood.bld plasma_1-HCT.bld rbc_HCT.bld
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
# Plotting plasma and blood TACs
  tac2svg -legend=y -yt="SUV" -mt="" -x2=10.2 plasma.svg -l blood.bld plasma.bld
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi

if [ ! -f cpet.tac ]; then
# Simulating regional muscle concentration curves (Cpet)
  sim_3tcm cpet.par plasma_1-HCT.bld blood.bld cpet.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacstudy --force cpet.dat sim1
  simframe cpet.dat frames.sif cpet.tac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tac2svg -legend=n cpet.svg -s cpet.tac -l cpet.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi

if [ ! -f dynamic.nii ] || [ ! -f mask.nii ]; then
  tac2nii cpet.tac 16 16 16 dynamic.nii mask.nii
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi

if [ ! -f ct.tac ] || [ ! -f ct.nii ]; then
# Simulating regional muscle concentration curves (1-Vb)*Cpet
  taccalc blood.bld x 0 zero.bld
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  sim_3tcm cpet.par plasma_1-HCT.bld zero.bld ct.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacstudy --force ct.dat sim1
  simframe ct.dat frames.sif ct.tac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tac2svg -legend=n ct.svg -s ct.tac -l ct.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tac2nii ct.tac 16 16 16 ct.nii
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi



printf "\n\n"
printf "=====================================================================\n"
printf "testing %s\n" $PROGRAM
printf "=====================================================================\n"



printf "\n 1.0.0 \n"
printf " Test case: Estimate K1 and Vb using PTAC and BTAC. \n"
printf " Expected result: Correct parametric images are calculated. \n\n"

rm -f k1.nii vb.nii

$PROGRAM -ptac=plasma_1-HCT.bld -thr=0 -p2=p2.nii blood.bld dynamic.nii k1.nii vb.nii
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
img2tif -s k1.nii k1_ptac.tif
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
img2tif -s vb.nii vb_ptac.tif
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
img2tif -s p2.nii p2_ptac.nii
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

img2dft vb.nii mask.nii vb.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
img2dft k1.nii mask.nii k1.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
img2dft p2.nii mask.nii p2.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

printf "\n--- Results for p2: \n"
taclist p2.tac
printf "\n--- Results for Vb: \n"
taclist vb.tac
printf "\n--- Results for K1: \n"
taclist k1.tac

tacmatch -abs=0.001 -rel=0.5 true_k1.tac k1.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.005 -rel=1 true_vb.tac vb.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n 1.1.0 \n"
printf " Test case: Save Vb-corrected image while PTAC is used as input. \n"
printf " Expected result: Correct Vb-corrected image is saved. \n\n"

rm -f cbv.nii

$PROGRAM -thr=0 -cbv=cbv.nii -ptac=plasma_1-HCT.bld blood.bld dynamic.nii k1.nii
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
img2tif -s cbv.nii cbv_ptac.tif
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
img2dft cbv.nii mask.nii cbv_ptac.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tac2svg -legend=y cbv_ptac.svg -s ct.tac -l cbv_ptac.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

imgmatch -abs=0.15 -rel=1 ct.nii cbv.nii
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"




printf "\n 2.0.0 \n"
printf " Test case: Estimate K1 and Vb using only BTAC. \n"
printf " Expected result: Correct parametric images are calculated. \n\n"

rm -f k1.nii vb.nii

$PROGRAM -thr=0 -p2=p2.nii -p3=p3.nii blood.bld dynamic.nii k1.nii vb.nii
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
img2tif -s k1.nii k1_btac.tif
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
img2tif -s vb.nii vb_btac.tif
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
img2tif -s p2.nii p2_btac.tif
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
img2tif -s p3.nii p3_btac.tif
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

img2dft k1.nii mask.nii k1.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
img2dft vb.nii mask.nii vb.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
img2dft p2.nii mask.nii p2.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
img2dft p2.nii mask.nii p2.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

printf "\n--- Results for p2: \n"
taclist p2.tac
printf "\n--- Results for p3: \n"
taclist p3.tac
printf "\n--- Results for Vb: \n"
taclist vb.tac
printf "\n--- Results for K1: \n"
taclist k1.tac

tacmatch -abs=0.03 -rel=0.1 true_k1.tac k1.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.01 -rel=0.1 true_vb.tac vb.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n 2.1.0 \n"
printf " Test case: Save Vb-corrected image while BTAC alone is used as input. \n"
printf " Expected result: Correct Vb-corrected image is saved. \n\n"

rm -f cbv.nii

$PROGRAM -thr=0 -cbv=cbv.nii blood.bld dynamic.nii k1.nii
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
img2tif -s cbv.nii cbv_btac.tif
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
img2dft cbv.nii mask.nii cbv_btac.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tac2svg -legend=y cbv_btac.svg -s ct.tac -l cbv_btac.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

imgmatch -abs=0.15 -rel=1 ct.nii cbv.nii
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n 3.0.0 \n"
printf " Test case: Estimate K1 and Vb using only BTAC, using median k2. \n"
printf " Expected result: Correct parametric images are calculated. \n\n"

rm -f k1.nii vb.nii

$PROGRAM -thr=0 -k2=median -p2=p2.nii -p3=p3.nii blood.bld dynamic.nii k1.nii vb.nii
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
img2tif -s k1.nii k1_median.tif
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
img2tif -s vb.nii vb_median.tif
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
img2tif -s p2.nii p2_median.tif
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
img2tif -s p3.nii p3_median.tif
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

img2dft k1.nii mask.nii k1.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
img2dft vb.nii mask.nii vb.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
img2dft p2.nii mask.nii p2.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
img2dft p3.nii mask.nii p3.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

printf "\n--- Results for p2: \n"
taclist p2.tac
printf "\n--- Results for p3: \n"
taclist p3.tac
printf "\n--- Results for Vb: \n"
taclist vb.tac
printf "\n--- Results for K1: \n"
taclist k1.tac

tacmatch -abs=0.03 -rel=0.1 true_k1.tac k1.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.01 -rel=0.1 true_vb.tac vb.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n 4.0.0 \n"
printf " Test case: Fix k2 to known value. \n"
printf " Expected result: Correct parametric images are calculated. \n\n"

rm -f k1.nii vb.nii

$PROGRAM -thr=0 -p2=p2.nii -k2=0.234 blood.bld dynamic.nii k1.nii vb.nii
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
img2tif -s k1.nii k1_fixed.tif
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
img2tif -s vb.nii vb_fixed.tif
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
img2tif -s p2.nii p2_fixed.tif
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

img2dft k1.nii mask.nii k1.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
img2dft vb.nii mask.nii vb.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
img2dft p2.nii mask.nii p2.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

printf "\n--- Results for p2: \n"
taclist p2.tac
printf "\n--- Results for Vb: \n"
taclist vb.tac
printf "\n--- Results for K1: \n"
taclist k1.tac

tacmatch -abs=0.03 -rel=0.1 true_k1.tac k1.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.01 -rel=0.1 true_vb.tac vb.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 4.1.0 \n"
printf " Test case: Save Vb-corrected image while k2 is fixed to known value. \n"
printf " Expected result: Correct Vb-corrected image is saved. \n\n"

rm -f cbv.nii

$PROGRAM -thr=0 -cbv=cbv.nii -k2=0.234 blood.bld dynamic.nii k1.nii
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
img2tif -s cbv.nii cbv_fixed.tif
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
img2dft cbv.nii mask.nii cbv_fixed.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tac2svg -legend=y cbv_fixed.svg -s ct.tac -l cbv_fixed.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

imgmatch -abs=0.1 -rel=1 ct.nii cbv.nii
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n===================================================================\n"
printf "   All passed! \n"
printf "===================================================================\n"
exit 0



