#!/bin/bash
#: Title      : test_imgsrtm
#: Date       : 2023-06-29
#: Author     : "Vesa Oikonen" <vesa.oikonen@utu.fi>
#: Options    : None

# Set the name of executable to test
if [ -n "${OS+1}" ] && [ "$OS" = "Windows_NT" ]; then
EXT=.exe;
else
EXT=;
fi

PROGRAM=../../imgsrtm$EXT;
if [ ! -f $PROGRAM ]; then
  printf "Failed: executable does not exist.\n"
  exit 1
fi


printf "=====================================================================\n"
printf "creating test data for %s\n" $PROGRAM
printf "=====================================================================\n"


printf "\n verifying that required data exists \n"



if [ ! -f plasma.dat ] || [ ! -f frames.dat ] || [ ! -f shortframes.dat ]; then
  printf "Failed: required data does not exist.\n"
  exit 1
fi
if [ ! -f perfect.par ] || [ ! -f correct_bp.dft ]; then
  printf "Failed: required data does not exist.\n"
  exit 1
fi
if [ ! -f correct_r1.tac ] || [ ! -f correct_k2.tac ]; then
  printf "Failed: required data does not exist.\n"
  exit 1
fi


if [ ! -f perfect.v ] || [ ! -f dynamic.v ]; then
  rm -f dynamic.img ana\dynamic.* nii\dynamic.*
  rm -f perfect.sif dynamic.sif
  printf "\n Simulate dynamic image applying SRTM\n"
  p2t_3c --silent -nosub plasma.dat  0.100 0.120 0 0 0 0 ref.tac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacren ref.tac 1 ref
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi

  simframe ref.tac shortframes.dat refperfect.tac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  simframe ref.tac frames.dat refdynamic.tac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi

  sim_rtcm perfect.par ref.tac perf.tac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi

  simframe perf.tac shortframes.dat perfect.tac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tac2svg -legend=yes perfect.svg perfect.tac -l refperfect.tac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  dft2img -scanner=hrrt -dim=9 -zoom=2.0 perfect.tac perfect.v test.roi
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  imgdecay -i=C-11 -dc perfect.v
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi

  simframe perf.tac frames.dat dynamic.tac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  dft2img -scanner=hrrt -dim=9 -zoom=2.0 dynamic.tac dynamic.v test.roi
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  imgdecay -i=C-11 -dc dynamic.v
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  printf "\n ok \n"
fi

if [ ! -f dynamic.img ]; then
  e7to63 -c=o dynamic.v dynamic.img
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi

if [ ! -f ana\dynamic.img ] || [ ! -f ana\dynamic.hdr ] || [ ! -f ana\dynamic.sif ]; then
  ecat2ana -sif -o=ana dynamic.v
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi

if [ ! -f nii\dynamic.nii ] || [ ! -f nii\dynamic.sif ]; then
  ecat2nii -sif -o=nii dynamic.v
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi

if [ ! -f perfect.sif ] || [ ! -f dynamic.sif ]; then
  imgweigh perfect.v perfect.sif
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  imgweigh dynamic.v dynamic.sif
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi



if [ ! -f srtm2_noiseless.v ] || [ ! -f srtm2_noisy.v ] || [ ! -f srtm2ref.tac ]; then
  printf "\n Simulating data for testing SRTM2 \n"
  p2t_3c -nosub plasma.dat 0.200 0.400 0 0 0 0 ref1.tac
  p2t_3c -nosub plasma.dat 0.300 0.600 0 0 0 0 ref2.tac
  p2t_3c -nosub plasma.dat 0.250 0.400 0 0 0 0 tis1.tac
  p2t_3c -nosub plasma.dat 0.200 0.250 0 0 0 0 tis2.tac
  p2t_3c -nosub plasma.dat 0.250 0.250 0 0 0 0 tis3.tac
  p2t_3c -nosub plasma.dat 0.250 0.125 0 0 0 0 tis4.tac
  p2t_3c -nosub plasma.dat 0.300 0.100 0 0 0 0 tis5.tac
  p2t_3c -nosub plasma.dat 0.300 0.060 0 0 0 0 tis6.tac
  p2t_3c -nosub plasma.dat 0.300 0.050 0 0 0 0 tis7.tac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacadd -ovr all.tac ref1.tac 1
  tacadd all.tac ref2.tac 1
  tacadd all.tac tis1.tac 1
  tacadd all.tac tis2.tac 1
  tacadd all.tac tis3.tac 1
  tacadd all.tac tis4.tac 1
  tacadd all.tac tis5.tac 1
  tacadd all.tac tis6.tac 1
  tacadd all.tac tis7.tac 1
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  simframe all.tac frames.dat srtm2.tac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  dft2img -scanner=hrrt -dim=9 -zoom=2.0 srtm2.tac srtm2_noiseless.v srtm2.roi
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  imgdecay -i=C-11 -dc srtm2_noiseless.v
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  fvar4img -minsd=0.001 srtm2_noiseless.v 0.002 srtm2_noisy.v
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  img2tif srtm2_noisy.v
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacadd -ovr srtm2ref.tac srtm2.tac 1
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  printf "\n ok \n"
fi
if [ ! -f correct_bp_srtm2.v ]; then
  printf "\n Make image with correct BPnd values \n"
  printf "60 0.0 0.0 0.25 0.6 1 3 5 9 11\n" > temp.dft
  printf "# timeunit := min\n" >> temp.dft
  dft2img -Scanner=HRRT -dim=9 -zoom=2.0 temp.dft correct_bp_srtm2.v
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  printf "\n ok \n"
fi
if [ ! -f correct_k2s_srtm2.v ]; then
  printf "\n Make image with correct k2' values \n"
  printf "60 0 0 0.4 0.4 0.4 0.4 0.4 0.4 0.4\n" > temp.dft
  printf "# timeunit := min\n" >> temp.dft
  printf "# unit := 1/min\n" >> temp.dft
  dft2img -Scanner=HRRT -dim=9 -zoom=2.0 temp.dft correct_k2s_srtm2.v
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  printf "\n ok \n"
fi


printf "\n\n"
printf "=====================================================================\n"
printf "testing %s\n" $PROGRAM
printf "=====================================================================\n"

printf "\n 1.0.0 \n"
printf " Test case 1: Noiseless ECAT7 image data simulated with SRTM. \n"
printf " Test case 2: No weighting. \n"
printf " Expected result: Correct results calculated. \n\n"

rm -f output_bp.v output_bp.dft

$PROGRAM -thr=0 perfect.v refperfect.tac output_bp.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
img2dft output_bp.v test.roi output_bp.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -x=n -y=y -yunit=y -abs=0.001 -rel=1 correct_bp.dft output_bp.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 1.1.0 \n"
printf " Test case 1: Noiseless ECAT6 image data simulated with SRTM. \n"
printf " Test case 2: Weights from image using option -w. \n"
printf " Expected result: Correct results calculated. \n\n"

rm -f output_bp.img output_bp.dft

$PROGRAM -w dynamic.img refdynamic.tac output_bp.img
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
img2dft output_bp.img test.roi output_bp.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -x=n -y=y -yunit=y -abs=0.001 -rel=1 correct_bp.dft output_bp.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n-------------------------------------------------------------------\n"



printf "\n 1.2.0 \n"
printf " Test case 1: Analyze image. \n"
printf " Test case 2: Default SIF available but no weights. \n"
printf " Expected result: Correct results calculated. \n\n"

rm -f ana/output.sif ana/output.hdr ana/output.img output_bp.dft

$PROGRAM -thr=0 ana/dynamic refdynamic.tac ana/output
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
img2dft ana/output test.roi output_bp.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -x=n -y=y -yunit=n -abs=0.001 -rel=1 correct_bp.dft output_bp.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n 1.2.3 \n"
printf " Test case 1: Analyze image. \n"
printf " Test case 2: No SIF available. \n"
printf " Expected result: Error. \n\n"

mv ana/dynamic.sif ana/dynamic.siiif
rm -f stdout.txt
rm -f ana/output.sif ana/output.hdr ana/output.img output_bp.dft

$PROGRAM -thr=0 ana/dynamic refdynamic.tac ana/output 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
mv ana/dynamic.siiif ana/dynamic.sif
iftisval stdout.txt "Error" "image frame times not available."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n-------------------------------------------------------------------\n"

printf "\n 1.3.0 \n"
printf " Test case 1: NIfTI image. \n"
printf " Test case 2: Default SIF available but no weights. \n"
printf " Expected result: Correct results calculated. \n\n"

rm -f nii/output.sif nii/output.nii output_bp.dft

$PROGRAM -thr=0 nii/dynamic refdynamic.tac nii/output
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
img2dft nii/output test.roi output_bp.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -x=n -y=y -yunit=n -abs=0.001 -rel=1 correct_bp.dft output_bp.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 1.2.3 \n"
printf " Test case 1: NIfTI image. \n"
printf " Test case 2: No SIF available. \n"
printf " Expected result: Error. \n\n"

mv nii/dynamic.sif nii/dynamic.siiif
rm -f stdout.txt
rm -f nii/output.sif nii/output.nii output_bp.dft

$PROGRAM -thr=0 nii/dynamic refdynamic.tac nii/output 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
mv nii/dynamic.siiif nii/dynamic.sif
iftisval stdout.txt "Error" "image frame times not available."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n-------------------------------------------------------------------\n"


printf "\n 2.0.0 \n"
printf " Test case 1: Option -end with large time. \n"
printf " Expected result: Correct results calculated. \n\n"

rm -f output_bp.v output_bp.dft

$PROGRAM -end=999 -d3 -thr=0 dynamic.v refdynamic.tac output_bp.v > stdout.txt
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt fittimeFinal "120 min"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
img2dft output_bp.v test.roi output_bp.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -x=n -y=y -yunit=y -abs=0.001 -rel=1 correct_bp.dft output_bp.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n 2.0.1 \n"
printf " Test case 1: Option -end with a bit shorter time than in the data. \n"
printf " Expected result: Correct results calculated. \n\n"

rm -f output_bp.v output_bp.dft

$PROGRAM -end=100 -d3 -thr=0 dynamic.v refdynamic.tac output_bp.v > stdout.txt
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt fittimeFinal "100 min"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
img2dft output_bp.v test.roi output_bp.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -x=n -y=y -yunit=y -abs=0.005 -rel=2 correct_bp.dft output_bp.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n 2.0.2 \n"
printf " Test case 1: Option -end with a lot shorter time than in the data. \n"
printf " Expected result: Correct results calculated. \n\n"

rm -f output_bp.v output_bp.dft

$PROGRAM -end=60 -d3 -thr=0 dynamic.v refdynamic.tac output_bp.v > stdout.txt
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt fittimeFinal "60 min"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
img2dft output_bp.v test.roi output_bp.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -x=n -y=y -yunit=y -abs=0.02 -rel=5 correct_bp.dft output_bp.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"





printf "\n-------------------------------------------------------------------\n"



printf "\n 2.1.0 \n"
printf " Test case: Option -dvr. \n"
printf " Expected result: Correct results calculated. \n\n"

rm -f output_dvr.v output_dvr.dft
taccalc correct_bp.dft + 1 temp.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

$PROGRAM -dvr -thr=0 dynamic.v refdynamic.tac output_dvr.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
img2dft output_dvr.v test.roi output_dvr.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -x=n -y=y -yunit=y -abs=0.001 -rel=1 temp.dft output_dvr.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n-------------------------------------------------------------------\n"


printf "\n 2.2.0 \n"
printf " Test case: Optional other images are saved. \n"
printf " Expected result: Correct results calculated. \n\n"

rm -f output_bp.v output_bp.dft 
rm -f output_r1.v output_r1.dft 
rm -f output_k2.v output_k2.dft 
rm -f output_theta3.v output_theta3.dft 
rm -f output_rp.v output_rp.dft 
rm -f output_dual.v output_dual.dft 

$PROGRAM -r1=output_r1.v -k2=output_k2.v -theta3=output_theta3.v -rp=output_rp.v -dual=output_dual.v -thr=0 dynamic.v refdynamic.tac output_bp.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

img2dft output_bp.v test.roi output_bp.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -x=n -y=y -yunit=y -abs=0.001 -rel=1 correct_bp.dft output_bp.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

img2dft output_r1.v test.roi output_r1.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -x=n -y=y -yunit=y -abs=0.001 -rel=1 correct_r1.tac output_r1.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

img2dft output_k2.v test.roi output_k2.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -x=n -y=y -yunit=n -abs=0.01 -rel=2 correct_k2.tac output_k2.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

img2dft output_theta3.v test.roi output_theta3.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

img2dft output_rp.v test.roi output_rp.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

img2dft output_dual.v test.roi output_dual.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

printf "\n passed. \n\n"




printf "\n 2.2.1 \n"
printf " Test case 1: Option -srtm2. \n"
printf " Test case 2: Optional k2s image can be saved with option -srtm2. \n"
printf " Note that SRTM2 requires different data where assumption holds. \n"
printf " Expected result: Correct results calculated. \n\n"

rm -f output_bp.v output_bp.dft 
rm -f output_k2s.v output_k2s.dft 

$PROGRAM -d -srtm2 -k2s=output_k2s.v -thr=0 srtm2_noiseless.v srtm2ref.tac output_bp.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

imgmax -both correct_bp_srtm2.v output_bp.v
img2dft output_bp.v srtm2.roi output_bp.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
imgmatch -abs=0.1 -rel=5 correct_bp_srtm2.v output_bp.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

imgmax -both correct_k2s_srtm2.v output_k2s.v
img2dft output_k2s.v srtm2.roi output_k2s.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
printf "\n k2ref estimate is bad when BP is small, thus leave out from the test\n"
img2dft correct_k2s_srtm2.v srtm2.roi correct_k2s_srtm2.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
cp -f correct_k2s_srtm2.dft test1.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
cp -f output_k2s.dft test2.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
tacdel test1.dft 1 2 3
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
tacdel test2.dft 1 2 3
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
tacmatch -abs=0.02 -rel=5 test1.dft test2.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"





printf "\n 2.3.0 \n"
printf " Test case 1: Noisy image without option -srtm2. \n"
printf " Expected result: Somewhat correct results calculated. \n\n"

rm -f output_bp.v output_bp.dft 

$PROGRAM srtm2_noisy.v srtm2ref.tac output_bp.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

imgmax -both correct_bp_srtm2.v output_bp.v
img2dft output_bp.v srtm2.roi output_bp.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
taclist output_bp.dft
imgmatch -abs=0.1 -rel=5 correct_bp_srtm2.v output_bp.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 2.3.1 \n"
printf " Test case 1: Noisy image with option -srtm2. \n"
printf " Expected result: Somewhat correct results calculated. \n\n"

rm -f output_bp.v output_bp.dft 

$PROGRAM -srtm2 srtm2_noisy.v srtm2ref.tac output_bp.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

imgmax -both correct_bp_srtm2.v output_bp.v
img2dft output_bp.v srtm2.roi output_bp.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
taclist output_bp.dft
imgmatch -abs=0.1 -rel=5 correct_bp_srtm2.v output_bp.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n===================================================================\n"


printf "\n 3.0.0 \n"
printf " Test case: Invalid option. \n"
printf " Expected result: Error. \n\n"

$PROGRAM -stupidoption dynamic.v refdynamic.tac output.v 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error "invalid option '-stupidoption'."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 3.0.1 \n"
printf " Test case: Extra argument. \n"
printf " Expected result: Error. \n\n"

$PROGRAM dynamic.v refdynamic.tac output.v extra 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error "invalid argument 'extra'."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 3.0.2 \n"
printf " Test case: Missing argument. \n"
printf " Expected result: Error. \n\n"

$PROGRAM refdynamic.tac output.v  1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Error" "missing command-line argument; use option --help"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

$PROGRAM dynamic.v output.v 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Error" "missing command-line argument; use option --help"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

$PROGRAM dynamic.v refdynamic.tac  1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Error" "missing command-line argument; use option --help"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"




printf "\n 3.1.0 \n"
printf " Test case: Input file does not exist. \n"
printf " Expected result: Error. \n\n"

$PROGRAM nonexistingfile.v refdynamic.tac output.v 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Error"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

$PROGRAM dynamic.v nonexistingfile.tac output.v 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Error"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"







printf "\n===================================================================\n"
printf "   All passed! \n"
printf "===================================================================\n"
exit 0

