/** @file img2tif.c
 *  @brief Create a TIFF image from a PET image or sinogram.
 *  @details Previous application name ecat2tif. 
 *  @copyright (c) Turku PET Centre
 *  @author Vesa Oikonen
 */
/// @cond
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <string.h>
#include <unistd.h>
#include <time.h>
/*****************************************************************************/
#include "libtpcmisc.h"
#include "libtpcimgio.h"
#include "libtpcimgp.h"
/*****************************************************************************/

/*****************************************************************************/
static char *info[] = {
  "Extract the matrices of an ECAT 6.3 or ECAT 7 image or sinogram, or",
  "NIfTI, Analyze 7.5, or microPET image, to TIFF 6.0 image.",
  " ",
  "Usage: @P [Options] imgfile [tiffile]",
  " ",
  "Options:",
  " -p=<Plane>",
  "     A specified image plane is extracted; by default all planes.",
  " -f=<Frame>",
  "     A specified image frame is extracted; by default all frames.",
  " -s[=Value]",
  "     Color scale of output images is fixed from 0 to max of all",
  "     image matrices, or from 0 to Value.",
  " -L",
  "     Log10 transform.",
  " -L1",
  "     Log10 transform, adding 1 to pixel values before transform.",
  " -rb",
  "     Apply rainbow colorscale instead of default grayscale.",
  " -rbw",
  "     Apply rainbow colorscale with white background.",
  " -gr",
  "     Apply grayscale (0=black, highest=white).",
  " -gi",
  "     Apply inverse grayscale; default.",
  " -th=<nr>",
  "     Number of matrices tiled horizontally; set to a large value to draw",
  "     all matrices in one row.",
  " -tv=<nr>",
  "     Number of matrices tiled vertically; set to a large value to draw",
  "     all matrices in one column.",
  " -stdoptions", // List standard options like --help, -v, etc
  " ",
  "Example 1: Make TIFF of plane 8 and frame 17, which is yet scaled to",
  "the level of whole image maximum:",
  "  @P -s -p=8 -f=17 s2345dy1.v s2345_pl08_fr17.tif",
  "Example 2: Make TIFF of all image matrices in rainbow color scale:",
  "  @P -rb s2345sum.img s2345sum.tif",
  "Example 3: Make TIFF of all image matrices, scaling the colors to value 3:",
  "  @P -S=3 a3456bp.v a3456bp.tif",
  " ",
  "See also: imgslice, imgmax, imgthrs, imgunit, img2flat",
  " ",
  "Keywords: image, sinogram, TIFF",
  0};
/*****************************************************************************/

/*****************************************************************************/
/* Turn on the globbing of the command line, since it is disabled by default in
   mingw-w64 (_dowildcard=0); in MinGW32 define _CRT_glob instead, if necessary;
   In Unix&Linux wildcard command line processing is enabled by default. */
/*
#undef _CRT_glob
#define _CRT_glob -1
*/
int _dowildcard = -1;
/*****************************************************************************/

/*****************************************************************************/
/**
 *  Main
 */
int main(int argc, char **argv)
{
  int     ai, help=0, version=0, verbose=1;
  int     pi, ret, fixedScale=0, trLog10=0;
  int     frame=-1, plane=-1;
  int     colorscale=PET_GRAYSCALE_INV;
  char    ecatfile[FILENAME_MAX], tiffile[FILENAME_MAX], *cptr=NULL;
  char    tmp[256];
  int     tileX=0, tileY=0;
  IMG     img;
  float   imgmax=-1.0;


  /*
   *  Get arguments
   */
  if(argc==1) {tpcPrintUsage(argv[0], info, stderr); return(1);}
  ecatfile[0]=tiffile[0]=(char)0;
  imgInit(&img);
  /* Options */
  for(ai=1; ai<argc; ai++) if(*argv[ai]=='-') { /* options */
    cptr=argv[ai]+1; if(*cptr=='-') cptr++; if(cptr==NULL) continue;
    if(tpcProcessStdOptions(argv[ai], &help, &version, &verbose)==0) continue;
    cptr=argv[ai]+1;
    if(strncasecmp(cptr, "P=", 2)==0 && strlen(cptr)>2) {
      plane=atoi(cptr+2); if(plane>0) continue;
    } else if(strncasecmp(cptr, "F=", 2)==0 && strlen(cptr)>2) {
      frame=atoi(cptr+2)-1; if(frame>=0) continue;
    } else if(strcasecmp(cptr, "RBW")==0) {
      colorscale=PET_RAINBOW_WB; continue;
    } else if(strncasecmp(cptr, "R", 1)==0) {
      colorscale=PET_RAINBOW; continue;
    } else if(strncasecmp(cptr, "GR", 2)==0) {
      colorscale=PET_GRAYSCALE; continue;
    } else if(strncasecmp(cptr, "GI", 2)==0) {
      colorscale=PET_GRAYSCALE_INV; continue;
    } else if(strcasecmp(cptr, "S")==0) {
      fixedScale=1; continue;
    } else if(strncasecmp(cptr, "S=", 2)==0) {
      fixedScale=1; imgmax=atof_dpi(cptr+2); if(imgmax>0.0) continue;
    } else if(strcasecmp(cptr, "L")==0) {
      trLog10=1; continue;
    } else if(strcasecmp(cptr, "L1")==0) {
      trLog10=2; continue;
    } else if(strncasecmp(cptr, "TH=", 3)==0 && strlen(cptr)>3) {
      tileX=atoi(cptr+3); if(tileX>0) continue; 
    } else if(strncasecmp(cptr, "TV=", 3)==0 && strlen(cptr)>3) {
      tileY=atoi(cptr+3); if(tileY>0) continue; 
    }
    fprintf(stderr, "Error: invalid option '%s'\n", argv[ai]); 
    return(1);
  } else break;
  
  /* Print help or version? */
  if(help==2) {tpcHtmlUsage(argv[0], info, ""); return(0);}
  if(help) {tpcPrintUsage(argv[0], info, stdout); return(0);}
  if(version) {tpcPrintBuild(argv[0], stdout); return(0);}

  /* Arguments */
  for(; ai<argc; ai++) {
    if(!ecatfile[0]) {
      strcpy(ecatfile, argv[ai]); continue;
    } else if(!tiffile[0]) {
      strcpy(tiffile, argv[ai]); continue;
    }
    fprintf(stderr, "Error: invalid argument '%s'\n", argv[ai]); 
    return(1);
  }

  /* Is something missing? */
  if(!ecatfile[0]) {tpcPrintUsage(argv[0], info, stdout); return(1);}

  /* If TIFF filename was not given, then make it from image filename */
  if(!tiffile[0]) {
    strcpy(tiffile, ecatfile);
    cptr=strrchr(tiffile, '.'); if(cptr!=NULL) *cptr=(char)0;
    strcat(tiffile, ".tif");
  }
  if(strcasecmp(ecatfile, tiffile)==0) {
    fprintf(stderr, "Error: same name for input and output file.\n"); 
    return(1);
  }

  
  /* In verbose mode print arguments and options */
  if(verbose>1) {
    for(ai=0; ai<argc; ai++)
      printf("%s ", argv[ai]);
    printf("\n");
    printf("ecatfile := %s\n", ecatfile);
    printf("tiffile := %s\n", tiffile);
    printf("plane := %d\nframe := %d\nfixedScale := %d\nimgmax := %g\n",
      plane, 1+frame, fixedScale, imgmax);
    printf("Log10 transformation := %d\n", trLog10);
    if(tileX>0) printf("tileX := %d\n", tileX);
    if(tileY>0) printf("tileY := %d\n", tileY);
  }


  /*
   *  Read image file
   */
  if(verbose>1) fprintf(stdout, "reading %s\n", ecatfile);
  ret=imgRead(ecatfile, &img);
  if(ret) {
    fprintf(stderr, "Error: %s\n", img.statmsg);
    if(verbose>1) printf("ret := %d\n", ret);
    return(2);
  }
  if(verbose>9) imgInfo(&img);
  if(imgNaNs(&img, 1)>0)
    if(verbose>0) fprintf(stderr, "Warning: missing pixel values.\n");


  /*
   *  Switch the plane number to the order nr
   *  Check the frame number
   */
  if(frame>=0 && img.dimt<(frame+1)) {
    fprintf(stderr, "Error: file does not contain frame %d\n", frame+1);
    imgEmpty(&img); return(3);
  }
  if(plane>0) {
    for(pi=0; pi<img.dimz; pi++)
      if(img.planeNumber[pi]==plane) {plane=pi; break;}
    if(pi==img.dimz) {
      fprintf(stderr, "Error: file does not contain plane %d\n", plane);
      imgEmpty(&img); return(3);
    }
    if(verbose>1) 
      printf("Required plane was on index %d in the datafile.\n", plane);
  }
  

  /*
   *  Make log10 transformation if required
   */
  if(trLog10>0) {
    if(verbose>0) fprintf(stdout, "computing logarithms.\n");
    if(trLog10>1) ret=imgArithmConst(&img, 1.0, '+', -1.0, verbose-3);
    ret=imgLog10(&img);
  }


  /*
   *  If fixed color scale was required, then search the max of all image.
   *  Unless, of course, max was specified by user.
   */
  if(fixedScale && imgmax<=0.0) {
    ret=imgMax(&img, &imgmax);
    if(verbose>0) fprintf(stdout, "maximum pixel value in %s is %g, unit %s\n",
      ecatfile, imgmax, imgUnit(img.unit) );
  }


  /*
   *  Make TIFF files from required matrices
   */
  if(fixedScale==0) imgmax=-1.0;
  ret=tiffWriteImg(&img, plane, frame, &imgmax, colorscale, tiffile,
                   tileX, tileY, verbose-2, tmp);
  if(ret) {
    fprintf(stderr, "Error (%d) in writing %s: %s.\n", ret, tiffile, tmp);
    imgEmpty(&img);
    return(7);
  }
  if(fixedScale==0) {
    if(verbose>0) 
      fprintf(stdout, "maximum pixel value in matrix is %g, unit %s\n",
              imgmax, imgUnit(img.unit) );
    if(verbose>1) printf("img.unit := %d\n", img.unit);
  } else if(verbose>1) printf("imgmax := %g\n", imgmax);
  if(verbose>0) printf("%s written.\n", tiffile);

  imgEmpty(&img);
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/// @endcond
