/** @file siflist.c
 *  @brief Prints on screen the contents of a SIF file;
 *         mainly for testing SIF functions.
 *  @copyright (c) Turku PET Centre
 *  @author Vesa Oikonen
 */
/// @cond
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <string.h>
#include <unistd.h>
#include <time.h>
/*****************************************************************************/
#include "libtpcmisc.h"
#include "libtpcimgio.h"
/*****************************************************************************/

/*****************************************************************************/
static char *info[] = {
  "Lists the information in a SIF file.",
  " ",
  "Usage: @P [Options] sif",
  " ",
  "Options:",
  " -i=<Br76|Cu62|C11|Ga68|Ge68|F18|N13|O15|Rb2>",
  "     Isotope, in case it is needed but not given inside SIF file.",
  " -decay=<corrected|non-corrected>",
  "     Weights are calculated for TACs corrected or non-corrected (default)",
  "     for physical decay.",
  " -stdoptions", // List standard options like --help, -v, etc
  " ",
  "See also: sifisot, sifcat, eframe, tacframe, tacformat, taclist, tacweigh",
  " ",
  "Keywords: image, SIF, time, head-curve",
  0};
/*****************************************************************************/

/*****************************************************************************/
/* Turn on the globbing of the command line, since it is disabled by default in
   mingw-w64 (_dowildcard=0); in MinGW32 define _CRT_glob instead, if necessary;
   In Unix&Linux wildcard command line processing is enabled by default. */
/*
#undef _CRT_glob
#define _CRT_glob -1
*/
int _dowildcard = -1;
/*****************************************************************************/

/*****************************************************************************/
/**
 *  Main
 */
int main(int argc, char **argv)
{
  int        ai, help=0, version=0, verbose=1;
  int        ret, decay_corrected=0;
  char      *cptr, siffile[FILENAME_MAX];
  double     halflife=-1.0;


  /*
   *  Get arguments
   */
  if(argc==1) {tpcPrintUsage(argv[0], info, stderr); return(1);}
  siffile[0]=(char)0;
  /* Options */
  for(ai=1; ai<argc; ai++) if(*argv[ai]=='-') { /* options */
    cptr=argv[ai]+1; if(*cptr=='-') cptr++; if(cptr==NULL) continue;
    if(tpcProcessStdOptions(argv[ai], &help, &version, &verbose)==0) continue;
    if(strncasecmp(cptr, "I=", 2)==0) {
      cptr+=2; halflife=60.*hlFromIsotope(cptr); if(halflife>0.0) continue;
    } else if(strncasecmp(cptr, "DECAY=", 6)==0) {
      cptr+=6;
      if(strncasecmp(cptr, "C", 1)==0 || strncasecmp(cptr, "Y", 1)==0) {
        decay_corrected=1; continue;}
      else if(strncasecmp(cptr, "N", 1)==0) {decay_corrected=0; continue;}
    }
    /* We should not be here */
    fprintf(stderr, "Error: invalid option '%s'.\n", argv[ai]);
    return(1);
  } else break;
  
  /* Print help or version? */
  if(help==2) {tpcHtmlUsage(argv[0], info, ""); return(0);}
  if(help) {tpcPrintUsage(argv[0], info, stdout); return(0);}
  if(version) {tpcPrintBuild(argv[0], stdout); return(0);}
  
  /* Process other arguments, starting from the first non-option */
  if(ai<argc) {strlcpy(siffile, argv[ai], FILENAME_MAX); ai++;}
  if(ai<argc) {fprintf(stderr, "Error: too many arguments.\n"); return(1);}

  /* Is something missing or wrong? */
  if(!siffile[0]) {
    fprintf(stderr, "Error: missing command-line argument; use option --help\n");
    return(1);
  }

  /* In verbose mode print arguments and options */
  if(verbose>1) {
    printf("siffile := %s\n", siffile);
    printf("decay_corrected := %d\n", decay_corrected);
    if(halflife>0) printf("halflife := %g\n", halflife);
  }


  /*
   *  Read sif file
   */
  if(verbose>1) printf("reading file %s\n", siffile);
  if(verbose>2) SIF_TEST=1;
  SIF sif; sifInit(&sif); 
  ret=sifRead(siffile, &sif);
  if(ret) {
    fprintf(stderr, "Error (%d) in reading '%s': %s\n", ret, siffile, siferrmsg);
    sifEmpty(&sif);
    return(2);
  }
  /* Get halflife from SIF file, if available */
  if(halflife<0.0 && strlen(sif.isotope_name)>0) {
    halflife=60.*hlFromIsotope(sif.isotope_name);
    if(verbose>1) printf("halflife := %g\n", halflife);
  }


  /*
   *  Calculate SIF weights
   */
  if(sif.colNr>2) {
    if(verbose>1) printf("calculating weights\n");
    /* If weights are calculated for decay corrected TACs, then check that
       we have the halflife */
    if(decay_corrected!=0 && halflife<=0.0) {
      fprintf(stderr, "Error: isotope is required for decay correction.\n");
      sifEmpty(&sif); return(3);
    }
    /* If weights are calculated for TACs not corrected for decay, then
       set halflife=0 */
    if(decay_corrected==0) halflife=0.0;
    sifWeight(&sif, halflife);
  }
  if(verbose>2) printf("halflife := %g\n", halflife);


  /*
   *  Print SIF data
   */
  if(verbose>10) printf("printing SIF contents\n");
  sifPrint(&sif);


  /*
   *  Free memory
   */
  if(verbose>10) printf("exiting\n");
  sifEmpty(&sif);

  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/// @endcond
