/** @file imgbox.c
 *  @brief Extract a box-like volume of 3D/4D PET image.
 *  @copyright (c) Turku PET Centre
 *  @author Vesa Oikonen
 */
/// @cond
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <string.h>
#include <unistd.h>
#include <time.h>
/*****************************************************************************/
#include "libtpcmisc.h"
#include "libtpcimgio.h"
#include "libtpcimgp.h"
/*****************************************************************************/

/*****************************************************************************/
static char *info[] = {
  "Extract a box-like volume of 3D/4D PET image in ECAT format.",
  "NIfTI-1 format is supported, except that SIF is not written.",
  "Image is processed frame-by-frame, thus even large files can be processed.",
  "Note that no backup is made of any previous output file.",
  " ",
  "Usage: @P [Options] imgfile [vrdfile outputfile]",
  " ",
  "Options:",
  " -zero",
  "     Image is not cropped, but voxels outside of the given volume are",
  "     set to zero.", 
  " -stdoptions", // List standard options like --help, -v, etc
  " ",
  "Volume range definition (vrd) file is an ASCII text file, which containts",
  "pixel coordinates (x y z; 1..dimension) of the two opposite corners of",
  "the cropped image volume, for example:",
  "     corner1 := 63 57 26",
  "     corner2 := 84 71 44",
  "If only input image filename is given, then program prints the image",
  "dimensions.",
  " ",
  "See also: imgdim, imgslim, esplit, e7vplavg, imgshrink, pxl2tac, imgdelfr",
  " ",
  "Keywords: image, cropping, mask, compression, tool",
  0};
/*****************************************************************************/

/*****************************************************************************/
/* Turn on the globbing of the command line, since it is disabled by default in
   mingw-w64 (_dowildcard=0); in MinGW32 define _CRT_glob instead, if necessary;
   In Unix&Linux wildcard command line processing is enabled by default. */
/*
#undef _CRT_glob
#define _CRT_glob -1
*/
int _dowildcard = -1;
/*****************************************************************************/

/*****************************************************************************/
/**
 *  Main
 */
int main(int argc, char **argv)
{
  int       ai, help=0, version=0, verbose=1;
  int       doCrop=1; // 0=zeroes; 1=crop
  int       ret, fi=0, xi, yi, zi, ti;
  int       newdimx, newdimy, newdimz;
  char      imgfile[FILENAME_MAX], outfile[FILENAME_MAX], vrdfile[FILENAME_MAX],
           *cptr, temp[FILENAME_MAX];
  IMG       test_img, img, outimg;
  VOL_RANGE vol_range={0,0,0,0,0,0};


  /*
   *  Get arguments
   */
  if(argc==1) {tpcPrintUsage(argv[0], info, stderr); return(1);}
  imgfile[0]=outfile[0]=vrdfile[0]=(char)0;
  /* Options */
  for(ai=1; ai<argc; ai++) if(*argv[ai]=='-') {
    cptr=argv[ai]+1; if(*cptr=='-') cptr++; if(cptr==NULL) continue;
    if(tpcProcessStdOptions(argv[ai], &help, &version, &verbose)==0) continue;
    if(strcasecmp(cptr, "CROP")==0) {
      doCrop=1; continue;
    } else if(strcasecmp(cptr, "ZERO")==0) {
      doCrop=0; continue;
    }
    fprintf(stderr, "Error: unknown option '%s'.\n", argv[ai]);
    return(1);
  } else break;
  
  /* Print help or version? */
  if(help==2) {tpcHtmlUsage(argv[0], info, ""); return(0);}
  if(help) {tpcPrintUsage(argv[0], info, stdout); return(0);}
  if(version) {tpcPrintBuild(argv[0], stdout); return(0);}

  /* Process other arguments, starting from the first non-option */
  if(ai<argc) {strlcpy(imgfile, argv[ai], FILENAME_MAX); ai++;}
  if(ai<argc) {strlcpy(vrdfile, argv[ai], FILENAME_MAX); ai++;}
  if(ai<argc) {strlcpy(outfile, argv[ai], FILENAME_MAX); ai++;}
  if(ai<argc) {fprintf(stderr, "Error: too many arguments.\n"); return(1);}

  /* Is something missing? */
  if(!imgfile[0] || (vrdfile[0] && !outfile[0])) {
    fprintf(stderr, "Error: missing command-line argument; use option --help\n");
    return(1);
  }


  /* In verbose mode print arguments and options */
  if(verbose>1) {
    printf("imgfile := %s\n", imgfile);
    if(outfile[0]) printf("outfile := %s\n", outfile);
    if(vrdfile[0]) printf("vrdfile := %s\n", vrdfile);
    printf("doCrop := %d\n", doCrop);
  }
  if(verbose>3) IMG_TEST=verbose-3; else IMG_TEST=0;


  /*
   *  Read 'header' information from image
   */
  if(verbose>1) printf("reading header from %s\n", imgfile);
  imgInit(&test_img);
  ret=imgReadHeader(imgfile, &test_img, IMG_UNKNOWN);
  if(ret!=0) {
    fprintf(stderr, "Error: %s\n", test_img.statmsg);
    return(2);
  }
  if(verbose>20) imgInfo(&test_img);

  /*
   *  Print original image dimensions
   */
  if(verbose>0 || !vrdfile[0]) {
    fprintf(stdout, "image_dimensions := %d %d %d\n",
      test_img.dimx, test_img.dimy, test_img.dimz);
    fprintf(stdout, "frame_nr := %d\n", test_img.dimt);
  }

  /*
   *  Quit if no VRD file
   */
  if(!vrdfile[0]) {imgEmpty(&test_img); return(0);}


  /*
   *  Read Volume Range Definition File
   */
  ret=vrdRead(vrdfile, &vol_range, temp);
  if(ret!=0) {
    fprintf(stderr, "Error in reading %s: %s\n", vrdfile, temp);
    imgEmpty(&test_img); return(3);
  }
  if(verbose>1) {
    printf("volume_range.x := %d - %d\n", vol_range.x1, vol_range.x2);
    printf("vol_range.y := %d - %d\n", vol_range.y1, vol_range.y2);
    printf("vol_range.z := %d - %d\n", vol_range.z1, vol_range.z2);
  }
  /* Check that box is inside image volume */
  if(vol_range.x1<1 || vol_range.x2>test_img.dimx) {
    fprintf(stderr, "Error: invalid x range.\n");
    imgEmpty(&test_img); return(4);
  }
  if(vol_range.y1<1 || vol_range.y2>test_img.dimy) {
    fprintf(stderr, "Error: invalid y range.\n");
    imgEmpty(&test_img); return(5);
  }
  if(vol_range.z1<1 || vol_range.z2>test_img.dimz) {
    fprintf(stderr, "Error: invalid z range.\n");
    imgEmpty(&test_img); return(6);
  }

  /* Calculate cropped image dimensions */
  newdimx=1+vol_range.x2-vol_range.x1;
  newdimy=1+vol_range.y2-vol_range.y1;
  newdimz=1+vol_range.z2-vol_range.z1;
  if(verbose>0 && doCrop!=0) {
    printf("  new_dimx := %d\n", newdimx);
    printf("  new_dimy := %d\n", newdimy);
    printf("  new_dimz := %d\n", newdimz);
  }
  /* Check if anything needs to be done */
  if(newdimx==test_img.dimx && newdimy==test_img.dimy && newdimz==test_img.dimz)
  {
    fprintf(stderr, "Error: whole image is included\n");
    imgEmpty(&test_img); return(7);
  }

  /* Remove existing image file */
  if(access(outfile, 0)!=-1 && remove(outfile)!=0) {
    fprintf(stderr, "Error: cannot overwrite %s\n", outfile);
    imgEmpty(&test_img); return(11);
  }

  /*
   *  Do the image cropping
   */
  if(doCrop!=0) {

  /*
   *  Read (again) image frame-by-frame, and write the selected slices
   */
  if(verbose>0)
    printf("cropping selected volume from %s to %s\n", imgfile, outfile);
  fi=0; imgInit(&outimg); imgInit(&img);
  for(ti=0; ti<test_img.dimt; ti++) {
    if(verbose>2) printf("    frame %d\n", ti+1);
    /* Read frame */
    ret=imgReadFrame(imgfile, ti+1, &img, 0);
    if(ret!=0) {
      fprintf(stderr, "Error: %s\n", imgStatus(ret));
      imgEmpty(&test_img); imgEmpty(&img);
      return(9);
    }
    /* If this is the first frame, then create output IMG */
    if(fi==0) {
      if(verbose>4) printf("     creating output IMG\n");
      ret=imgAllocateWithHeader(&outimg, newdimz, newdimy, newdimx, 1, &img);
    } else { /* otherwise just copy the header */
      if(verbose>4) printf("     copying header\n");
      ret=imgCopyhdr(&img, &outimg);
    }
    if(ret!=0) {
      fprintf(stderr, "Error: %s\n", imgStatus(ret));
      imgEmpty(&test_img); imgEmpty(&img); imgEmpty(&outimg);
      return(10);
    }
    /* Copy pixel data */
    if(verbose>3) printf("     copying pixel data\n");
    int nxi, nyi, nzi;
    for(zi=vol_range.z1-1, nzi=0; zi<=vol_range.z2-1; zi++, nzi++) {
      for(yi=vol_range.y1-1, nyi=0; yi<=vol_range.y2-1; yi++, nyi++) {
        for(xi=vol_range.x1-1, nxi=0; xi<=vol_range.x2-1; xi++, nxi++) {
          if(verbose>10) printf("  img1[%d][%d][%d]=%g -> img2[%d][%d][%d]\n",
            zi, yi, xi, img.m[zi][yi][xi][0], nxi, nyi, nxi);
          outimg.m[nzi][nyi][nxi][0]=img.m[zi][yi][xi][0];
        }
      }
    }
    /* Write the image frame */
    if(verbose>3) printf("     writing frame %d\n", fi+1);
    /* Notice that this function does not support Analyze format;
       therefore write format is changed to NIfTI. */
    if(outimg._fileFormat==IMG_ANA || outimg._fileFormat==IMG_ANA_L) {
      if(verbose>10) 
        printf("outimg._fileFormat %d -> %d\n", outimg._fileFormat, IMG_NIFTI_1S);
      outimg._fileFormat=IMG_NIFTI_1S;
    }
    ret=imgWriteFrame(outfile, fi+1, &outimg, 0); //printf("ret := %d\n", ret);
    if(ret!=STATUS_OK) {
      fprintf(stderr, "Error: %s\n", imgStatus(ret));
      if(verbose>18) imgInfo(&outimg);
      imgEmpty(&test_img); imgEmpty(&img); imgEmpty(&outimg);
      return(13);
    }
    if(verbose>2) printf("    frame written.\n");
    /* Prepare for the next frame */
    fi++;
  } // next frame
  imgEmpty(&test_img); imgEmpty(&img); imgEmpty(&outimg);
  if(verbose>0) printf("Specified image volume written in %s\n", outfile);
  return(0);

  } // cropping


  /*
   *  Set voxels to zero outside the given volume
   */
  if(verbose>0)
    printf("masking voxels outside selected volume from %s to %s\n", imgfile, outfile);
  fi=0; imgInit(&img);
  for(ti=0; ti<test_img.dimt; ti++) {
    if(verbose>2) printf("    frame %d\n", ti+1);
    /* Read frame */
    ret=imgReadFrame(imgfile, ti+1, &img, 0);
    if(ret!=0) {
      fprintf(stderr, "Error: %s\n", imgStatus(ret));
      imgEmpty(&test_img); imgEmpty(&img);
      return(9);
    }
    /* Edit pixel data */
    if(verbose>3) printf("     editing pixel data\n");
    for(zi=0; zi<img.dimz; zi++)
      for(yi=0; yi<img.dimy; yi++)
        for(xi=0; xi<img.dimx; xi++) {
          if(zi<vol_range.z1-1 || zi>vol_range.z2-1) {
            img.m[zi][yi][xi][0]=0.0; continue;}
          if(yi<vol_range.y1-1 || yi>vol_range.y2-1) {
            img.m[zi][yi][xi][0]=0.0; continue;}
          if(xi<vol_range.x1-1 || xi>vol_range.x2-1) {
            img.m[zi][yi][xi][0]=0.0; continue;}
        }
    /* Write the image frame */
    if(verbose>3) printf("     writing frame %d\n", fi+1);
    ret=imgWriteFrame(outfile, fi+1, &img, 0); //printf("ret := %d\n", ret);
    if(ret!=STATUS_OK) {
      fprintf(stderr, "Error: %s\n", imgStatus(ret));
      imgEmpty(&test_img); imgEmpty(&img);
      return(13);
    }
    if(verbose>2) printf("    frame written.\n");
    /* Prepare for the next frame */
    fi++;
  } // next frame
  imgEmpty(&test_img); imgEmpty(&img);
  if(verbose>0) printf("Masked image written in %s\n", outfile);

  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/// @endcond
