/** @file imgflip.c
 *  @brief Flip PET image in x, y, z directions.
 *  @copyright (c) Turku PET Centre
 *  @author Vesa Oikonen
 */
/// @cond
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <math.h>
#include <string.h>
#include <time.h>
/*****************************************************************************/
#include "libtpcmisc.h"
#include "libtpcimgio.h"
#include "libtpcimgp.h"
/*****************************************************************************/

/*****************************************************************************/
static char *info[] = {
  "Flip PET image in X, Y, and/or Z direction(s).",
  "By default, if no options are given, flipping in X and Y directions.",
  " ",
  "Usage: @P [Options] imagefile outputfile",
  " ",
  "Options:",
  " -x",
  "     Flip image in X direction (horizontally).",
  " -y",
  "     Flip image in Y direction (vertically).",
  " -z",
  "     Flip image in Z direction (image planes/slices).",
  " -up | -right",
  "     Flip image to view it from above or right side.",
  " -stdoptions", // List standard options like --help, -v, etc
  " ",
  "See also: esplit, img2cube, imgbox, imgslice, img2tif, ecat2ana",
  " ",
  "Keywords: image, tool, software testing",
  0};
/*****************************************************************************/

/*****************************************************************************/
/* Turn on the globbing of the command line, since it is disabled by default in
   mingw-w64 (_dowildcard=0); in MinGW32 define _CRT_glob instead, if necessary;
   In Unix&Linux wildcard command line processing is enabled by default. */
/*
#undef _CRT_glob
#define _CRT_glob -1
*/
int _dowildcard = -1;
/*****************************************************************************/

/*****************************************************************************/
/**
 *  Main
 */
int main(int argc, char **argv)
{
  int       ai, help=0, version=0, verbose=1;
  int       zi, ret;
  int       flip_vertical=0, flip_horizontal=0, flip_planes=0;
  int       flip_right=0, flip_up=0;
  char      imgfile[FILENAME_MAX], resfile[FILENAME_MAX],
           *cptr, tmp[FILENAME_MAX];
  IMG       img;


  /*
   *  Get arguments
   */
  if(argc==1) {tpcPrintUsage(argv[0], info, stderr); return(1);}
  imgfile[0]=resfile[0]=(char)0;
  imgInit(&img);
  /* Options */
  for(ai=1; ai<argc; ai++) if(*argv[ai]=='-') { /* options */
    cptr=argv[ai]+1; if(*cptr=='-') cptr++; if(cptr==NULL) continue;
    if(tpcProcessStdOptions(argv[ai], &help, &version, &verbose)==0) continue;
    if(strcasecmp(cptr, "X")==0) {
      flip_horizontal=1; continue;
    } else if(strcasecmp(cptr, "Y")==0) {
      flip_vertical=1; continue;
    } else if(strcasecmp(cptr, "Z")==0) {
      flip_planes=1; continue;
    } else if(strspn(cptr, "XYZxyz")==strlen(cptr)) {
      if(strpbrk("Xx", cptr)!=NULL) flip_horizontal=1;
      if(strpbrk("Yy", cptr)!=NULL) flip_vertical=1;
      if(strpbrk("Zz", cptr)!=NULL) flip_planes=1;
      continue;
    } else if(strcasecmp(cptr, "RIGHT")==0 || strcasecmp(cptr, "RIGTH")==0) {
      flip_right=1; continue;
    } else if(strcasecmp(cptr, "UP")==0) {
      flip_up=1; continue;
    }
    fprintf(stderr, "Error: invalid option '%s'.\n", argv[ai]);
    return(1);
  } else break;
  
  /* Print help or version? */
  if(help==2) {tpcHtmlUsage(argv[0], info, ""); return(0);}
  if(help) {tpcPrintUsage(argv[0], info, stdout); return(0);}
  if(version) {tpcPrintBuild(argv[0], stdout); return(0);}

  /* Process other arguments, starting from the first non-option */
  for(; ai<argc; ai++) {
    if(!imgfile[0]) {    /* Dynamic ECAT file */
      strcpy(imgfile, argv[ai]); continue;
    } else if(!resfile[0]) {    /* Output file */
      strcpy(resfile, argv[ai]); continue;
    }
    /* We should not be here */
    fprintf(stderr, "Error: invalid argument '%s'.\n", argv[ai]);
    return(1);
  } /* next argument */

  /* Is something missing? */
  if(!resfile[0]) {
    fprintf(stderr, "Error: missing command-line argument; use option --help\n");
    return(1);
  }
  if(flip_horizontal==0 && flip_vertical==0 && flip_planes==0
     && flip_up==0 && flip_right==0)
  {
    if(verbose>0)
      fprintf(stderr, "Warning: no flipping was requested; assuming X and Y.\n");
    flip_vertical=flip_horizontal=1;
  }
  if(flip_right && flip_up) {
    fprintf(stderr, "Error: do not use options -right and -up together.\n");
    return(1);
  }

  /* In verbose mode print arguments and options */
  if(verbose>1) {
    printf("imgfile := %s\n", imgfile);
    printf("resfile := %s\n", resfile);
    printf("flip_x := %d\n", flip_horizontal);
    printf("flip_y := %d\n", flip_vertical);
    printf("flip_z := %d\n", flip_planes);
    printf("flip_right := %d\n", flip_right);
    printf("flip_up := %d\n", flip_up);
  }
  if(verbose>10) IMG_TEST=verbose-10; else IMG_TEST=0;


  /*
   *  Read the contents of the image file to img data structure
   */
  if(verbose>1) fprintf(stdout, "reading %s\n", imgfile);
  ret=imgRead(imgfile, &img);
  if(ret) {
    fprintf(stderr, "Error: %s\n", img.statmsg); if(verbose>1) imgInfo(&img);
    imgEmpty(&img); return(2);
  }
  /* If planes are to be flipped, then check that they are continuous */
  if(flip_planes || flip_right || flip_up) {
    for(zi=1; zi<img.dimz; zi++)
      if(abs(img.planeNumber[zi]-img.planeNumber[zi-1])>1) {
        fprintf(stderr, "Error: plane numbers are not continuous.\n");
        imgEmpty(&img); return(2);
      }  
  }
  /* and that there are dimensions to flip */
  if(flip_right || flip_up) {
    if(img.dimz<2) {
      fprintf(stderr, "Error: image has only one plane.\n"); 
      imgEmpty(&img); return(2);
    }
    if(img.dimx<2) {
      fprintf(stderr, "Error: image has only one column.\n"); 
      imgEmpty(&img); return(2);
    }
    if(img.dimy<2) {
      fprintf(stderr, "Error: image has only one row.\n"); 
      imgEmpty(&img); return(2);
    }
  }

  /* Check if resfile exists; rename to backup file, if necessary */
  ret=backupExistingFile(resfile, NULL, tmp); if(ret!=0) {
    fprintf(stderr, "Error: %s\n", tmp); 
    imgEmpty(&img); return(13);
  }


  /*
   *  Flipping XYZ
   */
  if(verbose>1) printf("flipping the image...\n");
  /* Flip horizontally */
  if(flip_horizontal) {
    if(verbose>0) fprintf(stdout, "horizontal flipping\n");
    imgFlipHorizontal(&img);
  }
  /* Flip vertically */
  if(flip_vertical) {
    if(verbose>0) fprintf(stdout, "vertical flipping\n");
    imgFlipVertical(&img);
  }
  /* Flip planes */
  if(flip_planes) {
    /* We checked before that plane numbers are continuous */
    if(verbose>0) fprintf(stdout, "flipping planes\n");
    /* Switch pointers to planes */
    imgFlipPlanes(&img);
    if(verbose>20) imgInfo(&img);
  }

  /*
   *  Look from right or from above
   */
  ret=0;
  if(flip_right) {
    if(verbose>0) fprintf(stdout, "flipping from right\n");
    ret=imgFlipRight(&img);
    if(verbose>20) imgInfo(&img);
  } else if(flip_up) {
    if(verbose>0) fprintf(stdout, "flipping from above\n");
    ret=imgFlipAbove(&img);
    if(verbose>20) imgInfo(&img);
  }
  if(ret!=0) {
    fprintf(stderr, "Error: cannot flip the image.\n"); 
    imgEmpty(&img); return(100+ret);
  }


  /*
   *  Save flipped image file
   */
  if(verbose>1) fprintf(stdout, "writing %s\n", resfile);
  ret=imgWrite(resfile, &img);
  if(ret) {
    fprintf(stderr, "Error: %s\n", img.statmsg); if(verbose>2) imgInfo(&img);
    imgEmpty(&img); return(11);
  }
  imgEmpty(&img);
  if(verbose>0) fprintf(stdout, "done.\n");

  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/// @endcond
