#!/bin/bash
#: Title      : test_imgdecay
#: Date       : 2023-06-29
#: Author     : "Vesa Oikonen" <vesa.oikonen@utu.fi>
#: Options    : None

# Set the name of executable to test
if [ -n "${OS+1}" ] && [ "$OS" = "Windows_NT" ]; then
EXT=.exe;
else
EXT=;
fi
PROGRAM=../../imgdecay$EXT;
if [ ! -f $PROGRAM ]; then
  printf "\nError: application file missing.\n\n"
  exit 1
fi


printf "=====================================================================\n"
printf "creating test data for %s\n" $PROGRAM
printf "=====================================================================\n"

# Check that required files exist
if [ ! -f origdata.dft ]; then
  printf "\nError: required file missing.\n\n"
  exit 1
fi

if [ ! -f original.img ]; then rm -f original.v ; fi
if [ ! -f original.roi ]; then rm -f original.v ; fi
if [ ! -f original.v ]; then
  printf "\n\n Creating original.v \n\n"
  rm -f original.img
  dft2img -Zoom=2 -Scanner=HR+ -Dim=32 origdata.dft plane1.v original.roi
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  imgcalc plane1.v x 2 plane2.v
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  imgadd original.v plane1.v plane2.v
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  esetstrt original.v 2007-05-11 10:03:23
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  img2dft -P original.v original.roi original.dft
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi

if [ ! -f original.img ]; then
  printf "\n\n Creating original.img \n"
  e7to63 original.v original.img
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi

printf "\n\n"
printf "=====================================================================\n"
printf "testing %s\n" $PROGRAM
printf "=====================================================================\n"

printf "\n0.1.0 Unknown option \n\n"

cp -f original.v test.v
$PROGRAM -stupidoption test.v 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error "invalid option '-stupidoption'."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n"

printf "\n===================================================================\n"

printf "\n0.2.0 Extra argument \n\n"

cp -f original.v test.v
$PROGRAM test.v extra 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error "invalid argument 'extra'."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n"

printf "\n===================================================================\n"

printf "\n0.3.0 Isotope option \n\n"

cp -f original.v test.v
$PROGRAM test.v 1>stdout.txt 2>&1
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "isotope" "unknown"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Decay correction" "unknown"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n0.3.1 With isotope option \n\n"

cp -f original.v test.v
$PROGRAM -d1 -i=C11 test.v 1>stdout.txt 2>&1
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "halflife" "1224"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

# Now image should have the isotope
$PROGRAM -d1 test.v 1>stdout.txt 2>&1
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "isotope" "C-11"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
# But nothing else should have been added
iftisval stdout.txt "halflife" "0"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Decay correction" "unknown"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
# and thus image data should still be the same
imgmatch -abs=0.05 original.v test.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n0.3.2 Previously set isotope can be changed \n\n"

$PROGRAM -d1 -i=F18 test.v 1>stdout.txt 2>&1
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "halflife" "6588"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
# we should have got a warning
iftisval stdout.txt "Warning" "possible decay correction should have been removed before isotope was changed to F-18"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
# image data should still be the same
imgmatch -abs=0.05 original.v test.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n0.3.3 Invalid isotope code by option \n\n"
$PROGRAM -d1 -i=Au-129 test.v 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error "invalid isotope with option -i"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n===================================================================\n"

printf "\n0.4.0 option -dc, -rc, -dd, and -rd \n\n"

printf "\n without option \n\n"
cp -f original.v test.v
$PROGRAM -d1 test.v 1>stdout.txt 2>&1
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt decaycor "0"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n any of the options gives an error while isotope is missing \n\n"

$PROGRAM -d1 -dd test.v 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt decaycor "2"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Error" "set isotope with option -i"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

$PROGRAM -d1 -rd test.v 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt decaycor "-2"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Error" "set isotope with option -i"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

$PROGRAM -d1 -dc test.v 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt decaycor "1"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Error" "set isotope with option -i"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

$PROGRAM -d1 -rc test.v 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt decaycor "-1"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Error" "set isotope with option -i"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n0.4.1 invalid combinations of -dc, -rc, -dd, and -rd \n\n"

printf "\n -dd -dc \n\n"
cp -f original.v test.v
# set isotope to test image
$PROGRAM -i=C-11 test.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
$PROGRAM -dd -dc test.v 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Error" "invalid combination of options."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n -rc -rd \n\n"
$PROGRAM -rc -rd test.v 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Error" "invalid combination of options."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n -dd -rd \n\n"
$PROGRAM -dd -rd test.v 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Error" "invalid combination of options."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n -dc -rd \n\n"
$PROGRAM -dc -rd test.v 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Error" "invalid combination of options."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n===================================================================\n"

printf "\n0.5.0 No input file name \n\n"

$PROGRAM -i=F-18 -d2 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n0.5.1 Input file does not exist \n\n"

$PROGRAM -i=F-18 nonexisting.v 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
iftisval stdout.txt "Error in reading nonexisting.v" "cannot open file"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n0.5.2 Input file in wrong format \n\n"

$PROGRAM -i=F-18 $PROGRAM 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
iftisval stdout.txt "Error in reading $PROGRAM" "unknown file format"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n===================================================================\n"


printf "\n0.6.0 backup is made when necessary \n\n"

cp -f original.v test.v
rm -f test.v.bak
$PROGRAM -i=F-18 test.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
$PROGRAM -i=C-11 test.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
if [ ! -f test.v.bak ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
if [ ! -f test.v ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

$PROGRAM test.v 1>stdout.txt 2>&1
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
iftisval stdout.txt "isotope" "C-11"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

$PROGRAM test.v.bak 1>stdout.txt 2>&1
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
iftisval stdout.txt "isotope" "F-18"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n0.6.1 backup is made when file was not rewritten \n\n"

rm -f test.v.bak
$PROGRAM test.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
if [ -f test.v.bak ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n===================================================================\n"
printf "    Use cases"
printf "\n===================================================================\n"

printf "\n1.0.0 status printing when information is NOT available \n\n"

printf "\n ECAT7 \n\n"

cp -f original.v test.v
$PROGRAM test.v 1>stdout.txt 2>&1
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "isotope" "unknown"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Decay correction" "unknown"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n ECAT6 \n\n"

cp -f original.img test.img
$PROGRAM test.img 1>stdout.txt 2>&1
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "isotope" "unknown"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Decay correction" "unknown"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n"


printf "\n1.0.1 status printing when information IS available \n\n"

printf "\n ECAT7 \n\n"

cp -f original.v test.v
$PROGRAM -i=C-11 -dc test.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

$PROGRAM test.v 1>stdout.txt 2>&1
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "isotope" "C-11"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Decay correction" "corrected"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "decay correction factors" ""
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n ECAT6 \n\n"

cp -f original.img test.img
$PROGRAM -i=C-11 -dc test.img
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

$PROGRAM test.img 1>stdout.txt 2>&1
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "isotope" "C-11"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Decay correction" "corrected"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "decay correction factors" ""
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n===================================================================\n"

printf "\n1.1.0 set isotope when not set before \n\n"

printf "\n ECAT7 \n\n"

cp -f original.v test.v
$PROGRAM --silent -i=C-11 test.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

$PROGRAM --silent test.v 1>stdout.txt 2>&1
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "isotope" "C-11"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n ECAT6 \n\n"

cp -f original.img test.img
$PROGRAM --silent -i=C-11 test.img
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

$PROGRAM --silent test.img 1>stdout.txt 2>&1
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "isotope" "C-11"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n1.1.1 set isotope when set before to the same isotope \n\n"

printf "\n ECAT7 \n\n"

$PROGRAM -i=C-11 test.v 1>stdout.txt 2>&1
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Note" "same isotope was previously set."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "isotope" "C-11"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n ECAT6 \n\n"

$PROGRAM -i=C-11 test.img 1>stdout.txt 2>&1
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Note" "same isotope was previously set."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "isotope" "C-11"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n1.1.2 set isotope when set before to another isotope \n\n"
printf "\n      image is not corrected for decay"

printf "\n ECAT7 \n\n"

$PROGRAM -i=F-18 test.v 1>stdout.txt 2>&1
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Warning" "possible decay correction should have been removed before isotope was changed to F-18"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

$PROGRAM test.v 1>stdout.txt 2>&1
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
iftisval stdout.txt "isotope" "F-18"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n ECAT6 \n\n"

$PROGRAM -i=F-18 test.img 1>stdout.txt 2>&1
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Warning" "possible decay correction should have been removed before isotope was changed to F-18"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

$PROGRAM test.img 1>stdout.txt 2>&1
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
iftisval stdout.txt "isotope" "F-18"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n1.1.3 set isotope when set before to another isotope \n\n"
printf "\n      image is corrected for decay"

printf "\n ECAT7 \n\n"

cp -f original.v test.v
$PROGRAM -i=C-11 -dc test.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

rm -f test.v.bak
$PROGRAM -i=F-18 test.v 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Error" "remove previous decay correction before changing isotope."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
# if backup was created then image was changed and that is an error
if [ -f test.v.bak ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n ECAT6 \n\n"

cp -f original.img test.img
$PROGRAM -i=C-11 -dc test.img
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

rm -f test.img.bak
$PROGRAM -i=F-18 test.img 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Error" "remove previous decay correction before changing isotope."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
# if backup was created then image was changed and that is an error
if [ -f test.img.bak ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n===================================================================\n"

printf "\n1.2.0 Correct for physical decay \n\n"
printf "\n      Image does not contain decay factors \n\n"

printf "\n ECAT7 \n\n"

cp -f original.v test.v
$PROGRAM --silent -i=C-11 -dd test.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

$PROGRAM -d2 test.v 1>stdout.txt 2>&1
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "isotope" "C-11"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "decay correction factors" ""
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "frame1" "612 1224 1.96052"
printf "\n ok \n\n"

img2dft -p original.v original.roi test1.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacunit -xconv=min test1.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacdecay -i=C-11 -decay=on test1.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
img2dft -p test.v original.roi test2.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.01 -rel=1 test1.dft test2.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n ECAT6 \n\n"

cp -f original.img test.img
$PROGRAM --silent -i=C-11 -dd test.img
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

$PROGRAM -d2 test.img 1>stdout.txt 2>&1
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "isotope" "C-11"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "decay correction factors" ""
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "frame1" "612 1224 1.96052"
printf "\n ok \n\n"

img2dft -p original.img original.roi test1.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacunit -xconv=min test1.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacdecay -i=C-11 -decay=on test1.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
img2dft -p test.img original.roi test2.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.01 -rel=1 test1.dft test2.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n===================================================================\n"

printf "\n1.2.1 Correct for physical decay \n\n"
printf "\n      Image contains decay factors (added in previous step) \n\n"

printf "\n ECAT7 \n\n"

$PROGRAM -dd test.v 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Error" "test.v is already corrected for decay."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n ECAT6 \n\n"

$PROGRAM -dd test.img 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Error" "test.img is already corrected for decay."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n===================================================================\n"

printf "\n1.3.0 Remove decay correction \n\n"
printf "\n      Image contains decay factors \n\n"

printf "\n ECAT7 \n\n"

cp -f original.v test.v
$PROGRAM -dd -i=C-11 test.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
$PROGRAM -rd test.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
rm -f stdout.txt
lshdr test.v 1 > stdout.txt
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
imgmatch -abs=0.01 original.v test.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n ECAT6 \n\n"

cp -f original.img test.img
$PROGRAM -dd -i=C-11 test.img
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
$PROGRAM -rd test.img
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
rm -f stdout.txt
lshdr test.img 1 > stdout.txt
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
imgmatch -abs=0.01 original.img test.img
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n-------------------------------------------------------------------\n"

printf "\n1.3.1 Remove decay correction \n\n"
printf "\n      Image does not contain decay factors \n\n"

printf "\n ECAT7 \n\n"

cp -f original.v test.v
$PROGRAM -rd -i=C-11 test.v 1>stdout.txt 2>&1
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Warning" "removing decay correction."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n\n"

img2dft -p original.v original.roi test1.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacunit -xconv=min test1.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacdecay -i=C-11 -decay=off test1.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
img2dft -p test.v original.roi test2.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.02 -rel=1 test1.dft test2.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
printf "\n passed. \n\n"


printf "\n ECAT6 \n\n"

cp -f original.img test.img
$PROGRAM -rd -i=C-11 test.img 1>stdout.txt 2>&1
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Warning" "removing decay correction."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n\n"

img2dft -p original.img original.roi test1.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacunit -xconv=min test1.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacdecay -i=C-11 -decay=off test1.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
img2dft -p test.img original.roi test2.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.02 -rel=1 test1.dft test2.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
printf "\n passed. \n\n"



printf "\n===================================================================\n"

printf "\n1.4.0 Add decay correction coefficients \n\n"
printf "\n      Image does not contain factors to start with \n\n"

printf "\n ECAT7 \n\n"

cp -f original.v test.v
$PROGRAM -dc -i=C-11 test.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

$PROGRAM -d2 test.v 1>stdout.txt 2>&1
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "isotope" "C-11"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Decay correction" "corrected"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "decay correction factors" ""
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "frame1" "612 1224 1.96052"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
# data values are not changed
imgmatch -abs=0.01 original.v test.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n ECAT6 \n\n"

cp -f original.img test.img
$PROGRAM -dc -i=C-11 test.img
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

$PROGRAM -d2 test.img 1>stdout.txt 2>&1
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "isotope" "C-11"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Decay correction" "corrected"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "decay correction factors" ""
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "frame1" "612 1224 1.96052"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
# data values are not changed
imgmatch -abs=0.01 original.img test.img
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n-------------------------------------------------------------------\n"

printf "\n1.4.1 Add decay correction coefficients \n\n"
printf "\n      Image already contains factors \n\n"

printf "\n ECAT7 \n\n"

$PROGRAM -dc test.v 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Error" "test.v is already corrected for decay."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n ECAT6 \n\n"

$PROGRAM -dc test.img 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Error" "test.img is already corrected for decay."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n===================================================================\n"

printf "\n1.5.0 Remove decay correction coefficients \n\n"
printf "\n      Image contains factors to start with \n\n"

printf "\n ECAT7 \n\n"

cp -f original.v test.v
$PROGRAM -dc -i=C-11 test.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

$PROGRAM -rc test.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

$PROGRAM -d2 test.v 1>stdout.txt 2>&1
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "isotope" "C-11"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Decay correction" "corrected"
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
# data values are not changed
imgmatch -abs=0.01 original.v test.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n ECAT6 \n\n"

cp -f original.img test.img
$PROGRAM -dc -i=C-11 test.img
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

$PROGRAM -rc test.img
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

$PROGRAM -d2 test.img 1>stdout.txt 2>&1
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "isotope" "C-11"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Decay correction" "corrected"
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
# data values are not changed
imgmatch -abs=0.01 original.img test.img
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n-------------------------------------------------------------------\n"

printf "\n1.5.1 Remove decay correction coefficients \n\n"
printf "\n      Image does not contain factors \n\n"

printf "\n ECAT7 \n\n"

cp -f original.v test.v
$PROGRAM -rc -i=C-11 test.v 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Error" "no need to delete decay correction factors."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n ECAT6 \n\n"

cp -f original.img test.img
$PROGRAM -rc -i=C-11 test.img 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Error" "no need to delete decay correction factors."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n===================================================================\n"
printf "   All passed! \n"
printf "===================================================================\n"
exit 0

