#!/bin/bash
#: Title      : test_imgfiltg
#: Date       : 2023-06-29
#: Author     : "Vesa Oikonen" <vesa.oikonen@utu.fi>
#: Options    : None

# Set the name of executable to test
if [ -n "${OS+1}" ] && [ "$OS" = "Windows_NT" ]; then
EXT=.exe;
else
EXT=;
fi
PROGRAM=../../imgfiltg$EXT;
if [ ! -f $PROGRAM ]; then
  printf "Failed: executable does not exist.\n"
  exit 1
fi


printf "=====================================================================\n"
printf "creating test data for %s\n" $PROGRAM
printf "=====================================================================\n"


printf "\n verifying that required data exists \n"

if [ ! -f cross.header ] || [ ! -f cross.dat ] || [ ! -f cross_tiny.dat ]; then
  printf "Failed: required data does not exist.\n"
  exit 1
fi
if [ ! -f square.header ] || [ ! -f square.dat ]; then
  printf "Failed: required data does not exist.\n"
  exit 1
fi
if [ ! -f frame.dat ] || [ ! -f imgfiltg.dft ]; then
  printf "Failed: required data does not exist.\n"
  exit 1
fi
if [ ! -f cross_octave_smoothed_4.dat ]; then
  printf "Failed: required data does not exist.\n"
  exit 1
fi
if [ ! -f cross_octave_smoothed_8.dat ]; then
  printf "Failed: required data does not exist.\n"
  exit 1
fi
if [ ! -f cross_tiny_octave_smoothed_2.dat ]; then
  printf "Failed: required data does not exist.\n"
  exit 1
fi


printf "\n making test data \n"

if [ ! -f cross.v ]; then
  asc2flat cross.dat cross.bin
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  flat2img -zoom=2.5 -scanner=HR+ cross.bin cross.v 1 1 100 100 
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  e7evhdr cross.v cross.header
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  imgunit -us=kBq/ml cross.v
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  esetstrt cross.v 2009-12-18 14:17:04
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  img2tif -rb -s cross.v cross.tif
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi

if [ ! -f cross_tiny.v ]; then
  asc2flat cross_tiny.dat cross_tiny.bin
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  flat2img -zoom=2.5 -scanner=HR+ cross_tiny.bin cross_tiny.v 1 1 12 12 
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  e7evhdr cross_tiny.v cross.header
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  imgunit -us=kBq/ml cross_tiny.v
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  esetstrt cross_tiny.v 2009-12-21 11:58:32
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  img2tif -rb -s cross_tiny.v cross_tiny.tif
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi


if [ ! -f dynamic_tiny.v ] || [ ! -f dynamic.v ] || [ ! -f dynamic.img ] || [ ! -f static.v ]; then
  rm -f ana/dynamic* nii/dynamic* static.v static_tiny.v dynamic3d_tiny.v
  dft2img -scanner=HR+ -zoom=2 -dim=12 imgfiltg.dft dynamic_tiny.v 
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  dft2img -scanner=HR+ -zoom=2 -dim=128 imgfiltg.dft dynamic.v 
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  imgunit -us=kBq/ml dynamic.v dynamic_tiny.v
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  esetstrt dynamic.v 2005-11-20 18:50:00
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  esetstrt dynamic_tiny.v 2009-12-16 18:03:00
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  e7to63 dynamic.v dynamic.img
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi

if [ ! -f ana/dynamic.img ] || [ ! -f ana/dynamic.hdr ] || [ ! -f ana/dynamic.sif ]; then
  rm -f ana/*
  ecat2ana -sif -o=ana dynamic.v
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi

if [ ! -f nii/dynamic.nii ] || [ ! -f nii/dynamic.sif ]; then
  rm -f nii/*
  ecat2nii -sif -o=nii dynamic.v
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi

if [ ! -f static.v ] || [ ! -f static_tiny.v ]; then
  imginteg -avg dynamic.v 0 0 static.v
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  imginteg -avg dynamic_tiny.v 0 0 static_tiny.v
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi

if [ ! -f dynamic3d_tiny.v ]; then
  imgadd -x=15 dynamic3d_tiny.v dynamic_tiny.v
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi


if [ ! -f cube.v ] || [ ! -f square.v ]; then
  asc2flat square.dat square.bin
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  flat2img -scanner=hrrt -zoom=2.5 square.bin square.v 1 1 100 100
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  e7evhdr square.v square.header
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  imgunit -us=kBq/ml square.v
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  esetstrt square.v 2013-03-07 10:37:34
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  imgcalc square.v x 0 empty.v
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  imgadd -x=5 temp1.v empty.v
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  imgadd -x=7 temp2.v temp1.v
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  imgadd -x=5 temp1.v square.v
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  imgadd -x=6 temp3.v temp1.v
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  imgadd cube.v temp2.v temp3.v temp2.v
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  img2tif -rb -s cube.v cube.tif
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi


if [ ! -f irregular.v ]; then
  printf "0 0 0 0 0 0 0 0\n"  > irregular.dat
  printf "0 0 1 3 0 0 0 0\n" >> irregular.dat
  printf "0 0 2 4 0 0 0 0\n" >> irregular.dat
  printf "0 0 0 0 0 0 0 0\n" >> irregular.dat

  printf "0 0 0 0 0 0 0 0\n" >> irregular.dat
  printf "0 0 1 3 0 0 0 0\n" >> irregular.dat
  printf "0 0 2 4 0 0 0 0\n" >> irregular.dat
  printf "0 0 0 0 0 0 0 0\n" >> irregular.dat

  printf "0 0 0 0 0 0 0 0\n" >> irregular.dat
  printf "0 0 1 3 0 0 0 0\n" >> irregular.dat
  printf "0 0 2 4 0 0 0 0\n" >> irregular.dat
  printf "0 0 0 0 0 0 0 0\n" >> irregular.dat

  printf "0 0 0 0 0 0 0 0\n" >> irregular.dat
  printf "0 0 1 3 0 0 0 0\n" >> irregular.dat
  printf "0 0 2 4 0 0 0 0\n" >> irregular.dat
  printf "0 0 0 0 0 0 0 0\n" >> irregular.dat

  printf "0 0 0 0 0 0 0 0\n" >> irregular.dat
  printf "0 0 1 3 0 0 0 0\n" >> irregular.dat
  printf "0 0 2 4 0 0 0 0\n" >> irregular.dat
  printf "0 0 0 0 0 0 0 0\n" >> irregular.dat

  printf "0 0 0 0 0 0 0 0\n" >> irregular.dat
  printf "0 0 1 3 0 0 0 0\n" >> irregular.dat
  printf "0 0 2 4 0 0 0 0\n" >> irregular.dat
  printf "0 0 0 0 0 0 0 0\n" >> irregular.dat

  printf "0 0 0 0 0 0 0 0\n" >> irregular.dat
  printf "0 0 1 3 0 0 0 0\n" >> irregular.dat
  printf "0 0 2 4 0 0 0 0\n" >> irregular.dat
  printf "0 0 0 0 0 0 0 0\n" >> irregular.dat

  asc2flat irregular.dat irregular.bin
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  flat2img -scanner=hrrt -zoom=2 irregular.bin irregular.v 7 1 8 4
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  imgswell -z=yes irregular.v temp.v
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  imgswell -z=yes temp.v temp2.v
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  imgswell -z=yes temp2.v temp.v
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  imgswell -z=yes temp.v temp2.v
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  imgswell -z=no temp2.v temp.v
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  imgswell -z=no temp.v irregular.v
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  imgdelpl irregular.v -1 0 
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  img2tif -rb -s irregular.v irregular.tif
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  imgdim irregular.v
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi


if [ ! -f zxy.v ] || [ ! -f zyx.v ]; then

  printf "x_pixel_size := 0.054167\n" > z.hdr
  printf "y_pixel_size := 0.08125\n" >> z.hdr
  printf "z_pixel_size := 0.08125\n" >> z.hdr

  printf "0 0 0 0 0 0\n"  > zxy.dat
  printf "0 0 0 0 0 0\n" >> zxy.dat
  printf "0 0 0 0 0 0\n" >> zxy.dat
  printf "0 0 0 0 0 0\n" >> zxy.dat

  printf "1 1 1 1 1 1\n" >> zxy.dat
  printf "1 1 1 1 1 1\n" >> zxy.dat
  printf "1 1 1 1 1 1\n" >> zxy.dat
  printf "1 1 1 1 1 1\n" >> zxy.dat

  printf "1 1 1 1 1 1\n" >> zxy.dat
  printf "1 1 1 1 1 1\n" >> zxy.dat
  printf "1 1 1 1 1 1\n" >> zxy.dat
  printf "1 1 1 1 1 1\n" >> zxy.dat

  printf "0 0 0 0 0 0\n" >> zxy.dat
  printf "0 0 0 0 0 0\n" >> zxy.dat
  printf "0 0 0 0 0 0\n" >> zxy.dat
  printf "0 0 0 0 0 0\n" >> zxy.dat

  asc2flat zxy.dat zxy.bin
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  flat2img -scanner=hrrt -zoom=2 zxy.bin zxy.v 4 1 6 4
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  imgswell -z=yes zxy.v temp.v
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  imgswell -z=yes temp.v temp2.v
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  imgswell -z=yes temp2.v temp.v
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  imgswell -z=yes temp.v temp2.v
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  imgswell -z=yes temp2.v zxy.v
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  img2tif -rb -s zxy.v zxy.tif
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  imgdim zxy.v
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi

  printf "0 0 0 0 0 0\n"  > zyx.dat
  printf "1 1 1 1 1 1\n" >> zyx.dat
  printf "1 1 1 1 1 1\n" >> zyx.dat
  printf "0 0 0 0 0 0\n" >> zyx.dat

  printf "0 0 0 0 0 0\n" >> zyx.dat
  printf "1 1 1 1 1 1\n" >> zyx.dat
  printf "1 1 1 1 1 1\n" >> zyx.dat
  printf "0 0 0 0 0 0\n" >> zyx.dat

  printf "0 0 0 0 0 0\n" >> zyx.dat
  printf "1 1 1 1 1 1\n" >> zyx.dat
  printf "1 1 1 1 1 1\n" >> zyx.dat
  printf "0 0 0 0 0 0\n" >> zyx.dat

  printf "0 0 0 0 0 0\n" >> zyx.dat
  printf "1 1 1 1 1 1\n" >> zyx.dat
  printf "1 1 1 1 1 1\n" >> zyx.dat
  printf "0 0 0 0 0 0\n" >> zyx.dat

  asc2flat zyx.dat zyx.bin
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  flat2img -scanner=hrrt -zoom=2 zyx.bin zyx.v 4 1 6 4
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  imgswell -z=yes zyx.v temp.v
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  imgswell -z=yes temp.v temp2.v
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  imgswell -z=yes temp2.v temp.v
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  imgswell -z=yes temp.v temp2.v
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  imgswell -z=yes temp2.v zyx.v
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  img2tif -rb -s zyx.v zyx.tif
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  imgdim zyx.v
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi

  e7evhdr zxy.v z.hdr
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  e7evhdr zxy.v z.hdr
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi


printf "\n\n Making comparison data based on smoothing in Octave \n\n"

if [ ! -f cross_octave_smoothed_4.v ]; then
  asc2flat cross_octave_smoothed_4.dat temp.bin
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  flat2img -scanner=hr+ -zoom=2.5 temp.bin temp.v 1 1 100 100
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  e7evhdr temp.v cross_octave.header
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  imgcalc temp.v x 100 cross_octave_smoothed_4.v
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  imgunit -us=kBq/ml cross_octave_smoothed_4.v
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  esetstrt cross_octave_smoothed_4.v 2009-12-18 14:17:04
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  img2tif -rb -s cross_octave_smoothed_4.v cross_octave_smoothed_4.tif
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi

if [ ! -f cross_octave_smoothed_8.v ]; then
  asc2flat cross_octave_smoothed_8.dat temp.bin
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  flat2img -scanner=hr+ -zoom=2.5 temp.bin temp.v 1 1 100 100
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  e7evhdr temp.v cross_octave.header
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  imgcalc temp.v x 100 cross_octave_smoothed_8.v
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  imgunit -us=kBq/ml cross_octave_smoothed_8.v
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  esetstrt cross_octave_smoothed_8.v 2009-12-28 13:31:00
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  img2tif -rb -s cross_octave_smoothed_8.v cross_octave_smoothed_8.tif
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi

if [ ! -f cross_tiny_octave_smoothed_2.v ]; then
  asc2flat cross_tiny_octave_smoothed_2.dat temp.bin
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  flat2img -scanner=hr+ -zoom=2.5 temp.bin temp.v 1 1 12 12
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  e7evhdr temp.v cross_octave.header
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  imgcalc temp.v x 100 cross_tiny_octave_smoothed_2.v
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  imgunit -us=kBq/ml cross_tiny_octave_smoothed_2.v
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  esetstrt cross_tiny_octave_smoothed_2.v 2009-12-21 11:58:32
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  img2tif -rb -s cross_tiny_octave_smoothed_2.v cross_tiny_octave_smoothed_2.tif
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi




printf "\n\n"
printf "=====================================================================\n"
printf "testing %s\n" $PROGRAM
printf "=====================================================================\n"

printf "\n 1.0.0 \n"
printf " Test case: Set SD to zero. \n"
printf " Expected result 1: Warning. \n"
printf " Expected result 2: Output image contents are the same as in input image. \n\n"

rm -f filtered.v stdout.txt
imgprofi -pxl=1,1,1 cross_tiny.v xo.pro yo.pro
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

$PROGRAM cross_tiny.v filtered.v 0 1>stdout.txt 2>&1
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
if [ ! -f filtered.v ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Warning" "with given SD no filtering is applied."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

imgprofi -pxl=1,1,1 filtered.v x.pro y.pro
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tac2svg -x1=-8 -x2=8 f0crossx.svg xo.pro x.pro
tac2svg -x1=-8 -x2=8 f0crossy.svg yo.pro y.pro

imgmatch cross_tiny.v filtered.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 1.1.0 \n"
printf " Test case 1: 2D image and 2D filtering. \n"
printf " Test case 2: Set SDxy to 8 pixels with default settings. \n"
printf " Expected result 1: Profile integrals are the same after smoothing. \n"
printf " Expected result 2: Correct output image as compared to Octave. \n\n"

rm -f cross_imgfiltg_8.v fir_cross_imgfiltg_8.tif
imgprofi -pxl=1,1,1 cross.v xo.pro yo.pro
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
imgprofi -pxl=1,1,1 cross_octave_smoothed_8.v xc.pro yc.pro
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

$PROGRAM cross.v cross_imgfiltg_8.v 8pxl
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
img2tif -rb -s cross_imgfiltg_8.v fir_cross_imgfiltg_8.tif
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

imgprofi -pxl=1,1,1 cross_imgfiltg_8.v x.pro y.pro
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tac2svg fir_crossx.svg xo.pro x.pro -l xc.pro
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tac2svg fir_crossy.svg yo.pro y.pro -l yc.pro
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

interpol -i -x=100 xo.pro test1.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
interpol -i -x=100 x.pro test2.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -rel=0.1 test1.dat test2.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n x passed. \n\n"

interpol -i -x=100 yo.pro test1.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
interpol -i -x=100 y.pro test2.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -rel=0.1 test1.dat test2.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n y passed. \n\n"

imgmatch -abs=0.25 -rel=0.1 cross_octave_smoothed_8.v cross_imgfiltg_8.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"




printf "\n 1.1.1 \n"
printf " Test case 1: 2D image and 2D filtering. \n"
printf " Test case 2: Set SD to 8 mm. \n"
printf " Test case 3: options to set pixel size and tolerance. \n"
printf " Expected result 1: Profile integrals are the same after smoothing. \n"
printf " Expected result 2: Correct output image as compared to Octave. \n\n"

rm -f cross_imgfiltg_8mm.v fir_cross_imgfiltg_8mm.tif
rm -f xmm.pro ymm.pro

$PROGRAM -method=FIR -xysize=1 -tol=1.0E-08 cross.v cross_imgfiltg_8mm.v 8mm
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
img2tif -rb -s cross_imgfiltg_8mm.v fir_cross_imgfiltg_8mm.tif
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

imgprofi -pxl=1,1,1 cross_imgfiltg_8mm.v xmm.pro ymm.pro 
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tac2svg fir_crossxmm.svg xo.pro x.pro -l xmm.pro 
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tac2svg fir_crossymm.svg yo.pro y.pro -l ymm.pro
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

interpol -i -x=100 xmm.pro test2.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -rel=0.1 test1.dat test2.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n x passed. \n\n"

interpol -i -x=100 ymm.pro test2.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -rel=0.1 test1.dat test2.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n y passed. \n\n"

imgmatch -abs=0.25 -rel=0.2 cross_octave_smoothed_8.v cross_imgfiltg_8mm.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 1.1.2 \n"
printf " Test case: Same, but using AM method. \n"
printf " Expected result: Correct output image as compared to Octave. \n\n"

rm -f cross_am_8mm.v am_cross_8mm.tif
rm -f xmm.pro ymm.pro

$PROGRAM -method=AM -xysize=1 -tolerance=1.0E-08 -steps=15 cross.v cross_am_8mm.v 8mm
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
img2tif -rb -s cross_am_8mm.v am_cross_8mm.tif
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

imgprofi -pxl=1,1,1 cross_am_8mm.v xmm.pro ymm.pro 
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tac2svg am_crossxmm.svg xo.pro x.pro -l xmm.pro 
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tac2svg am_crossymm.svg yo.pro y.pro -l ymm.pro
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

interpol -i -x=100 xmm.pro test2.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -rel=0.1 test1.dat test2.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n x passed. \n\n"

interpol -i -x=100 ymm.pro test2.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -rel=0.1 test1.dat test2.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n y passed. \n\n"

imgmatch -abs=1 -rel=0.2 cross_octave_smoothed_8.v cross_am_8mm.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"





printf "\n 1.1.2 \n"
printf " Test case: Same, but using undocumented EBox method. \n"
printf " Expected result: Not too bad as compared to Octave. \n\n"

rm -f cross_ebox_8mm.v ebox_cross_8mm.tif
rm -f xmm.pro ymm.pro

$PROGRAM -method=EBox -xysize=1 -steps=15 cross.v cross_ebox_8mm.v 8mm
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
img2tif -rb -s cross_ebox_8mm.v ebox_cross_8mm.tif
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

imgprofi -pxl=1,1,1 cross_ebox_8mm.v xmm.pro ymm.pro 
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tac2svg ebox_crossxmm.svg xo.pro x.pro -l xmm.pro 
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tac2svg ebox_crossymm.svg yo.pro y.pro -l ymm.pro
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

interpol -i -x=100 xmm.pro test2.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -rel=0.1 test1.dat test2.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n x passed. \n\n"

interpol -i -x=100 ymm.pro test2.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -rel=0.1 test1.dat test2.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n y passed. \n\n"

imgmatch -abs=5 -rel=1 cross_octave_smoothed_8.v cross_ebox_8mm.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"




printf "\n 2.0.0 \n"
printf " Test case 1: 3D image and 3D filtering. \n"
printf " Test case 2: Set SD to zero. \n"
printf " Expected result 1: Warning. \n"
printf " Expected result 2: Output image contents are the same as in input image. \n\n"

rm -f cube_filtered.v

$PROGRAM cube.v cube_filtered.v 0 0 1>stdout.txt 2>&1
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
if [ ! -f cube_filtered.v ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Warning" "with given SD no filtering is applied."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

imgmatch cube.v cube_filtered.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n 2.1.0 \n"
printf " Test case 1: 3D image and 3D filtering. \n"
printf " Test case 2: Reasonable SD. \n"
printf " Expected result: Profiles are the same from all directions. \n\n"

rm -f cube_filtered.v

$PROGRAM -steps=20 cube.v cube_filtered.v 5 5
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
img2tif -rb -s cube_filtered.v fir_cube_filtered.tif
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
imgmatch -abs=0.1 -rel=0.1 cube.v cube_filtered.v
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

imgflip -right cube_filtered.v cube_filtered_right.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
imgprofi -pxl=50,50,50 cube_filtered.v profilex.dat profiley.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
imgprofi -pxl=50,50,50 cube_filtered_right.v profilez.dat profiley.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
tacren profilex.dat 1 X__
tacren profiley.dat 1 Y__
tacren profilez.dat 1 Z__
tac2svg -legend=y -mt="Profiles of filtered cube" fir_cube_filtered_profiles.svg -l profilex.dat profiley.dat profilez.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -x=y -y=y -tacnames=n -xunit=y -yunit=y -abs=0.1 -rel=0.1 profilex.dat profilez.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -x=y -y=y -tacnames=n -xunit=y -yunit=y -abs=0.1 -rel=0.1 profiley.dat profilez.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n 2.1.1 \n"
printf " Test case 1: 3D image but 2D filtering. \n"
printf " Test case 2: Reasonable SD. \n"
printf " Expected result: Profiles are the same from all directions. \n\n"

rm -f cube_filtered_2d.v
imgprofi -pxl=1,1,1 cube.v cube_profile_x.dat cube_profile_y.dat cube_profile_z.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

$PROGRAM -steps=20 cube.v cube_filtered_2d.v 5
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
img2tif -rb -s cube_filtered_2d.v fir_cube_filtered_2d.tif
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
imgmatch -abs=0.1 -rel=0.1 cube.v cube_filtered.v
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
imgmatch -abs=0.1 -rel=0.1 cube_filtered.v cube_filtered_2d.v 
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

imgflip -right cube_filtered_2d.v cube_filtered_right.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
img2tif -rb -s cube_filtered_right.v fir_cube_filtered_2d_right.tif
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
imgprofi -pxl=50,50,50 cube_filtered_2d.v profilex.dat profiley.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
imgprofi -pxl=50,50,50 cube_filtered_right.v profilez.dat profiley.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
imgprofi -pxl=1,1,1 cube_filtered_2d.v temp.dat temp.dat profilez2d.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacren profilex.dat 1 X__
tacren profiley.dat 1 Y__
tacren profilez.dat 1 Z__
tac2svg -legend=y -mt="Profiles of filtered cube" fir_cube_filtered_2d_profiles.svg -l profilex.dat profiley.dat profilez.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -x=y -y=y -tacnames=n -xunit=y -yunit=y -abs=0.1 -rel=0.1 cube_profile_z.dat profilez2d.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -x=y -y=y -tacnames=n -xunit=y -yunit=y -abs=0.1 -rel=0.1 profilex.dat profilez.dat
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -x=y -y=y -tacnames=n -xunit=y -yunit=y -abs=0.1 -rel=0.1 profiley.dat profilez.dat
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"




printf "\n 2.1.2 \n"
printf " Test case 1: 3D image but filtering only in z dimension. \n"
printf " Test case 2: Reasonably low SD. \n"
printf " Expected result: . \n\n"

rm -f cube_filtered_z.v
imgprofi -pxl=50,50,50 cube.v oprofilex.dat oprofiley.dat oprofilez.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi

$PROGRAM cube.v cube_filtered_z.v 0 3
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
img2tif -rb -s cube_filtered_z.v cube_filtered_z.tif
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
imgmatch -abs=0.1 -rel=0.1 cube.v cube_filtered_z.v
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
imgmatch -abs=0.1 -rel=0.1 cube_filtered.v cube_filtered_z.v 
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

imgprofi -pxl=50,50,50 cube_filtered_z.v profilex.dat profiley.dat profilez.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -x=y -y=y -tacnames=n -xunit=y -yunit=y -abs=0.5 -rel=1 oprofilex.dat profilex.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -x=y -y=y -tacnames=n -xunit=y -yunit=y -abs=0.5 -rel=1 oprofiley.dat profiley.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -x=y -y=y -tacnames=n -xunit=y -yunit=y -abs=0.5 -rel=1 oprofilez.dat profilez.dat
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 2.2.0 \n"
printf " Test case 1: 3D image and 3D filtering. \n"
printf " Test case 2: Low SD. \n"
printf " Expected result: With low SD the mid profile is close to 2D filtering. \n\n"


rm -f cube_filtered3d.v cube_filtered2d.v

$PROGRAM cube.v cube_filtered3d.v 3 3
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
$PROGRAM cube.v cube_filtered2d.v 3
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

imgprofi -pxl=50,50,50 cube_filtered3d.v profile3dx.dat profile3dy.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
imgprofi -pxl=50,50,50 cube_filtered2d.v profile2dx.dat profile2dy.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacren profile3dx.dat 1 3D_3pxl_
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
tacren profile2dx.dat 1 2D_3pxl_
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
tac2svg -legend=y -mt="2D vs 3D profiles" cube_filtered_profiles_2D_vs_3D.svg -l profile3dx.dat profile2dx.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

tacmatch -abs=1.5 -rel=1 -x=y -y=y -tacnames=n -xunit=y -yunit=y profile3dx.dat profile2dx.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n 2.2.1 \n"
printf " Test case 1: 3D image. \n"
printf " Test case 2: Irregular image dimensions. \n"
printf " Expected result 1: No error. \n\n"
printf " Expected result 2: With low SD the mid profile is close to 2D filtering. \n\n"

rm -f irregular_filtered3d.v irregular_filtered2d.v

$PROGRAM irregular.v irregular_filtered3d.v 6 2
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
img2tif -rb -s irregular_filtered3d.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
$PROGRAM irregular.v irregular_filtered2d.v 6
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

imgprofi -pxl=128,64,55 irregular_filtered3d.v profile3dx.dat profile3dy.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
imgprofi -pxl=128,64,55 irregular_filtered2d.v profile2dx.dat profile2dy.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacren profile3dx.dat 1 3D_6pxl_
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
tacren profile2dx.dat 1 2D_6pxl_
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
tac2svg -legend=y -mt="2D vs 3D profiles" irregular_filtered_profiles_2D_vs_3D.svg -l profile3dx.dat profile2dx.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

tacmatch -abs=0.02 -rel=1 -x=y -y=y -tacnames=n -xunit=y -yunit=y profile3dx.dat profile2dx.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.02 -rel=1 -x=y -y=y -tacnames=n -xunit=y -yunit=y profile3dy.dat profile2dy.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n 2.3.0 \n"
printf " Test case 1: 3D image and 3D filtering. \n"
printf " Test case 2: z dimension. \n"
printf " Expected result: Same result with z y swapped image. \n\n"

rm -f zxy_filtered3d.v zyx_filtered32.v

$PROGRAM zxy.v zxy_filtered3d.v 6 6
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
img2tif -rb -s zxy_filtered3d.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
$PROGRAM zyx.v zyx_filtered3d.v 6 6
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
img2tif -rb -s zyx_filtered3d.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

imgprofi -pxl=96,64,64 zxy_filtered3d.v zxy_profile3dx.dat zxy_profile3dy.dat zxy_profile3dz.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
imgprofi -pxl=96,64,64 zyx_filtered3d.v zyx_profile3dx.dat zyx_profile3dy.dat zyx_profile3dz.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
imgprofi -pxl=96,64,64 zxy.v oprofile3dx.dat oprofile3dy.dat oprofile3dz.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacren zxy_profile3dz.dat 1 3D_zxy_
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
tacren zyx_profile3dy.dat 1 3D_zyx_
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
tacren oprofile3dz.dat 1 Orig
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
tac2svg -legend=y -mt="ZXY vs ZYX profiles" filtered_profiles_ZXY_vs_ZYX.svg -l oprofile3dz.dat zxy_profile3dz.dat zyx_profile3dy.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

tacmatch -abs=0.01 -rel=1 -x=y -y=y -tacnames=n -xunit=y -yunit=y zxy_profile3dz.dat zyx_profile3dy.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n Test case: Same works with high smoothing. \n"

$PROGRAM zxy.v zxy_filtered3d_high.v 20 20
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
img2tif -rb -s zxy_filtered3d_high.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
$PROGRAM zyx.v zyx_filtered3d_high.v 20 20
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
img2tif -rb -s zyx_filtered3d_high.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

imgprofi -pxl=96,64,64 zxy_filtered3d_high.v zxy_profile3dx.dat zxy_profile3dy.dat zxy_profile3dz.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
imgprofi -pxl=96,64,64 zyx_filtered3d_high.v zyx_profile3dx.dat zyx_profile3dy.dat zyx_profile3dz.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacren zxy_profile3dz.dat 1 3D_ZXY_
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
tacren zyx_profile3dy.dat 1 2D_ZYX_
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
tac2svg -legend=y -mt="ZXY vs ZYX profiles" filtered_profiles_ZXY_vs_ZYX_high.svg -l oprofile3dz.dat zxy_profile3dz.dat zyx_profile3dy.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

tacmatch -abs=0.01 -rel=1 -x=y -y=y -tacnames=n -xunit=y -yunit=y zxy_profile3dz.dat zyx_profile3dy.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"




printf "\n===================================================================\n"



printf "\n 5.0.0 \n"
printf " Test case: ECAT 6.3 format. \n"
printf " Expected result: Same as with ECAT 7. \n\n"

rm -f filtgecat63.img
$PROGRAM dynamic.v dynamic_filtered.v 10
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

$PROGRAM dynamic.img filtgecat63.img 10
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

imgmatch -abs=0.1 -rel=0.1 dynamic_filtered.v filtgecat63.img
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 5.1.0 \n"
printf " Test case: Analyze format. \n"
printf " Expected result: Same as with ECAT 7. \n\n"

rm -f ana/filtgana*

$PROGRAM ana/dynamic ana/filtgana 10
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

imgmatch -abs=0.2 -rel=0.2 dynamic_filtered.v ana/filtgana
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 5.2.0 \n"
printf " Test case: NIfTI format. \n"
printf " Expected result: Same as with ECAT 7. \n\n"

rm -f nii/filtgnii*

$PROGRAM nii/dynamic nii/filtgnii 10
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

imgmatch -abs=0.2 -rel=0.2 dynamic_filtered.v nii/filtgnii
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 5.3.0 \n"
printf " Test case: Static ECAT 7 format. \n"
printf " Expected result: No error. \n\n"

rm -f static_filtered.v

$PROGRAM static.v static_filtered.v 5pxl
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
img2tif -rb -s static_filtered.v static_filtered.tif
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"




printf "\n===================================================================\n"




printf "\n 6.0.0 \n"
printf " Test case: Invalid option. \n"
printf " Expected result: Error. \n\n"

rm -f output.v
$PROGRAM -stupidoption static.v output.v 5 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error "invalid option '-stupidoption'."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n 6.0.1 \n"
printf " Test case: Extra argument. \n"
printf " Expected result: Error. \n\n"

$PROGRAM static.v output.v 5 0 extra 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error "too many arguments."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n 6.0.2 \n"
printf " Test case: Missing argument. \n"
printf " Expected result: Error. \n\n"

$PROGRAM  output.v 5 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Error"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

$PROGRAM static.v  0  1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Error"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

$PROGRAM static.v 2 2  1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Error"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

$PROGRAM static.v output.v 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Error"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 6.1.0 \n"
printf " Test case: Input file does not exist. \n"
printf " Expected result: Error. \n\n"

$PROGRAM nonexistingfile.v output.v 5 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Error"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 6.2.0 \n"
printf " Test case: Trying to do 3D smoothing for 2D image. \n"
printf " Expected result: Error. \n\n"

$PROGRAM cross.v output.v 5 5 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Error" "invalid Z dimension for 3D filtering."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n===================================================================\n"
printf "   All passed! \n"
printf "===================================================================\n"
exit 0


