#!/bin/bash
#: Title      : test_imginteg
#: Date       : 2023-06-29
#: Author     : "Vesa Oikonen" <vesa.oikonen@utu.fi>
#: Options    : None

# Set the name of executable to test
if [ -n "${OS+1}" ] && [ "$OS" = "Windows_NT" ]; then
EXT=.exe;
else
EXT=;
fi
PROGRAM=../../imginteg$EXT;
if [ ! -f $PROGRAM ]; then
  printf "Failed: executable does not exist.\n"
  exit 1
fi


printf "=====================================================================\n"
printf "creating test data for %s\n" $PROGRAM
printf "=====================================================================\n"


printf "\n verifying that required data exists \n"

if [ ! -f test_dyn.dft ] || [ ! -f test_dyn2.dft ]; then
  printf "Failed: required data does not exist.\n"
  exit 1
fi
if [ ! -f test_arg1.dft ] || [ ! -f test_arg2.dft ]; then
  printf "Failed: required data does not exist.\n"
  exit 1
fi
if [ ! -f test_largegap.dft ] || [ ! -f test_smallgap.dft ]; then
  printf "Failed: required data does not exist.\n"
  exit 1
fi
if [ ! -f test_largeoverlap.dft ] || [ ! -f test_smalloverlap.dft ]; then
  printf "Failed: required data does not exist.\n"
  exit 1
fi
if [ ! -f test_latescan.dft ]; then
  printf "Failed: required data does not exist.\n"
  exit 1
fi

if [ ! -f correct_integral1.dft ] || [ ! -f correct_integral2.dft ]; then
  printf "Failed: required data does not exist.\n"
  exit 1
fi
if [ ! -f correct_integral8a.dft ]; then
  printf "Failed: required data does not exist.\n"
  exit 1
fi


printf "\n making test data \n"

if [ ! -f dynamic.v ] || [ ! -f dynamic.img ]; then
  dft2img -scanner=HR+ -zoom=2 -dim=8 imginteg.dft dynamic.v dynamic.roi
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  e7to63 dynamic.v dynamic.img
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi
if [ ! -f twoframe.v ]; then
  dft2img -scanner=HR+ -zoom=2 -dim=8 imginteg2.dft twoframe.v twoframe.roi
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi
if [ ! -f static.v ]; then
  dft2img -scanner=HR+ -zoom=2 -dim=8 imginteg3.dft static.v static.roi
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi

if [ ! -f test_dyn.v ] || [ ! -f test_dyn.img ]; then
  dft2img -scanner=HR+ -zoom=2 -dim=8 test_dyn.dft test_dyn.v test_dyn.roi
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  e7to63 test_dyn.v test_dyn.img
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi
if [ ! -f test_dyn2.v ]; then
  dft2img -scanner=HR+ -zoom=2 -dim=8 test_dyn2.dft test_dyn2.v test_dyn2.roi
fi
if [ ! -f test_arg1.v ]; then
  dft2img -scanner=HR+ -zoom=2 -dim=8 test_arg1.dft test_arg1.v test_arg1.roi
fi
if [ ! -f test_arg2.v ]; then
  dft2img -scanner=HR+ -zoom=2 -dim=8 test_arg2.dft test_arg2.v test_arg2.roi
fi
if [ ! -f test_largegap.v ]; then
  dft2img -scanner=HR+ -zoom=2 -dim=8 test_largegap.dft test_largegap.v
fi
if [ ! -f test_largeoverlap.v ]; then
  dft2img -scanner=HR+ -zoom=2 -dim=8 test_largeoverlap.dft test_largeoverlap.v
fi
if [ ! -f test_smallgap.v ]; then
  dft2img -scanner=HR+ -zoom=2 -dim=8 test_smallgap.dft test_smallgap.v test_smallgap.roi
fi
if [ ! -f test_smalloverlap.v ]; then
  dft2img -scanner=HR+ -zoom=2 -dim=8 test_smalloverlap.dft test_smalloverlap.v test_smalloverlap.roi
fi
if [ ! -f test_latescan.v ]; then
  dft2img -scanner=HR+ -zoom=2 -dim=8 test_latescan.dft test_latescan.v test_latescan.roi
fi



printf "\n\n"
printf "=====================================================================\n"
printf "testing %s\n" $PROGRAM
printf "=====================================================================\n"


printf "\n 0.1.0 \n"
printf " Test case: No command line options or arguments. \n"
printf " Expected result: Error. \n\n"

$PROGRAM 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Usage"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "See also"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n 0.1.1 \n"
printf " Test case: Unknown option. \n"
printf " Expected result: Error. \n\n"

$PROGRAM -stupidoption test_dyn.v 0 0 output.v 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error "invalid option '-stupidoption'."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n 0.1.2 \n"
printf " Test case: Too many arguments. \n"
printf " Expected result: Error. \n\n"

$PROGRAM test_dyn.v 0 0 output.v extra 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error "invalid argument 'extra'."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n 0.1.3 \n"
printf " Test case: Missing argument. \n"
printf " Expected result: Error. \n\n"

$PROGRAM  0 0 output.v 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

$PROGRAM test_dyn.v  0 output.v 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

$PROGRAM test_dyn.v 0 0   1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n===================================================================\n"
printf "   Basic tests \n"
printf "===================================================================\n"

printf "\n 1.0.0 \n"
printf " Test case: AUC20-30 from ECAT6 image. \n"
printf " Expected result: Correct result. \n\n"

rm -f integral1.img integral1.dft
$PROGRAM dynamic.img 20 10 integral1.img
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
img2dft integral1.img dynamic.roi integral1.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -x=y -y=y -xunit=y -yunit=y -abs=0.01 -rel=0.1 correct_integral1.dft integral1.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n 1.0.1 \n"
printf " Test case: Mean20-30 from ECAT6 image. \n"
printf " Expected result: Correct result. \n\n"

rm -f integral1b.img integral1b.dft
$PROGRAM -avg dynamic.img 20 10 integral1b.img
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
img2dft integral1b.img dynamic.roi integral1b.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
taccalc integral1b.dft x 10 temp.dft
tacmatch -x=y -y=y -xunit=y -yunit=n -abs=0.01 -rel=0.1 correct_integral1.dft temp.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n-----------------------------------------------------------------\n\n"

printf "\n 1.1.0 \n"
printf " Test case: AUC0-90 from ECAT7 image. \n"
printf " Expected result: Correct result. \n\n"

rm -f integral2.v integral2.dft
$PROGRAM dynamic.v 0 90 integral2.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
img2dft integral2.v dynamic.roi integral2.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -x=y -y=y -xunit=y -yunit=y -abs=0.01 -rel=0.1 correct_integral2.dft integral2.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n-----------------------------------------------------------------\n\n"

printf "\n 1.2.0 \n"
printf " Test case: AUC25-50 from Analyze image with SIF. \n"
printf " Expected result: Correct result. \n\n"

rm -f ana/dynamic* ana/integral3* integral3.dft
ecat2ana -sif -o=ana dynamic.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

$PROGRAM ana/dynamic 25 25 ana/integral3.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
img2dft ana/integral3.v dynamic.roi integral3.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -x=y -y=y -xunit=y -yunit=n -abs=0.01 -rel=0.1 correct_integral3.dft integral3.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n-----------------------------------------------------------------\n\n"

printf "\n 1.3.0 \n"
printf " Test case: AUC outside ECAT7 frames. \n"
printf " Expected result: Error. \n\n"

$PROGRAM dynamic.v 60 30 failed.v 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error "integration time range oversteps data range."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

$PROGRAM dynamic.v 0 180 failed.v 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error "integration time range oversteps data range."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n-----------------------------------------------------------------\n\n"

printf "\n 1.4.0 \n"
printf " Test case: AUC from static image with appropriate time range. \n"
printf " Expected result: Correct result. \n\n"

rm -f staticout.v
$PROGRAM integral2.v 0 90 staticout.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
imgcalc integral2.v x 90 staticout2.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
imgmatch -abs=0.1 -rel=0.1 staticout2.v staticout.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 1.4.1 \n"
printf " Test case: AUC from static image with inappropriate time range. \n"
printf " Expected result: Error. \n\n"

$PROGRAM integral2.v 0 30 failed.v 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error "for static image the integration time range must be exactly as long as the scan."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 1.4.2 \n"
printf " Test case: AUC from static image where 1st frame starts later than 0. \n"
printf " Expected result: Correct result, if correct time range is given. \n\n"

rm -f output.v
$PROGRAM static.v 39 90 output.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
imgcalc static.v x 90 temp.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
imgmatch -abs=0.1 -rel=0.1 temp.v output.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 1.4.3 \n"
printf " Test case: AUC from static image where 1st frame starts later than 0. \n"
printf " Expected result: Correct result, when zeroes as time range. \n\n"

rm -f output.v
$PROGRAM static.v 0 0 output.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
imgmatch -abs=0.1 -rel=0.1 temp.v output.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n-----------------------------------------------------------------\n\n"

printf "\n 1.5.0 \n"
printf " Test case: AUC from two-frame image where 1st frame starts later than 0. \n"
printf " Expected result: Correct result with exact time range for 1st frame. \n\n"

rm -f output.v
$PROGRAM twoframe.v 39 90 output.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
imgmatch -abs=0.1 -rel=0.1 temp.v output.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n 1.5.1 \n"
printf " Test case: AUC from two-frame image where 1st frame starts later than 0. \n"
printf " Expected result: Correct result with exact time range for both frames. \n\n"

rm -f output.v output.dft
$PROGRAM twoframe.v 39 120 output.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
img2dft output.v twoframe.roi output.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -x=y -y=y -xunit=y -abs=0.1 -rel=0.1 correct_integral8a.dft output.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n 1.5.2 \n"
printf " Test case: AUC from two-frame image where 1st frame starts later than 0. \n"
printf " Expected result: Correct result with zeroes as time range. \n\n"

rm -f output.v output.dft
$PROGRAM twoframe.v 0 0 output.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
img2dft output.v twoframe.roi output.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -x=y -y=y -xunit=y -abs=0.1 -rel=0.1 correct_integral8a.dft output.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"




printf "\n===================================================================\n"
printf "   Image formats \n"
printf "===================================================================\n"

printf "\n 2.0.0 \n"
printf " Test case: Image format ECAT7. \n"
printf " Expected result: Correct result with zeroes as time range. \n\n"

rm -f output.v
cp -f test_dyn.dft temp.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
tacunit -xconv=sec temp.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
dftinteg temp.dft 0 0 test.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
tacunit -xconv=min test.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

$PROGRAM --silent test_dyn.v 0 0 output.v 1>stdout.txt 2>&1
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
img2dft output.v test_dyn.roi output.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.01 -rel=0.1 test.dft output.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n 2.0.1 \n"
printf " Test case: Image format ECAT6. \n"
printf " Expected result: Correct result with zeroes as time range. \n\n"

rm -f output.img
$PROGRAM --silent test_dyn.img 0 0 output.img
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
imgmatch output.v output.img
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 2.1.0 \n"
printf " Test case: Image format Analyze with SIF. \n"
printf " Expected result: Correct result with zeroes as time range. \n\n"

rm -f ana/* output.v output3.dft
ecat2ana -sif -o=ana test_dyn.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

$PROGRAM --silent ana/test_dyn 0 0 ana/output
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
ana2ecat ana/output
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
img2dft output.v test_dyn.roi output3.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -x=n output.dft output3.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 2.1.1 \n"
printf " Test case: Image format Analyze without SIF. \n"
printf " Expected result: Error. \n\n"

rm -f ana/*.sif
$PROGRAM --silent ana/test_dyn 0 0 ana/output 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Error" "image does not contain frame times."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"





printf "\n 2.2.0 \n"
printf " Test case: Image format NIfTI with SIF. \n"
printf " Expected result: Correct result with zeroes as time range. \n\n"

rm -f nii/* output.v output3.dft
ecat2nii -sif -o=nii test_dyn.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

$PROGRAM --silent nii/test_dyn 0 0 nii/output
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
nii2ecat nii/output
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
img2dft output.v test_dyn.roi output3.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -x=n output.dft output3.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 2.2.1 \n"
printf " Test case: Image format NIfTI without SIF. \n"
printf " Expected result: Error. \n\n"

rm -f nii/*.sif
$PROGRAM --silent nii/test_dyn 0 0 nii/output 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Error" "image does not contain frame times."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"




printf "\n===================================================================\n"
printf "   Time frames and range \n"
printf "===================================================================\n"

printf "\n 3.0.0 \n"
printf " Test case: AUC from full time frames. \n"
printf " Expected result: Correct result. \n\n"

rm -f output.v output.dft
cp -f test_dyn.dft temp.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
tacunit -xconv=sec temp.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
dftinteg temp.dft 180 720 test.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

$PROGRAM test_dyn.v 180 720 output.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
img2dft output.v test_dyn.roi output.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.001 -rel=0.1 test.dft output.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 3.1.0 \n"
printf " Test case: AUC from partial frames. \n"
printf " Expected result: Correct result. \n\n"

rm -f output.v output.dft
cp -f test_dyn.dft temp.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
tacunit -xconv=sec temp.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
dftinteg temp.dft 150 750 test.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

$PROGRAM test_dyn.v 150 750 output.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
img2dft output.v test_dyn.roi output.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.001 -rel=0.1 test.dft output.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 3.1.1 \n"
printf " Test case: AUC from partial frames. \n"
printf " Expected result: Correct result. \n\n"

rm -f output.v output.dft
cp -f test_dyn.dft temp.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
tacunit -xconv=sec temp.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
dftinteg temp.dft 300 660 test.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

$PROGRAM test_dyn.v 300 660 output.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
img2dft output.v test_dyn.roi output.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.001 -rel=0.1 test.dft output.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 3.1.2 \n"
printf " Test case: AUC from partial frames. \n"
printf " Expected result: Correct result. \n\n"

rm -f output.v output.dft
cp -f test_dyn.dft temp.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
tacunit -xconv=sec temp.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
dftinteg temp.dft 480 240 test.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

$PROGRAM test_dyn.v 480 240 output.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
img2dft output.v test_dyn.roi output.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.001 -rel=0.1 test.dft output.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 3.2.0 \n"
printf " Test case: Small gap between time frames. \n"
printf " Expected result: Gap is ignored. \n\n"

rm -f output.v output2.v
$PROGRAM test_dyn.v 60 840 output.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
$PROGRAM test_smallgap.v 60 840 output2.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
imgmatch -roughly output.v output2.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 3.2.1 \n"
printf " Test case 1: Large gap between time frames. \n"
printf " Test case 2: AUC times given in minutes. \n"
printf " Expected result: Gap is filled, result is okeyish. \n\n"

rm -f output3.v
$PROGRAM -min test_largegap.v 1 14 output3.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
imgmatch -around output.v output3.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n 3.3.0 \n"
printf " Test case: Small overlap between time frames. \n"
printf " Expected result: Gap is ignored. \n\n"

rm -f output4.v
$PROGRAM -min test_smalloverlap.v 1 14 output4.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
imgmatch -roughly output.v output4.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 3.3.1 \n"
printf " Test case: Large overlap between time frames. \n"
printf " Expected result: Error. \n\n"

$PROGRAM test_largeoverlap.v 60 840 failed.v 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Error" "image test_largeoverlap.v has overlapping frame times."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"




printf "\n 3.4.0 \n"
printf " Test case: AUC from zero to certain end time before end of image data. \n"
printf " Expected result: Correct result \n\n"

rm -f output.v output.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
cp -f test_dyn.dft temp.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
tacunit -xconv=sec temp.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
dftinteg temp.dft 0 900 test.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
tacunit -xconv=min test.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

$PROGRAM test_dyn.v 0 900 output.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
img2dft output.v test_dyn.roi output.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.01 -rel=0.1 test.dft output.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 3.4.1 \n"
printf " Test case: AUC from zero to end time a bit over end of image data. \n"
printf " Expected result: Correct result \n\n"

rm -f output.v output.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
cp -f test_dyn.dft temp.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
tacunit -xconv=sec temp.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
dftinteg temp.dft 0 2400 test.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
tacunit -xconv=min test.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

$PROGRAM test_dyn.v 0 2400 output.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
img2dft output.v test_dyn.roi output.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.01 -rel=0.1 test.dft output.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 3.4.2 \n"
printf " Test case: AUC from zero to end time much over the end of image data. \n"
printf " Expected result: Error. \n\n"

$PROGRAM test_dyn.v 0 3600 failed.v 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Error" "integration time range oversteps data range."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n 3.5.0 \n"
printf " Test case: AUC from zero when image starts a bit later. \n"
printf " Expected result: Initial part is extrapolated, correct result. \n\n"

rm -f output.v output.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
cp -f test_dyn2.dft temp.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
tacunit -xconv=sec temp.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
dftinteg temp.dft 0 900 test.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
tacunit -xconv=min test.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

$PROGRAM -min test_dyn2.v 0 15 output.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
img2dft output.v test_dyn2.roi output.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.01 -rel=0.1 test.dft output.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 3.5.1 \n"
printf " Test case: AUC from zero with late scan image. \n"
printf " Expected result: Error. \n\n"

$PROGRAM -min test_latescan.v 0 60 failed.v 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Error" "integration time range oversteps data range."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 3.5.2 \n"
printf " Test case: time range much after the image data ends. \n"
printf " Expected result: Error. \n\n"

$PROGRAM -avg test_dyn.v 7200 3600 failed.v 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Error" "integration time range oversteps data range."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n-----------------------------------------------------------------\n\n"

printf "\n 3.6.0 \n"
printf " Test case: Average of dynamic data between two late times. \n"
printf " Expected result: Initial part is extrapolated, correct result. \n\n"

rm -f output.v output.dft
cp -f test_dyn.dft temp.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
tacunit -xconv=sec temp.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
dftinteg -avg temp.dft 600 900 test.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
tacunit -xconv=min test.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

$PROGRAM -avg test_dyn.v 600 900 output.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
img2dft output.v test_dyn.roi output.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.01 -rel=0.1 test.dft output.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 3.6.1 \n"
printf " Test case: Average from late scan, letting program determine time range. \n"
printf " Expected result: Correct result. \n\n"

rm -f output.v output.dft
cp -f test_latescan.dft temp.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
tacunit -xconv=sec temp.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
dftinteg -avg temp.dft 0 0 test.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
tacunit -xconv=min test.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

$PROGRAM -avg test_latescan.v 0 0 output.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
img2dft output.v test_latescan.roi output.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.01 -rel=0.1 test.dft output.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 3.6.2 \n"
printf " Test case: Average from late scan, letting program determine duration. \n"
printf " Expected result: Correct result. \n\n"

rm -f output.v output.dft
cp -f test_latescan.dft temp.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
tacunit -xconv=sec temp.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
dftinteg -avg temp.dft 3000 600 test.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
tacunit -xconv=min test.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

$PROGRAM -avg -min test_latescan.v 50 0 output.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
img2dft output.v test_latescan.roi output.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.01 -rel=0.1 test.dft output.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"





printf "\n===================================================================\n"
printf "   Usage in ARG method \n"
printf "===================================================================\n"



printf "\n 4.0.0 \n"
printf " Test case: Integral from static study with user-defined time range. \n"
printf " Expected result: Correct result. \n\n"

rm -f output.v output.dft
dftinteg test_arg1.dft 0 90 test.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

$PROGRAM test_arg1.v 0 90 output.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
img2dft output.v test_arg1.roi output.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.01 -rel=0.1 test.dft output.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 4.0.1 \n"
printf " Test case: Integral from static study with automatic time range. \n"
printf " Expected result: Correct result. \n\n"

rm -f output.v output.dft

$PROGRAM test_arg1.v 0 0 output.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
img2dft output.v test_arg1.roi output.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.01 -rel=0.1 test.dft output.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 4.1.0 \n"
printf " Test case: Static study does not start from zero time. Automatic time range. \n"
printf " Expected result: Correct result. \n\n"

rm -f output.v output.dft
dftinteg test_arg2.dft 0 0 test.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

$PROGRAM test_arg2.v 0 0 output.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
img2dft output.v test_arg1.roi output.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.01 -rel=0.1 test.dft output.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n===================================================================\n"
printf "   All passed! \n"
printf "===================================================================\n"
exit 0


