#!/bin/bash
#: Title      : test_imgthrs
#: Date       : 2023-06-29
#: Author     : "Vesa Oikonen" <vesa.oikonen@utu.fi>
#: Options    : None

# Set the name of executable to test
if [ -n "${OS+1}" ] && [ "$OS" = "Windows_NT" ]; then
EXT=.exe;
else
EXT=;
fi
PROGRAM=../../imgthrs$EXT;
if [ ! -f $PROGRAM ]; then
  printf "Failed: executable does not exist.\n"
  exit 1
fi


printf "=====================================================================\n"
printf "creating test data for %s\n" $PROGRAM
printf "=====================================================================\n"


printf "\n verifying that required data exists \n"

if [ ! -f test_dyn.tac ]; then
  printf "Failed: required data does not exist.\n"
  exit 1
fi


printf "\n making test data \n"

if [ ! -f dynamic.v ] || [ ! -f dynamic.img ]; then
  rm -f static.v
  
  dft2img -scanner=HR+ -zoom=2 -dim=8 test_dyn.tac dynamic.v dynamic.roi
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  e7to63 dynamic.v dynamic.img
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi

  rm -f ana/dynamic.*
  rmdir ana
  ecat2ana -sif -o=ana dynamic.v
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi

  rm -f nifti/dynamic.*
  rmdir nifti
  ecat2nii -sif -o=nifti dynamic.v
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi

if [ ! -f static.v ] || [ ! -f static.img ]; then
  imginteg -avg dynamic.v 0 0 static.v
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  imginteg -avg dynamic.img 0 0 static.img
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  imginteg -avg ana/dynamic 0 0 ana/static
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  imginteg -avg nifti/dynamic 0 0 nifti/static
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi



if [ ! -f mask.dat ] || [ ! -f mask.v ]; then
rm -f mask.bin mask.v
printf "0 0 0 0 0 1 1 2 2 2 3 3 4 5 5 7\n"  > mask.dat
printf "0 0 0 0 0 1 1 2 2 3 3 4 5 5 6 7\n" >> mask.dat
printf "0 0 0 0 0 1 1 2 3 3 3 4 5 5 6 7\n" >> mask.dat
printf "0 0 0 0 1 1 1 2 3 3 3 4 5 6 6 7\n" >> mask.dat
printf "0 0 0 0 1 1 2 2 3 3 4 4 5 6 6 7\n" >> mask.dat
printf "0 0 0 0 1 1 2 3 3 3 4 4 6 6 6 7\n" >> mask.dat
printf "0 0 0 1 1 1 2 3 3 4 4 5 6 6 6 8\n" >> mask.dat
printf "0 0 0 1 1 2 2 3 3 4 4 5 6 6 7 8\n" >> mask.dat
printf "0 0 0 1 1 2 2 3 3 4 4 5 6 7 7 8\n" >> mask.dat
printf "0 0 1 1 1 2 2 3 3 4 4 5 6 7 7 8\n" >> mask.dat
printf "0 0 1 1 1 2 2 3 3 4 4 5 7 7 7 8\n" >> mask.dat
printf "0 0 1 1 1 3 3 3 3 4 4 5 7 7 7 8\n" >> mask.dat
printf "0 1 1 1 2 3 3 3 4 4 5 6 7 7 7 8\n" >> mask.dat
printf "0 1 1 1 2 3 3 4 4 4 5 6 7 7 8 8\n" >> mask.dat
printf "0 1 1 1 2 3 3 4 4 4 5 6 7 7 8 9\n" >> mask.dat
printf "1 1 1 2 2 3 3 4 4 5 5 6 7 8 8 9\n" >> mask.dat
fi

if [ ! -f mask.bin ] || [ ! -f mask.v ]; then
  asc2flat mask.dat mask.bin
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  flat2img -scanner=HRRT mask.bin mask.v 1 1 16 16 
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi


if [ ! -f lower.lim ] || [ ! -f upper.lim ]; then
  printf "90\n"  > lower.lim
  printf "100\n" > upper.lim
fi

if [ ! -f limits.ift ]; then
  printf "lower_limit :=  90\n"  > limits.ift
  printf "upper_limit := 100\n" >> limits.ift
fi




printf "\n\n"
printf "=====================================================================\n"
printf "testing %s\n" $PROGRAM
printf "=====================================================================\n"



printf "\n 1.0.0 \n"
printf " Test case 1: Threshold limits 0 and 90. \n"
printf " Test case 2: Dynamic ECAT 7 format. \n"
printf " Expected result: Correct thresholded image saved. \n\n"

rm -f output.v output.tac
$PROGRAM dynamic.v 0 90 output.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

img2dft output.v dynamic.roi output.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

tacadd -ovr temp1.tac test_dyn.tac roi1
tacadd -ovr temp2.tac output.tac roi1
tacmatch -abs=0.1 -rel=0.1 temp1.tac temp2.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

taccalc temp1.tac x 0 temp1.tac
tacadd -ovr temp2.tac output.tac roi2
tacmatch -abs=0.1 -rel=0.1 temp1.tac temp2.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

tacadd -ovr temp2.tac output.tac roi3
tacmatch -abs=0.1 -rel=0.1 temp1.tac temp2.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

tacadd -ovr temp1.tac test_dyn.tac roi4
tacadd -ovr temp2.tac output.tac roi4
tacmatch -abs=0.1 -rel=0.1 temp1.tac temp2.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 1.0.1 \n"
printf " Test case 1: Threshold limits 0 and 90. \n"
printf " Test case 2: Dynamic ECAT 6.3 format. \n"
printf " Expected result: Correct thresholded image saved. \n\n"

rm -f output.img
$PROGRAM dynamic.img 0 90 output.img
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
imgmatch -rel=1 -abs=0.1 output.v output.img
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 1.0.2 \n"
printf " Test case 1: Threshold limits 0 and 90. \n"
printf " Test case 2: Dynamic NIfTI format. \n"
printf " Expected result: Correct thresholded image saved. \n\n"

rm -f nifti/output.nii
$PROGRAM nifti/dynamic 0 90 nifti/output
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
imgmatch -rel=1 -abs=0.1 output.v nifti/output
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 1.0.3 \n"
printf " Test case 1: Threshold limits 0 and 90. \n"
printf " Test case 2: Dynamic Analyze 7.5 format. \n"
printf " Expected result: Correct thresholded image saved. \n\n"

rm -f ana/output.img ana/output.hdr ana/output.sif
$PROGRAM ana/dynamic 0 90 ana/output
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
imgmatch -rel=1 -abs=0.1 output.v ana/output
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n===================================================================\n"

printf "\n 1.1.0 \n"
printf " Test case 1: Mask image can be saved. \n"
printf " Test case 2: Thresholded image needs not to be saved. \n"
printf " Expected result: Mask image can be used to do thresholding. \n\n"

rm -f outmask.v
$PROGRAM -mask=outmask.v dynamic.v 0 90
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
cp -f dynamic.v temp.v
imgmask temp.v outmask.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
imgmatch -rel=1 -abs=0.1 output.v temp.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n===================================================================\n"

printf "\n 1.2.0 \n"
printf " Test case 1: Sum image can be thresholded. \n"
printf " Test case 2: Both mask and thresholded image are saved. \n"
printf " Expected result: Mask and thresholded sum image are correct. \n\n"

rm -f outmask.v sumoutput.v
$PROGRAM -mask=outmask.v static.v 0 90 sumoutput.v 
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

imginteg -avg output.v 0 0 temp.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
imgmatch -rel=1 -abs=0.1 temp.v sumoutput.v 
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

cp -f static.v temp.v
imgmask temp.v outmask.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
imgmatch -rel=1 -abs=0.1 sumoutput.v temp.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n===================================================================\n"

printf "\n 2.0.0 \n"
printf " Test case 1: Integral end time given. \n"
printf " Test case 2: Threshold limits 90 and 100. \n"
printf " Expected result: Thresholded image is correct. \n\n"

rm -f output.v
$PROGRAM -end=3 dynamic.v 90 100 output.v 
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

img2dft output.v dynamic.roi output.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

tacadd -ovr temp1.tac test_dyn.tac roi4
tacadd -ovr temp2.tac output.tac roi4
tacmatch -abs=0.1 -rel=0.1 temp1.tac temp2.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

taccalc temp1.tac x 0 temp1.tac
tacadd -ovr temp2.tac output.tac roi1
tacmatch -abs=0.1 -rel=0.1 temp1.tac temp2.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

tacadd -ovr temp2.tac output.tac roi2
tacmatch -abs=0.1 -rel=0.1 temp1.tac temp2.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

tacadd -ovr temp2.tac output.tac roi3
tacmatch -abs=0.1 -rel=0.1 temp1.tac temp2.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n 2.0.1 \n"
printf " Test case 1: Integral start time given. \n"
printf " Test case 2: Threshold limits 90 and 100. \n"
printf " Expected result: Thresholded image is correct. \n\n"

rm -f output.v
$PROGRAM -start=3 dynamic.v 90 100 output.v 
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

img2dft output.v dynamic.roi output.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

tacadd -ovr temp1.tac test_dyn.tac roi2
tacadd -ovr temp2.tac output.tac roi2
tacmatch -abs=0.1 -rel=0.1 temp1.tac temp2.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

taccalc temp1.tac x 0 temp1.tac
tacadd -ovr temp2.tac output.tac roi1
tacmatch -abs=0.1 -rel=0.1 temp1.tac temp2.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

tacadd -ovr temp2.tac output.tac roi3
tacmatch -abs=0.1 -rel=0.1 temp1.tac temp2.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

tacadd -ovr temp2.tac output.tac roi4
tacmatch -abs=0.1 -rel=0.1 temp1.tac temp2.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n 2.1.0 \n"
printf " Test case: Threshold limits given in files. \n"
printf " Expected result: Thresholded image is correct. \n\n"

rm -f output.v
$PROGRAM -start=3 dynamic.v lower.lim upper.lim output.v 
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

img2dft output.v dynamic.roi output.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

tacadd -ovr temp1.tac test_dyn.tac roi2
tacadd -ovr temp2.tac output.tac roi2
tacmatch -abs=0.1 -rel=0.1 temp1.tac temp2.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

taccalc temp1.tac x 0 temp1.tac
tacadd -ovr temp2.tac output.tac roi1
tacmatch -abs=0.1 -rel=0.1 temp1.tac temp2.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

tacadd -ovr temp2.tac output.tac roi3
tacmatch -abs=0.1 -rel=0.1 temp1.tac temp2.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

tacadd -ovr temp2.tac output.tac roi4
tacmatch -abs=0.1 -rel=0.1 temp1.tac temp2.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


rm -f output.v
$PROGRAM -start=3 dynamic.v limits.ift limits.ift output.v 
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

img2dft output.v dynamic.roi output.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

tacadd -ovr temp1.tac test_dyn.tac roi2
tacadd -ovr temp2.tac output.tac roi2
tacmatch -abs=0.1 -rel=0.1 temp1.tac temp2.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

taccalc temp1.tac x 0 temp1.tac
tacadd -ovr temp2.tac output.tac roi1
tacmatch -abs=0.1 -rel=0.1 temp1.tac temp2.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

tacadd -ovr temp2.tac output.tac roi3
tacmatch -abs=0.1 -rel=0.1 temp1.tac temp2.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

tacadd -ovr temp2.tac output.tac roi4
tacmatch -abs=0.1 -rel=0.1 temp1.tac temp2.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"




printf "\n===================================================================\n"


printf "\n 3.0.0 \n"
printf " Test case: Option -abs with dynamic image. \n"
printf " Expected result: Error. \n\n"

$PROGRAM -abs dynamic.v 0 1000 output.v 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error "do not use absolute thresholds for dynamic image."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 3.0.1 \n"
printf " Test case 1: Option -abs with static image. \n"
printf " Test case 2: Threshold limits so wide that nothing is removed. \n"
printf " Expected result: Error. \n\n"

rm -f output.v
$PROGRAM -abs static.v -10000 10000000 output.v 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error "no pixels thresholded."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 3.1.0 \n"
printf " Test case 1: Option -abs with static image. \n"
printf " Test case 2: Threshold limits 0 and 81. \n"
printf " Expected result: Thresholded image is correct. \n\n"

rm -f output.v
$PROGRAM -mask=testmask.v static.v 0 90 testoutput.v 
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

$PROGRAM -abs -mask=outmask.v static.v 0 81 output.v 
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
imgmatch testoutput.v output.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
imgmatch testmask.v outmask.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 3.2.0 \n"
printf " Test case 1: Option -abs with mask image. \n"
printf " Test case 2: Threshold limits 4.5 and 5.5 \n"
printf " Expected result: Thresholded image is correct. \n\n"

rm -f testmask.v output.v
$PROGRAM -abs -mask=testmask.v mask.v 4.5 5.5 testoutput.v 
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
img2dft testoutput.v testmask.v output.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
taclist -ift output.tac > output.ift
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval -abs=0.001 output.ift value[1][1] 5
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n 3.2.1 \n"
printf " Test case 1: Option -abs with mask image. \n"
printf " Test case 2: Threshold limits 5 and 5 \n"
printf " Expected result: Thresholded image is correct. \n\n"

rm -f testmask.v output.v
$PROGRAM -abs -mask=testmask.v mask.v 5 5 testoutput.v 
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
img2dft testoutput.v testmask.v output.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
taclist -ift output.tac > output.ift
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval -abs=0.001 output.ift value[1][1] 5
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n testing with scaled image 1 \n\n"
imgcalc mask.v x 1000 temp.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
$PROGRAM -abs --debug=3 -mask=testmask.v temp.v 5000 5000 testoutput.v 
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
img2dft testoutput.v testmask.v output.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
taclist -ift output.tac > output.ift
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval -abs=0.001 output.ift value[1][1] 5000
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n testing with scaled image 2 \n\n"
imgcalc mask.v + 1000 temp.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
$PROGRAM -abs -mask=testmask.v temp.v 1005 1005 testoutput.v 
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
img2dft testoutput.v testmask.v output.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
taclist -ift output.tac > output.ift
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval -abs=0.001 output.ift value[1][1] 1005
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"





printf "\n===================================================================\n"




printf "\n 4.0.0 \n"
printf " Test case: Invalid option. \n"
printf " Expected result: Error. \n\n"

$PROGRAM -stupidoption dynamic.v 0 90 output.v 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error "invalid option '-stupidoption'."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n 4.0.1 \n"
printf " Test case: Extra argument. \n"
printf " Expected result: Error. \n\n"

$PROGRAM dynamic.v 0 90 output.v extra 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error "too many arguments."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n 4.0.2 \n"
printf " Test case: Missing argument. \n"
printf " Expected result: Error. \n\n"

$PROGRAM   0 90 output.v 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Error"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

$PROGRAM dynamic.v 0  1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Error"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

$PROGRAM dynamic.v 90 output.v 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Error"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 4.1.0 \n"
printf " Test case: Input file does not exist. \n"
printf " Expected result: Error. \n\n"

$PROGRAM nonexistingfile.v 0 90 output.v 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Error"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 4.2.0 \n"
printf " Test case: Invalid limits. \n"
printf " Expected result: Error. \n\n"

$PROGRAM dynamic.v 90 60 output.v 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Error"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n===================================================================\n"
printf "   All passed! \n"
printf "===================================================================\n"
exit 0


