/** @file imgaafind.c
 *  @brief Find abdominal aorta in PET image.
 *  @details Trial program, not for production use. 
 *  @copyright (c) Turku PET Centre
 *  @author Vesa Oikonen
 */
/// @cond
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <string.h>
#include <unistd.h>
#include <time.h>
/*****************************************************************************/
#include "libtpcmisc.h"
#include "libtpcimgio.h"
#include "libtpcimgp.h"
/*****************************************************************************/

/*****************************************************************************/
static char *info[] = {
  "Trial program to find the location of abdominal aorta in dynamic PET image.",
  " ",
  "Usage: @P [Options] imgfile maskfile",
  " ",
  "Options:",
  " -peak=<filename>",
  "     Image containing possible peak pixels.",
  " -stdoptions", // List standard options like --help, -v, etc
  " ",
  "See also: eabaort, imgprofi, imgthrs, imgfsegm, imgzavg, imgmax, img2tif",
  " ",
  "Keywords: image, input, blood, aorta, mask",
  0};
/*****************************************************************************/

/*****************************************************************************/
/* Turn on the globbing of the command line, since it is disabled by default in
   mingw-w64 (_dowildcard=0); in MinGW32 define _CRT_glob instead, if necessary;
   In Unix&Linux wildcard command line processing is enabled by default. */
/*
#undef _CRT_glob
#define _CRT_glob -1
*/
int _dowildcard = -1;
/*****************************************************************************/

/*****************************************************************************/
/**
 *  Main
 */
int main(int argc, char **argv)
{
  int        ai, help=0, version=0, verbose=1;
  int        ret;
  char      *cptr, imgfile[FILENAME_MAX], maskfile[FILENAME_MAX];
  char       peakfile[FILENAME_MAX];


  /*
   *  Get arguments
   */
  if(argc==1) {tpcPrintUsage(argv[0], info, stderr); return(1);}
  imgfile[0]=maskfile[0]=(char)0;
  peakfile[0]=(char)0;
  /* Options */
  for(ai=1; ai<argc; ai++) if(*argv[ai]=='-') { /* options */
    cptr=argv[ai]+1; if(*cptr=='-') cptr++; if(cptr==NULL) continue;
    if(tpcProcessStdOptions(argv[ai], &help, &version, &verbose)==0) continue;
    if(strncasecmp(cptr, "PEAK=", 5)==0) {
      strlcpy(peakfile, cptr+5, FILENAME_MAX); continue;
    }
    fprintf(stderr, "Error: invalid option '%s'\n", argv[ai]); 
    return(1);
  } else break;
  
  /* Print help or version? */
  if(help==2) {tpcHtmlUsage(argv[0], info, ""); return(0);}
  if(help) {tpcPrintUsage(argv[0], info, stdout); return(0);}
  if(version) {tpcPrintBuild(argv[0], stdout); return(0);}
  
  /* Process other arguments, starting from the first non-option */
  if(ai<argc) {strlcpy(imgfile, argv[ai], FILENAME_MAX); ai++;}
  if(ai<argc) {strlcpy(maskfile, argv[ai], FILENAME_MAX); ai++;}
  if(ai<argc) {fprintf(stderr, "Error: too many arguments.\n"); return(1);}

  /* Is something missing or wrong? */
  if(!maskfile[0]) {
    fprintf(stderr, "Error: missing command-line argument; use option --help\n");
    return(1);
  }

  /* In verbose mode print arguments and options */
  if(verbose>1) {
    printf("imgfile := %s\n", imgfile);
    printf("maskfile := %s\n", maskfile);
    if(peakfile[0]) printf("peakfile := %s\n", peakfile);
  }


  /*
   *  Read dynamic image
   */
  if(verbose>0) fprintf(stdout, "reading dynamic image %s\n", imgfile);
  IMG img; imgInit(&img);
  ret=imgRead(imgfile, &img);
  if(ret) {
    fprintf(stderr, "Error: %s\n", img.statmsg);
    if(verbose>1) printf("ret := %d\n", ret);
    return(2);
  }
  if(imgNaNs(&img, 1)>0)
    if(verbose>0) fprintf(stderr, "Warning: missing pixel values.\n");
  /* Check if PET data is raw or image */
  if(img.type!=IMG_TYPE_IMAGE) {
    fprintf(stderr, "Error: %s is not an image.\n", imgfile);
    imgEmpty(&img); return(2);
  }
  int dimt, dimz, dimy, dimx;
  dimt=img.dimt; dimz=img.dimz; dimy=img.dimy; dimx=img.dimx;
  if(verbose>0) fprintf(stdout, "  dim[x,y,z,t] := %d, %d, %d, %d\n", dimx, dimy, dimz, dimt);


  /*
   *  Filter in x,y dimension and in z dimension
   */
  IMG xyfilt; imgInit(&xyfilt);
  IMG zfilt; imgInit(&zfilt);
  ret=imgDup(&img, &xyfilt);
  if(ret==0) ret=imgDup(&img, &zfilt);
  if(ret) {
    fprintf(stderr, "Error: cannot setup IMG data.\n");
    if(verbose>1) fprintf(stderr, "ret := %d\n", ret);
    imgEmpty(&img); imgEmpty(&xyfilt); imgEmpty(&zfilt); return(3);
  }
  ret=imgMeanFilter(&xyfilt, 5, 5, 0, 0, verbose-4);
  if(ret==0) ret=imgMeanFilter(&zfilt, 0, 0, 5, 0, verbose-4);
  if(ret!=0) {
    fprintf(stderr, "Error: cannot filter the image.\n");
    if(verbose>1) fprintf(stderr, "ret := %d\n", ret);
    imgEmpty(&img); imgEmpty(&xyfilt); imgEmpty(&zfilt); return(4);
  }


  /*
   *  Subtract filtered images at peak time
   */
  IMG peak; imgInit(&peak);
  ret=imgAllocateWithHeader(&peak, dimz, dimy, dimx, 1, &img);
  if(ret) {
    fprintf(stderr, "Error: out of memory.\n");
    imgEmpty(&img); imgEmpty(&xyfilt); imgEmpty(&zfilt); imgEmpty(&peak);
    return(5);
  }
  for(int z=0; z<dimz; z++)
    for(int y=0; y<dimy; y++)
      for(int x=0; x<dimx; x++) {
        /* Find highest pixel value in t dimension */
        float maxv=0.0;
        int maxt=0;
        for(int t=0; t<dimt; t++) {
          if(xyfilt.m[z][y][x][t]>maxv) {maxt=t; maxv=xyfilt.m[z][y][x][t];}
        }
        peak.m[z][y][x][0]=zfilt.m[z][y][x][maxt]-xyfilt.m[z][y][x][maxt];
        if(peak.m[z][y][x][0]<0.0) peak.m[z][y][x][0]=0.0;
        /* divide by peak 'time' */
        if(maxt>0) peak.m[z][y][x][0]/=(float)maxt;
      }

  /* Divide by distance from the image centre */
  for(int y=0; y<dimy; y++) {
    for(int x=0; x<dimx; x++) {
      float d=1.0+hypotf((float)x-dimx/2, (float)y-dimy/2); // always >= 1
      for(int z=0; z<dimz; z++)
        peak.m[z][y][x][0]/=d;
    }
  }

  /* Write peak file */
  if(peakfile[0]) {
    if(verbose>1) printf("writing peak image...\n");
    ret=imgWrite(peakfile, &peak);
    if(ret) {
      fprintf(stderr, "Error: %s\n", peak.statmsg);
      imgEmpty(&img); imgEmpty(&xyfilt); imgEmpty(&zfilt); imgEmpty(&peak);
      return(11);
    }
    if(verbose>0) fprintf(stdout, "Peak image %s saved.\n", peakfile);
  }

  /* Free filter images */
  imgEmpty(&xyfilt); imgEmpty(&zfilt);


  /*
   *  Make mask
   */
  IMG mask; imgInit(&mask);
  ret=imgAllocateWithHeader(&mask, dimz, dimy, dimx, 1, &img);
  if(ret) {
    fprintf(stderr, "Error: out of memory.\n");
    imgEmpty(&img); imgEmpty(&peak);
    return(6);
  }

  {
    /* Mean over image planes */
    IMG zimg; imgInit(&zimg);
    if(imgMeanZ(&peak, &zimg)) {
      fprintf(stderr, "Error: cannot calculate mean over z dimension.\n");
      imgEmpty(&img); imgEmpty(&peak); imgEmpty(&mask); imgEmpty(&zimg);
      return(6);
    }

    /* Get peak position from that (omitting image border) */
    int px=1, py=1;
    float pmax=zimg.m[0][1][1][0];
    for(int y=1; y<dimy-1; y++) for(int x=1; x<dimx-1; x++) if(zimg.m[0][y][x][0]>pmax) {
      pmax=zimg.m[0][y][x][0]; px=x; py=y;
    }
    if(verbose>1) printf("estimated x,y position is %d,%d\n", 1+px, 1+py);

    /* Grow it */
    IMG zmask; imgInit(&zmask);
    imgAllocateWithHeader(&zmask, 1, dimy, dimx, 1, &zimg);
    imgRegionGrowingByThreshold(&zimg, 0, py, px, 0.5*pmax, 10.*pmax, &zmask, verbose-3);

    /* Inside that mask find max pixel in each image plane */
    float zmax[dimz];
    int xpos[dimz], ypos[dimz];
    for(int z=0; z<dimz; z++) {
      zmax[z]=0.0;
      int py=0, px=0;
      for(int y=0; y<dimy; y++) for(int x=0; x<dimx; x++)
        if(zmask.m[0][y][x][0]>0.5 && peak.m[z][y][x][0]>zmax[z]) {
          py=y; px=x; zmax[z]=peak.m[z][y][x][0];
        }
      //mask.m[z][py][px][0]=1.0;
      xpos[z]=px; ypos[z]=py;
    }

    /* Median of plane max values */
    float m=fmedian(zmax, dimz);
    /* Grow regions starting from the max of each plane */
    if(verbose>1) printf("growing from max of each plane\n");
    for(int z=0; z<dimz; z++) {
      imgRegionGrowingByThreshold(&peak, z, ypos[z], xpos[z], 0.99*m, 100.*m, &mask, verbose-3);
    }

    imgEmpty(&zimg); imgEmpty(&zmask);
  }

  /* Write file */
  if(maskfile[0]) {
    if(verbose>1) printf("writing mask image...\n");
    ret=imgWrite(maskfile, &mask);
    if(ret) {
      fprintf(stderr, "Error: %s\n", mask.statmsg);
      imgEmpty(&img); imgEmpty(&peak); imgEmpty(&mask);
      return(12);
    }
    if(verbose>0) fprintf(stdout, "Mask image %s saved.\n", maskfile);
  }

  imgEmpty(&img); imgEmpty(&peak); imgEmpty(&mask);
  if(verbose>0) printf("done.\n");
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/// @endcond
