#!/bin/bash
#: Title      : test_eabaort
#: Date       : 2023-06-30
#: Author     : "Vesa Oikonen" <vesa.oikonen@utu.fi>
#: Options    : None

# Set the name of executable to test
if [ -n "${OS+1}" ] && [ "$OS" = "Windows_NT" ]; then
EXT=.exe;
else
EXT=;
fi
PROGRAM=../../eabaort$EXT;

printf "=====================================================================\n"
printf "creating test data for %s\n" $PROGRAM
printf "=====================================================================\n"


printf "\n verifying that required data exists \n"

if [ ! -f $PROGRAM ]; then
  printf "Failed: executable does not exist.\n"
  exit 1
fi

if [ ! -f radiowater.fit ] || [ ! -f frames.dat ] ; then
  printf "Failed: required data does not exist.\n"
  exit 1
fi
printf "ok\n"


if [ ! -f simple_blood.tac ] || [ ! -f simple_bkg.tac ] ; then
  printf "start[seconds]	end[kBq/cc]	Blood\n" > simple_blood.tac
  printf "0.00000	600.00000	1.000000e+00\n" >> simple_blood.tac
  printf "start[seconds]	end[kBq/cc]	Backgr\n" > simple_bkg.tac
  printf "0.00000	600.00000	0.000000e+00\n" >> simple_bkg.tac
fi


# One plane one frame vessel with RC 1 following Germano model
# zero background peak 1 no noise
if [ ! -f simple.v ] ; then
  simiart -bar -dim=80 -pxlsize=1.0 -diameter=20 -fwhm=6 simple_blood.tac simple_bkg.tac simple.v
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  img2tif simple.v
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi

# One plane one frame vessel with RC 0.499 following Germano model
# zero background peak 1 no noise
if [ ! -f germano.v ] ; then
  simiart -bar -dim=80 -pxlsize=1.0 -diameter=10 -fwhm=10 simple_blood.tac simple_bkg.tac germano.v
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  img2tif germano.v
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi

# One plane one frame vessel with RC 0.499 following Gaussian model
# zero background peak 1 no noise
if [ ! -f gaussian.v ] ; then
  simiart -gaussian -dim=80 -pxlsize=1.0 -diameter=10 -fwhm=10 simple_blood.tac simple_bkg.tac gaussian.v
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  img2tif gaussian.v
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi


# Make BTAC in seconds
if [ ! -f blood.tac ] || [ ! -f radiowater.dat ] ; then
  rm -f dynamic20.v dynamic10.v
  fit2dat -c=0,600,1 radiowater.fit radiowater.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tactime -nogap radiowater.dat 32.5
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacstudy -force radiowater.dat sim1
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  simframe -sec radiowater.dat frames.dat blood.tac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi

# Simulate tissue TAC in seconds
if [ ! -f tissue.tac ] || [ ! -f tissue.dat ] ; then
  rm -f dynamic20.v dynamic10.v
  p2t_3c -nosub radiowater.dat 0.15 0.17 0 0 0 0 tissue.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  simframe -sec tissue.dat frames.dat tissue.tac 
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tac2svg simulated_tacs.svg -l radiowater.dat -s tissue.tac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi

# Realistic vessel size and FWHM leads to RC of 0.987
if [ ! -f dynamic20.v ] ; then
  rm -f dynamic20n.v
  simiart -gaussian -dim=128 -pxlsize=1.0 -diameter=20 -fwhm=8 blood.tac tissue.tac dynamic20.v
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  img2tif dynamic20.v
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi

# Smaller vessel size and realistic FWHM leads to lowered RC depending on the difference
# between background and blood activities but for the difference the RC would be 0.660.
if [ ! -f dynamic10.v ] ; then
  rm -f dynamic10n.v
  simiart -gaussian -dim=64 -pxlsize=1.0 -diameter=10 -fwhm=8 blood.tac tissue.tac dynamic10.v
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  img2tif dynamic10.v
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi

# Simulating noise and vessel changing position in additional planes
if [ ! -f dynamic20n.v ] ; then
  rm -f temp.v
  FNAME="dynamic20"
  imgmove -fill $FNAME.v -1 1 1-20 moved01.v
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  imgmove -fill $FNAME.v -2 2 1-20 moved02.v
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  imgmove -fill $FNAME.v -3 3 1-20 moved03.v
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  imgmove -fill $FNAME.v -4 4 1-20 moved04.v
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  imgmove -fill $FNAME.v -5 5 1-20 moved05.v
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  imgmove -fill $FNAME.v -6 6 1-20 moved06.v
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  imgmove -fill $FNAME.v -7 7 1-20 moved07.v
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  imgmove -fill $FNAME.v -8 8 1-20 moved08.v
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  imgmove -fill $FNAME.v -9 9 1-20 moved09.v
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  imgadd temp.v $FNAME.v moved??.v
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  fvar4img -i=O-15 -minsd=0.05 temp.v 0.005 $FNAME"n.v"
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  img2tif $FNAME"n.v"
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi

# Simulating noise and vessel changing position in additional planes
if [ ! -f dynamic10n.v ] ; then
  rm -f temp.v
  FNAME="dynamic10"
  imgmove -fill $FNAME.v -1 1 1-20 moved01.v
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  imgmove -fill $FNAME.v -2 2 1-20 moved02.v
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  imgmove -fill $FNAME.v -3 3 1-20 moved03.v
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  imgmove -fill $FNAME.v -4 4 1-20 moved04.v
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  imgmove -fill $FNAME.v -5 5 1-20 moved05.v
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  imgmove -fill $FNAME.v -6 6 1-20 moved06.v
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  imgmove -fill $FNAME.v -7 7 1-20 moved07.v
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  imgmove -fill $FNAME.v -8 8 1-20 moved08.v
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  imgmove -fill $FNAME.v -9 9 1-20 moved09.v
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  imgadd temp.v $FNAME.v moved??.v
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  fvar4img -i=O-15 -minsd=0.05 temp.v 0.005 $FNAME"n.v"
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  img2tif $FNAME"n.v"
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi


# Lowered RC and different vessel size on different planes for testing
# that the best plane can be found.
# RC is 0.660 on the best plane which is number 5.
if [ ! -f dynamicvar.v ] || [ ! -f dynamicvarn.v ] ; then
  FNAME="dynamicvar"
  simiart -gaussian -dim=128 -pxlsize=1.0 -diameter=5 -fwhm=8 blood.tac tissue.tac plane01v.v
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  simiart -gaussian -dim=128 -pxlsize=1.0 -diameter=6 -fwhm=8 blood.tac tissue.tac plane02v.v
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  simiart -gaussian -dim=128 -pxlsize=1.0 -diameter=7 -fwhm=8 blood.tac tissue.tac plane03v.v
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  simiart -gaussian -dim=128 -pxlsize=1.0 -diameter=8 -fwhm=8 blood.tac tissue.tac plane04v.v
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  simiart -gaussian -dim=128 -pxlsize=1.0 -diameter=10 -fwhm=8 blood.tac tissue.tac plane05v.v
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  simiart -gaussian -dim=128 -pxlsize=1.0 -diameter=8 -fwhm=8 blood.tac tissue.tac plane06v.v
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  simiart -gaussian -dim=128 -pxlsize=1.0 -diameter=7 -fwhm=8 blood.tac tissue.tac plane07v.v
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  simiart -gaussian -dim=128 -pxlsize=1.0 -diameter=4 -fwhm=8 blood.tac tissue.tac plane08v.v
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  imgadd $FNAME.v plane??v.v
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  img2tif $FNAME.v
  rm -f plane??v.v
  fvar4img -i=O-15 -minsd=0.04 $FNAME.v 0.004 $FNAME"n.v"
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  img2tif $FNAME"n.v"
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi



printf "\n\n"
printf "=====================================================================\n"
printf "testing %s\n" $PROGRAM
printf "=====================================================================\n"

printf "\n Basic tests with one small noiseless image matrix\n"

printf "\n 1.0.0 \n"
printf " Test case: Image simulated with the Germano model with RC=1. Correct FWHM given. \n"
printf " Expected result 1: Correct diameter and Rc estimate. \n"
printf " Expected result 2: Estimated TACs will be correct. \n\n"

rm -f output100.tac bkg100.tac

$PROGRAM -model=germano -fwhm=6 -bkg=bkg100.tac simple.v output100.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

iftisval -abs=0.01 output100.tac RC 1.0
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval -abs=0.2 output100.tac vessel_diameter 20.0
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

tacmatch -abs=0.01 simple_bkg.tac bkg100.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.01 simple_blood.tac output100.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 1.0.1 \n"
printf " Test case: Image simulated with the Germano model with RC=0.499. Correct FWHM given. \n"
printf " Expected result 1: Correct diameter and Rc estimate. \n"
printf " Expected result 2: Estimated BTAC will not be correct. \n\n"

rm -f output101.tac bkg101.tac

$PROGRAM --debug=5 -model=germano -fwhm=10 -bkg=bkg101.tac germano.v output101.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

iftisval -abs=0.005 output101.tac RC 0.499
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval -abs=0.1 output101.tac vessel_diameter 10.0
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

tacmatch -abs=0.01 simple_bkg.tac bkg101.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n 1.1.0 \n"
printf " Test case 1: Image simulated with the Gaussian model with RC=0.499. \n"
printf " Test case 2: Correct FWHM given. \n"
printf " Expected result 1: Correct diameter and Rc estimate. \n"
printf " Expected result 2: Estimated BTAC will be close to correct. \n\n"

rm -f output110.tac bkg110.tac

$PROGRAM -model=gaussian -fwhm=10 -bkg=bkg110.tac gaussian.v output110.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

iftisval -abs=0.05 output110.tac RC 0.499
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval -abs=1.0 output110.tac vessel_diameter 10.0
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

tacmatch -abs=0.01 simple_bkg.tac bkg110.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.10 simple_blood.tac output110.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 1.1.1 \n"
printf " Test case 1: Image simulated with the Gaussian model with RC=0.499. \n"
printf " Test case 2: Correct diameter given. \n"
printf " Expected result 1: Correct diameter and Rc estimate. \n"
printf " Expected result 2: Estimated BTAC will be close to correct. \n\n"

rm -f output111.tac bkg111.tac

$PROGRAM -model=gaussian -fwhm=fit -diameter=10 -bkg=bkg111.tac gaussian.v output111.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

iftisval -abs=0.05 output111.tac RC 0.499
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval -abs=0.5 output111.tac vessel_diameter 10.0
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

tacmatch -abs=0.01 simple_bkg.tac bkg111.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.05 simple_blood.tac output111.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "=====================================================================\n"
printf "\n Testing for general functionality with noisy dynamic images\n"


printf "\n 2.0.0 \n"
printf " Test case 1: Noisy image fitted with Gaussian model. \n"
printf " Test case 2: Image contains several planes with drifting vessel position. \n"
printf " Test case 3: FWHM given. \n"
printf " Expected result 1: Correct diameter and Rc estimate. \n"
printf " Expected result 2: Estimated BTAC will be close to correct. \n\n"

rm -f output200.tac bkg200.tac

$PROGRAM -model=gaussian -fwhm=8 -bkg=bkg200.tac dynamic20n.v output200.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

iftisval -abs=0.05 output200.tac RC 0.987
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval -abs=0.5 output200.tac vessel_diameter 20.0
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

tac2svg -y1=0 output200.svg -s blood.tac tissue.tac -l output200.tac bkg200.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi

tacmatch -abs=1 tissue.tac bkg200.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=2 blood.tac output200.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

rm -f output201.tac bkg201.tac

$PROGRAM -model=gaussian -fwhm=8 -bkg=bkg201.tac dynamic10n.v output201.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

iftisval -abs=0.05 output201.tac RC 0.660
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval -abs=0.5 output201.tac vessel_diameter 10.0
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

tac2svg -y1=0 output201.svg -s blood.tac tissue.tac -l output201.tac bkg201.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi

tacmatch -abs=1 tissue.tac bkg201.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=3 -rel=3 blood.tac output201.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 2.1.0 \n"
printf " Test case 1: Noisy image fitted with Gaussian model. \n"
printf " Test case 2: Image contains several planes with drifting vessel position. \n"
printf " Test case 3: Diameter given. \n"
printf " Expected result 1: Correct diameter and Rc estimate. \n"
printf " Expected result 2: Estimated BTAC will be close to correct. \n\n"

rm -f output210.tac bkg210.tac

$PROGRAM -model=gaussian -fwhm=fit -diameter=10 -bkg=bkg210.tac dynamic10n.v output210.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

iftisval -abs=0.05 output210.tac RC 0.660
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval -abs=0.5 output210.tac vessel_diameter 10.0
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

tacmatch -abs=1 -rel=1 tissue.tac bkg210.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=2 -rel=2 blood.tac output210.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"





printf "\n 2.2.0 \n"
printf " Test case 1: Noisy image fitted with Gaussian model. \n"
printf " Test case 2: Image contains several planes with variable vessel diameter. \n"
printf " Test case 3: FWHM given. \n"
printf " Expected result 1: Best plane found. \n"
printf " Expected result 2: Correct diameter and Rc estimate for the best plane. \n"
printf " Expected result 3: Estimated BTAC will be close to correct. \n\n"

rm -f stdout.txt output220.tac bkg220.tac

$PROGRAM --debug=2 -model=gaussian -fwhm=8 -plane=best dynamicvarn.v output220.tac > stdout.txt
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

iftisval stdout.txt best_plane 5
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

iftisval -abs=1.0 output220.tac vessel_diameter 10.0
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval -abs=0.06 output220.tac RC 0.660
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

tacmatch -abs=10 blood.tac output220.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 2.2.1 \n"
printf " Test case 1: Noisy image fitted with Gaussian model. \n"
printf " Test case 2: Image contains several planes with variable vessel diameter. \n"
printf " Test case 3: Diameter given. \n"
printf " Expected result 1: Best plane found. \n"
printf " Expected result 2: Correct diameter and Rc estimate for the best plane. \n"
printf " Expected result 3: Estimated BTAC will be close to correct. \n\n"

rm -f stdout.txt output221.tac bkg221.tac

$PROGRAM --debug=2 -model=gaussian -fwhm=fit -diameter=10 -plane=best dynamicvarn.v output221.tac > stdout.txt
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

iftisval stdout.txt best_plane 5
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

iftisval -abs=1.0 output221.tac FWHM 8.0
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval -abs=0.06 output221.tac RC 0.660
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

tacmatch -abs=10 blood.tac output221.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 2.2.2 \n"
printf " Test case 1: Noisy image fitted with Gaussian model. \n"
printf " Test case 2: Image contains several planes with variable vessel diameter. \n"
printf " Test case 3: Neither diameter or FWHM given. \n"
printf " Expected result 1: Best plane found. \n"
printf " Expected result 2: Correct diameter and Rc estimate for the best plane. \n"
printf " Expected result 3: Estimated BTAC will be close to correct. \n\n"

rm -f stdout.txt output222.tac bkg222.tac

$PROGRAM --debug=2 -model=gaussian -fwhm=fit -plane=best dynamicvarn.v output222.tac > stdout.txt
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

iftisval stdout.txt best_plane 5
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

iftisval -abs=2.0 output221.tac FWHM 8.0
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval -abs=0.20 output221.tac RC 0.660
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

tacmatch -abs=20 blood.tac output221.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"




printf "\n===================================================================\n"
printf "   All passed! \n"
printf "===================================================================\n"
exit 0

