#!/bin/bash
#: Title      : test_fit_bpr
#: Date       : 2023-06-29
#: Author     : "Vesa Oikonen" <vesa.oikonen@utu.fi>
#: Options    : None

# Set the name of executable to test
if [ -n "${OS+1}" ] && [ "$OS" = "Windows_NT" ]; then
EXT=.exe;
else
EXT=;
fi
PROGRAM=../../fit_bpr$EXT;
if [ ! -f $PROGRAM ]; then
  printf "Failed: executable does not exist.\n"
  exit 1
fi


printf "=====================================================================\n"
printf "creating test data for %s\n" $PROGRAM
printf "=====================================================================\n"


printf "\n verifying that required data exists \n"

if [ ! -f PE2I_bprat.fit ] || [ ! -f DPA714_bprat.fit ]; then
  printf "Failed: required data does not exist.\n"
  exit 1
fi
if [ ! -f FEMPA_bprat.fit ] || [ ! -f ORM_bprat.fit ]; then
  printf "Failed: required data does not exist.\n"
  exit 1
fi
if [ ! -f PBR28HAB_bprat.fit ] || [ ! -f PBR28MAB_bprat.fit ] || [ ! -f PBR28LAB_bprat.fit ]; then
  printf "Failed: required data does not exist.\n"
  exit 1
fi


printf "\n making test data \n"

if [ ! -f PE2I_bprat.dat ]; then
  fit2dat -a=90 PE2I_bprat.fit PE2I_bprat.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  fit2dat -x=1,5,10,20,40,80 PE2I_bprat.fit PE2I.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tac2svg PE2I_bprat.svg -s PE2I.dat -l PE2I_bprat.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi

if [ ! -f DPA714_bprat.dat ]; then
  fit2dat -a=150 DPA714_bprat.fit DPA714_bprat.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  fit2dat -x=1,3,6,10,30,60,90,120 DPA714_bprat.fit DPA714.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tac2svg DPA714_bprat.svg -s DPA714.dat -l DPA714_bprat.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi

if [ ! -f FEMPA_bprat.dat ]; then
  fit2dat -a=150 FEMPA_bprat.fit FEMPA_bprat.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  fit2dat -x=1,5,10,30,60,90,130 FEMPA_bprat.fit FEMPA.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tac2svg FEMPA_bprat.svg -s FEMPA.dat -l FEMPA_bprat.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi

if [ ! -f ORM_bprat.dat ]; then
  fit2dat -a=90 ORM_bprat.fit ORM_bprat.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  fit2dat -x=1,3,5,10,30,60,90 ORM_bprat.fit ORM.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tac2svg ORM_bprat.svg -s ORM.dat -l ORM_bprat.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi

if [ ! -f PBR28HAB_bprat.dat ] || [ ! -f PBR28MAB_bprat.dat ] || [ ! -f PBR28LAB_bprat.dat ]; then
  fit2dat -a=90 PBR28HAB_bprat.fit PBR28HAB_bprat.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  fit2dat -a=90 PBR28MAB_bprat.fit PBR28MAB_bprat.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  fit2dat -a=90 PBR28LAB_bprat.fit PBR28LAB_bprat.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  fit2dat -x=1,3,5,10,20,40,60,90 PBR28HAB_bprat.fit PBR28HAB.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  fit2dat -x=1,3,5,10,20,40,60,90 PBR28MAB_bprat.fit PBR28MAB.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  fit2dat -x=1,3,5,10,20,40,60,90 PBR28LAB_bprat.fit PBR28LAB.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tac2svg PBR28_bprat.svg -s PBR28?AB.dat -l PBR28?AB_bprat.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi


printf "\n\n"
printf "=====================================================================\n"
printf "testing %s\n" $PROGRAM
printf "=====================================================================\n"



printf "\n 0.1.0 \n"
printf " Test case: Unknown option. \n"
printf " Expected result: Error. \n"

$PROGRAM -stupidoption FEMPA.dat output.fit 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error "invalid option '-stupidoption'."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 0.1.1 \n"
printf " Test case: Extra argument entered. \n"
printf " Expected result: Error. \n"

$PROGRAM -d2 FEMPA.dat output.fit extra 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error "too many arguments: 'extra'."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 0.2.0 \n"
printf " Test case: Defaults. \n"
printf " Expected result: Default values of variables are correct. \n"

$PROGRAM -d2 FEMPA.dat output.fit 1>stdout.txt 2>&1
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt dftfile "FEMPA.dat"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt fitfile "output.fit"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt svgfile 
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt model "1"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt min_meas "1"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt reliability_level "2"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt weights "0"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt fixedDelay "0"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n 0.2.1 \n"
printf " Test case: Options are correctly interpreted. \n"
printf " Expected result: Values of variables are correct. \n"

$PROGRAM -d2 -model=hillb -delay=1,1 -bl=0.3 -w1 -fast -min=abs -svg=output.svg FEMPA.dat output.fit 1>stdout.txt 2>&1
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt dftfile "FEMPA.dat"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt fitfile "output.fit"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt svgfile "output.svg"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt model "2"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt min_meas "2"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt reliability_level "1"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt weights "1"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt fixedDelay "1.1"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n 0.3.0 \n"
printf " Test case: Missing argument(s). \n"
printf " Expected result: Error. \n"

$PROGRAM  output.fit 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error "missing command-line argument; use option --help"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

$PROGRAM FEMPA.dat  1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error "missing command-line argument; use option --help"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

$PROGRAM -d2 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error "missing command-line argument; use option --help"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n\n---------------------------------------------------------------\n\n"

printf "\n 1.0.0 \n"
printf " Test case: Input file does not exist. \n"
printf " Expected result: Error. \n"

$PROGRAM -d2 nonexistingfile.dat output.fit 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Error in reading 'nonexistingfile.dat'" "cannot open file"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n 1.1.0 \n"
printf " Test case: Input file in wrong format. \n"
printf " Expected result: Error. \n"

echo meh ngh      >  bad.dat
echo zzzzhhhh bop >> bad.dat
$PROGRAM -d2 bad.dat output.fit 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Error in reading 'bad.dat'" "wrong format"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 1.5.0 \n"
printf " Test case 1: Fraction file is catenated from two studies. \n"
printf " Test case 2: Result file is written in correct format. \n"
printf " Expected result 1: Results are the same whether data is sorted or not. \n"
printf " Expected result 2: Result file can be processed by other program. \n"
printf " Expected result 3: Optional SVG plots are created. \n\n"

rm -f output1.fit output2.fit temp1.dat temp2.dat
tacformat -f=SIMPLE FEMPA.dat temp1.dat
cat temp1.dat temp1.dat >> temp2.dat

$PROGRAM -wf -svg=output1.svg temp1.dat output1.fit
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

$PROGRAM -wf -svg=output2.svg temp2.dat output2.fit
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

fit2res output1.fit output2.fit
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi

resmatch -abs=0.1 -around -res=1-6 output1.res output2.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

if [ ! -f output1.svg ] ; then printf "Failed!\n" ; exit 1 ; fi
if [ ! -f output2.svg ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n===================================================================\n"
printf "   Testing that noiseless blood/plasma ratio data for certain tracers are well fitted \n"
printf "===================================================================\n"

printf "\n 2.0.0 \n"
printf " Test case 1: DPA-714 \n"
printf " Test case 2: Function iHillb \n"
printf " Expected result: Fitted curve and parameters are close to correct. \n\n"

rm -f output.fit output.dat
$PROGRAM -wf -svg=test200.svg -model=ihillb DPA714.dat output.fit
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
fit2dat -f=DPA714.dat output.fit output.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.01 -rel=1 DPA714.dat output.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
fit2res DPA714_bprat.fit output.fit
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
resmatch -around -res=1-5 DPA714_bprat.res output.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 2.1.0 \n"
printf " Test case 1: FEMPA \n"
printf " Test case 2: Function iHillb \n"
printf " Expected result: Fitted curve and parameters are close to correct. \n\n"

rm -f output.fit output.dat
$PROGRAM -wf -svg=test210.svg -model=ihillb FEMPA.dat output.fit
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
fit2dat -f=FEMPA.dat output.fit output.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.01 -rel=1 FEMPA.dat output.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
fit2res FEMPA_bprat.fit output.fit
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
resmatch -around -res=1-5 FEMPA_bprat.res output.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 2.2.0 \n"
printf " Test case 1: ORM-13070 \n"
printf " Test case 2: Function Hillb \n"
printf " Test case 3: Baseline fixed with option -bl \n"
printf " Test case 4: Delay fitted with option -delay \n"
printf " Expected result: Fitted curve and parameters are close to correct. \n\n"

rm -f output.fit output.dat
$PROGRAM -wf -bl=0.62 -delay -svg=test220.svg -model=hillb ORM.dat output.fit
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
fit2dat -f=ORM.dat output.fit output.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.01 -rel=1 ORM.dat output.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
fit2res ORM_bprat.fit output.fit
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
resmatch -around -res=1-5 ORM_bprat.res output.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 2.3.0 \n"
printf " Test case 1: PE2I \n"
printf " Test case 2: Function igv \n"
printf " Test case 3: Delay fitted with option -delay \n"
printf " Expected result: Fitted curve and parameters are close to correct. \n\n"

rm -f output.fit output.dat
$PROGRAM -wf -delay -svg=test230.svg -model=igv PE2I.dat output.fit
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
fit2dat -f=PE2I.dat output.fit output.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.01 -rel=1 PE2I.dat output.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
fit2res PE2I_bprat.fit output.fit
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
resmatch -around -res=1-5 PE2I_bprat.res output.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"




printf "\n 2.4.0 \n"
printf " Test case 1: PBR28 HAB and MAB\n"
printf " Test case 2: Function ihillb \n"
printf " Test case 3: Input file contains more than one fraction curve \n"
printf " Expected result 1: All curves are fitted. \n"
printf " Expected result 2: Fitted curves are close to correct. \n\n"

rm -f output.fit output.dat
tacadd -ovr temp.dat PBR28HAB.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
tacadd      temp.dat PBR28MAB.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

$PROGRAM -wf -svg=test240.svg -model=ihillb temp.dat output.fit
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
fit2dat -f=temp.dat output.fit output.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.01 -rel=1 temp.dat output.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 2.4.1 \n"
printf " Test case 1: PBR28 LAB\n"
printf " Test case 2: Function hillb \n"
printf " Expected result: Fitted curves are close to correct. \n\n"

rm -f output.fit output.dat

$PROGRAM -wf -svg=test241.svg -model=hillb PBR28LAB.dat output.fit
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
fit2dat -f=PBR28LAB.dat output.fit output.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.01 -rel=1 PBR28LAB.dat output.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



rm -f *.bak
printf "\n===================================================================\n"
printf "   All passed! \n"
printf "===================================================================\n"
exit 0

