#!/bin/bash
#: Title      : test_fit_feng
#: Date       : 2023-06-29
#: Author     : "Vesa Oikonen" <vesa.oikonen@utu.fi>
#: Options    : None

# Set the exit status of a pipeline to the exit status of failed, not the last command in the pipeline
set -o pipefail

# Set the name of executable to test
if [ -n "${OS+1}" ] && [ "$OS" = "Windows_NT" ]; then
EXT=.exe;
else
EXT=;
fi

PROGRAM=../../fit_feng$EXT;
if [ ! -f $PROGRAM ]; then
  printf "Failed: executable does not exist.\n"
  exit 1
fi

printf "=====================================================================\n"
printf "creating test data for %s\n" $PROGRAM
printf "=====================================================================\n"


printf "\n verifying that required data exists \n"




if [ ! -f ../ptac/c11-carfentanil_metabcorr.kbq ]; then
  printf "Failed: required data does not exist.\n"
  exit 1
fi
if [ ! -f ../ptac/c11-flumazenil_metabcorr.kbq ]; then
  printf "Failed: required data does not exist.\n"
  exit 1
fi
if [ ! -f ../ptac/c11-mp4b_metabcorr.kbq ]; then
  printf "Failed: required data does not exist.\n"
  exit 1
fi
if [ ! -f ../ptac/c11-raclopride_metabcorr.kbq ]; then
  printf "Failed: required data does not exist.\n"
  exit 1
fi
if [ ! -f ../ptac/c11-way100635_metabcorr.kbq ]; then
  printf "Failed: required data does not exist.\n"
  exit 1
fi
if [ ! -f ../ptac/f18-choline_metabcorr.kbq ]; then
  printf "Failed: required data does not exist.\n"
  exit 1
fi
if [ ! -f ../ptac/f18-fbpa_totplasma.kbq ]; then
  printf "Failed: required data does not exist.\n"
  exit 1
fi
if [ ! -f ../ptac/f18-fdg_arteriaplasma.kbq ]; then
  printf "Failed: required data does not exist.\n"
  exit 1
fi
if [ ! -f ../ptac/f18-fdg_venaplasma.kbq ]; then
  printf "Failed: required data does not exist.\n"
  exit 1
fi
if [ ! -f ../ptac/f18-fdopa_metabcor.kbq ]; then
  printf "Failed: required data does not exist.\n"
  exit 1
fi
if [ ! -f ../ptac/o15-h2o_muscle.kbq ]; then
  printf "Failed: required data does not exist.\n"
  exit 1
fi
if [ ! -f ../ptac/o15-h2o_onkol.kbq ]; then
  printf "Failed: required data does not exist.\n"
  exit 1
fi
if [ ! -f ../ptac/o15-o2_metabcorr_delaycorr_muscle.kbq ]; then
  printf "Failed: required data does not exist.\n"
  exit 1
fi
if [ ! -f ../ptac/o15-o2_totblood_brain.kbq ]; then
  printf "Failed: required data does not exist.\n"
  exit 1
fi
if [ ! -f ../ptac/o15-o2_totblood_muscle.kbq ]; then
  printf "Failed: required data does not exist.\n"
  exit 1
fi
if [ ! -f ../ptac/o15-h2o_blo1.kbq ]; then
  printf "Failed: required data does not exist.\n"
  exit 1
fi
if [ ! -f ../ptac/o15-h2o_blo2.kbq ]; then
  printf "Failed: required data does not exist.\n"
  exit 1
fi

if [ ! -f ae392wholebladder.tac ]; then
  printf "Failed: required data does not exist.\n"
  exit 1
fi



if [ ! -f samples.bld ]; then
  printf "time[minutes]\tPTAC[kBq/cc]\n" > samples.bld
  printf "0.0\t0.0\n" >> samples.bld
  printf "1.0\t0.0\n" >> samples.bld
  printf "2.0\t0.0\n" >> samples.bld
  printf "3.0\t0.0\n" >> samples.bld
  printf "4.0\t0.0\n" >> samples.bld
  printf "5.0\t0.0\n" >> samples.bld
  printf "6.0\t0.0\n" >> samples.bld
  printf "7.0\t0.0\n" >> samples.bld
  printf "8.0\t0.0\n" >> samples.bld
  printf "9.0\t0.0\n" >> samples.bld
  printf "10.0\t0.0\n" >> samples.bld
  printf "12.0\t0.0\n" >> samples.bld
  printf "14.0\t0.0\n" >> samples.bld
  printf "16.0\t0.0\n" >> samples.bld
  printf "18.0\t0.0\n" >> samples.bld
  printf "20.0\t0.0\n" >> samples.bld
  printf "23.0\t0.0\n" >> samples.bld
  printf "26.0\t0.0\n" >> samples.bld
  printf "30.0\t0.0\n" >> samples.bld
  printf "35.0\t0.0\n" >> samples.bld
  printf "40.0\t0.0\n" >> samples.bld
  printf "45.0\t0.0\n" >> samples.bld
  printf "50.0\t0.0\n" >> samples.bld
  printf "55.0\t0.0\n" >> samples.bld
  printf "60.0\t0.0\n" >> samples.bld
fi


if [ ! -f s2exp.fit ] || [ ! -f s2exp.bld ]; then
  rm -f s2exp.bld
  printf "FIT1\tfit_feng 0.0.0  (c) 2001-2011 by Turku PET Centre\n" > s2exp.fit
  printf "Date:\t2018-04-20 21:33:21\n" >> s2exp.fit
  printf "Data file:\ts2exp.bld\n" >> s2exp.fit
  printf "Data unit:\tkBq/mL\n" >> s2exp.fit
  printf "Time unit:\tmin\n" >> s2exp.fit
  printf "Nr of VOIs:\t1\n" >> s2exp.fit
  printf "Region        Plane\tStart\tEnd\tdataNr\tWSS\tparNr\tType\tParameters\n" >> s2exp.fit
  printf "ptac . .\t0\t60\t50\t0.0\t5\t1312\t100\t-0.1\t10\t-0.005\t1.0\n" >> s2exp.fit
fi

if [ ! -f s2exp.bld ]; then
  fit2dat -f=samples.bld s2exp.fit s2exp.bld
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tac2svg -mt=s2exp -legend=n s2exp.svg -l s2exp.bld
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi

if [ ! -f s3exp.fit ] || [ ! -f s3exp.bld ]; then
  rm -f s3exp.bld
  printf "FIT1\tfit_feng 0.0.0  (c) 2001-2011 by Turku PET Centre\n" > s3exp.fit
  printf "Date:\t2018-04-20 21:33:21\n" >> s3exp.fit
  printf "Data file:\ts3exp.bld\n" >> s3exp.fit
  printf "Data unit:\tkBq/mL\n" >> s3exp.fit
  printf "Time unit:\tmin\n" >> s3exp.fit
  printf "Nr of VOIs:\t1\n" >> s3exp.fit
  printf "Region        Plane\tStart\tEnd\tdataNr\tWSS\tparNr\tType\tParameters\n" >> s3exp.fit
  printf "ptac . .\t0\t60\t50\t0.0\t7\t1313\t100\t-0.2\t80\t-0.5\t250\t-0.005\t1.0\n" >> s3exp.fit
fi

if [ ! -f s3exp.bld ]; then
  fit2dat -f=samples.bld s3exp.fit s3exp.bld
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tac2svg -mt=s3exp -legend=n s3exp.svg -l s3exp.bld
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi

if [ ! -f s3expi.tac ]; then
  fit2dat -i -f=samples.bld s3exp.fit s3expi.tac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tac2svg -mt="s3exp integral" -legend=n s3expi.svg -l s3expi.tac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi

if [ ! -f s4exp.fit ] || [ ! -f s4exp.bld ]; then
  rm -f s4exp.bld
  printf "FIT1\tfit_feng 0.0.0  (c) 2001-2011 by Turku PET Centre\n" > s4exp.fit
  printf "Date:\t2018-04-20 21:33:21\n" >> s4exp.fit
  printf "Data file:\ts4exp.bld\n" >> s4exp.fit
  printf "Data unit:\tkBq/mL\n" >> s4exp.fit
  printf "Time unit:\tmin\n" >> s4exp.fit
  printf "Nr of VOIs:\t1\n" >> s4exp.fit
  printf "Region        Plane\tStart\tEnd\tdataNr\tWSS\tparNr\tType\tParameters\n" >> s4exp.fit
  printf "ptac . .\t0\t60\t50\t0.0\t9\t1314\t300\t-4\t200\t-0.4\t100\t-0.04\t20\t0.0\t1.0\n" >> s4exp.fit
fi

if [ ! -f s4exp.bld ]; then
  fit2dat -f=samples.bld s4exp.fit s4exp.bld
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tac2svg -mt=s4exp -legend=n s4exp.svg -l s4exp.bld
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi



if [ ! -f snexp.fit ] || [ ! -f snexp.bld ]; then
  rm -f snexp.bld
  printf "FIT1\tfit_feng 0.0.0  (c) 2001-2011 by Turku PET Centre\n" > snexp.fit
  printf "Date:\t2018-04-20 21:33:21\n" >> snexp.fit
  printf "Data file:\tsnexp.bld\n" >> snexp.fit
  printf "Data unit:\tkBq/mL\n" >> snexp.fit
  printf "Time unit:\tmin\n" >> snexp.fit
  printf "Nr of VOIs:\t3\n" >> snexp.fit
  printf "Region        Plane\tStart\tEnd\tdataNr\tWSS\tparNr\tType\tParameters\n" >> snexp.fit
  printf "ptac 2 .\t0\t60\t50\t0.0\t5\t1312\t100\t-0.1\t10\t-0.005\t1.0\n" >> snexp.fit
  printf "ptac 3 .\t0\t60\t50\t0.0\t7\t1313\t100\t-0.2\t80\t-0.5\t250\t-0.005\t1.0\n" >> snexp.fit
  printf "ptac 4 .\t0\t60\t50\t0.0\t9\t1314\t300\t-4\t200\t-0.4\t100\t-0.04\t20\t0.0\t1.0\n" >> snexp.fit
fi

if [ ! -f snexp.bld ]; then
  fit2dat -f=samples.bld snexp.fit snexp.bld
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tac2svg -mt=snexp -legend=y snexp.svg -l snexp.bld
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi


if [ ! -f badfile.dat ]; then
  printf "# badfile\n" > badfile.dat
  printf "ugh banzai\n" >> badfile.dat
  printf "2 meh\n" >> badfile.dat
fi



# Temporarily jump over first test
#if false; then


printf "\n\n"
printf "=====================================================================\n"
printf "testing %s\n" $PROGRAM
printf "=====================================================================\n"


printf "\n 1.0.0 \n"
printf " Test case 1: Reasonable noise-free PTAC with 2 exponentials. \n"
printf " Test case 2: Option -n=2. \n"
printf " Test case 3: Fitted TAC saved with option -fit. \n"
printf " Test case 4: Fitted TAC plotted with options -svg -svg1 and -svg2. \n"
printf " Expected result 1: Parameters close to correct are saved. \n"
printf " Expected result 2: Saved fitted TAC is close to original data. \n"
printf " Expected result 3: SVG files are created. \n\n"

rm -f output.bld fit_s2exp1.svg fit_s2exp2.svg fit_s2exp.svg output.fit

$PROGRAM -n=2 -fit=output.bld -svg1=fit_s2exp1.svg -svg2=fit_s2exp2.svg -svg=fit_s2exp.svg s2exp.bld output.fit
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
parmatch -parnames=n -abs=0.000001 -rel=1 s2exp.fit output.fit
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.1 -rel=1 s2exp.bld output.bld
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
if [ ! -f fit_s2exp1.svg ] || [ ! -f fit_s2exp2.svg ]; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
if [ ! -f fit_s2exp.svg ]; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 1.0.1 \n"
printf " Test case 1: Reasonable noise-free PTAC with 3 exponentials. \n"
printf " Test case 2: Option -n=3. \n"
printf " Test case 3: Fitted TAC saved with option -fit. \n"
printf " Test case 4: Fitted TAC plotted with options -svg -svg1 and -svg2. \n"
printf " Expected result 1: Parameters close to correct are saved. \n"
printf " Expected result 2: Saved fitted TAC is close to original data. \n"
printf " Expected result 3: SVG files are created. \n\n"

rm -f output.bld fit_s3exp1.svg fit_s3exp2.svg fit_s3exp.svg output.fit

$PROGRAM -n=3 -fit=output.bld -svg1=fit_s3exp1.svg -svg2=fit_s3exp2.svg -svg=fit_s3exp.svg s3exp.bld output.fit
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
parmatch -parnames=n -abs=0.000001 -rel=1 s3exp.fit output.fit
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.1 -rel=1 s3exp.bld output.bld
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
if [ ! -f fit_s3exp1.svg ] || [ ! -f fit_s3exp2.svg ]; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
if [ ! -f fit_s3exp.svg ]; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 1.0.2 \n"
printf " Test case 1: Reasonable noise-free PTAC with 4 exponentials. \n"
printf " Test case 2: Option -n=4. \n"
printf " Test case 3: Fitted TAC saved with option -fit. \n"
printf " Test case 4: Fitted TAC plotted with options -svg -svg1 and -svg2. \n"
printf " Expected result 1: Parameters close to correct are saved. \n"
printf " Expected result 2: Saved fitted TAC is close to original data. \n"
printf " Expected result 3: SVG files are created. \n\n"

rm -f output.bld fit_s4exp1.svg fit_s4exp2.svg fit_s4exp.svg output.fit

$PROGRAM -n=4 -fit=output.bld -svg1=fit_s4exp1.svg -svg2=fit_s4exp2.svg -svg=fit_s4exp.svg s4exp.bld output.fit
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
parmatch -parnames=n -abs=0.0001 -rel=5 s4exp.fit output.fit
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.1 -rel=1 s4exp.bld output.bld
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
if [ ! -f fit_s4exp1.svg ] || [ ! -f fit_s4exp2.svg ]; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
if [ ! -f fit_s4exp.svg ]; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 1.0.3 \n"
printf " Test case 1: Reasonable noise-free PTAC with 2-4 exponentials. \n"
printf " Test case 2: Option -n=A. \n"
printf " Test case 3: Fitted TAC saved with option -fit. \n"
printf " Test case 4: Fitted TAC plotted with options -svg -svg1 and -svg2. \n"
printf " Expected result 1: Parameters close to correct are saved. \n"
printf " Expected result 2: Saved fitted TAC is close to original data. \n"
printf " Expected result 3: SVG files are created. \n\n"

rm -f output.bld fit_snexp1.svg fit_snexp2.svg fit_snexp.svg output.fit

$PROGRAM -n=A -fit=output.bld -svg1=fit_snexp1.svg -svg2=fit_snexp2.svg -svg=fit_snexp.svg snexp.bld output.fit
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
parmatch -parnames=n -abs=0.01 -rel=20 snexp.fit output.fit
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.1 -rel=1 snexp.bld output.bld
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
if [ ! -f fit_snexp1.svg ] || [ ! -f fit_snexp2.svg ]; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
if [ ! -f fit_snexp.svg ]; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n 1.1.0 \n"
printf " Test case 1: Reasonable noise-free PTAC integral with 3 exponentials. \n"
printf " Test case 2: Option -n=3 and -integ. \n"
printf " Test case 3: Fitted TAC saved with option -fit. \n"
printf " Test case 4: Fitted TAC plotted with options -svg -svg1 and -svg2. \n"
printf " Expected result 1: Parameters are saved. \n"
printf " Expected result 2: Saved fitted TAC is close to original data. \n"
printf " Expected result 3: SVG files are created. \n\n"

rm -f output.tac fit_s3exp1.svg fit_s3exp2.svg fit_s3exp.svg output.fit

$PROGRAM -integ -n=3 -safe -delay=1 -fit=output.tac -svg1=fit_s3expi1.svg -svg=fit_s3expi.svg s3expi.tac output.fit
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
parmatch -parnames=n -abs=1000 -rel=1000 s3exp.fit output.fit
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=1.0 -rel=1 s3expi.tac output.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
if [ ! -f fit_s3expi1.svg ] || [ ! -f fit_s3expi.svg ]; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

# omit previous tests
#fi

printf "\n 1.2.0 \n"
printf " Test case 1: Option -res to save parameters in result format. \n"
printf " Test case 2: One TAC with 3 exponentials. \n"
printf " Expected result: Parameters are saved in result format. \n\n"

rm -f output.res output.fit

$PROGRAM -delay=1 -res=output.res s3exp.bld output.fit
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
parmatch -parnames=n -abs=1000 -rel=1000 s3exp.fit output.fit
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

parformat -f=ift output.fit fit.ift
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
parformat -f=ift output.res output.ift
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

iftmatch -abs=0.01 fit.ift output.ift p1 A1
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftmatch -abs=0.01 fit.ift output.ift p2 L1
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftmatch -abs=0.01 fit.ift output.ift p3 A2
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftmatch -abs=0.01 fit.ift output.ift p4 L2
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftmatch -abs=0.01 fit.ift output.ift p5 A3
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftmatch -abs=0.01 fit.ift output.ift p6 L3
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftmatch -abs=0.01 fit.ift output.ift p7 dT
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n 1.2.1 \n"
printf " Test case 1: Option -res to save parameters in result format. \n"
printf " Test case 2: Three TACs with different number of exponentials. \n"
printf " Expected result: Parameters are saved in result format. \n\n"

rm -f output.res output.fit

$PROGRAM -d1 -n=A -delay=1 -fast -res=output.res snexp.bld output.fit
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

paradd -ovr -tac="ptac-2" fit1.fit output.fit
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
paradd -ovr -tac="ptac-3" fit2.fit output.fit
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
paradd -ovr -tac="ptac-4" fit3.fit output.fit
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
parformat -f=ift fit1.fit fit1.ift
parformat -f=ift fit2.fit fit2.ift
parformat -f=ift fit3.fit fit3.ift
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

paradd -ovr -tac="ptac 2" output1.res output.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
paradd -ovr -tac="ptac 3" output2.res output.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
paradd -ovr -tac="ptac 4" output3.res output.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
parformat -f=ift output1.res output1.ift
parformat -f=ift output2.res output2.ift
parformat -f=ift output3.res output3.ift
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

iftmatch -abs=0.01 fit1.ift output1.ift p1 A1
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftmatch -abs=0.01 fit1.ift output1.ift p2 L1
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftmatch -abs=0.01 fit1.ift output1.ift p3 A2
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftmatch -abs=0.01 fit1.ift output1.ift p4 L2
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftmatch -abs=0.0001 fit1.ift output1.ift p7 dT
if [ $? -ne 0 ] ; then
  iftmatch -abs=0.0001 fit1.ift output1.ift p5 dT
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi
printf "\n passed. \n\n"

iftmatch -abs=0.01 fit2.ift output2.ift p1 A1
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftmatch -abs=0.01 fit2.ift output2.ift p2 L1
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftmatch -abs=0.01 fit2.ift output2.ift p3 A2
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftmatch -abs=0.01 fit2.ift output2.ift p4 L2
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftmatch -abs=0.0001 fit2.ift output2.ift p5 dT
  if [ $? -ne 0 ] ; then
  iftmatch -abs=0.0001 fit2.ift output2.ift p7 dT
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi;
printf "\n passed. \n\n"

iftmatch -abs=0.01 fit3.ift output3.ift p1 A1
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftmatch -abs=0.01 fit3.ift output3.ift p2 L1
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftmatch -abs=0.01 fit3.ift output3.ift p3 A2
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftmatch -abs=0.01 fit3.ift output3.ift p4 L2
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftmatch -abs=0.0001 fit3.ift output3.ift p9 dT
if [ $? -ne 0 ] ; then
  iftmatch -abs=0.0001 fit3.ift output3.ift p7 dT
  if [ $? -ne 0 ] ; then
    iftmatch -abs=0.0001 fit3.ift output3.ift p5 dT
    if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  fi
fi
printf "\n passed. \n\n"



printf "\n===================================================================\n"
printf "   Constraints \n"
printf "===================================================================\n"

printf "\n 2.0.0 \n"
printf " Test case: constraint file is created with option -lim=fname.\n"
printf " Expected result: File is created with correct parameters. \n\n"

rm -f output.par
$PROGRAM -lim=output.par
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

iftisval output.par A1_lower
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
iftisval output.par A1_upper
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
iftisval output.par L4_lower
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
iftisval output.par L4_upper
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
iftisval output.par DT_lower 0
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
iftisval output.par DT_upper 5
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 2.0.1 \n"
printf " Test case: existing constraint file and option -lim=fname.\n"
printf " Expected result: Error. \n\n"

$PROGRAM -lim=output.par 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Error" "parameter constraint file output.par exists."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 2.0.2 \n"
printf " Test case: trying to use nonexisting constraint file.\n"
printf " Expected result: Error. \n\n"

$PROGRAM -lim=nonexisting.par s2exp.bld output.fit 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Error in reading 'nonexisting.par'" "cannot open file"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 2.0.3 \n"
printf " Test case: trying to use constraint file with bad contents.\n"
printf " Expected result: Error. \n\n"

$PROGRAM -lim=badfile.dat s2exp.bld output.fit 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Error in reading 'badfile.dat'" "file contains no data"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 2.1.0 \n"
printf " Test case 1: set constraints for all parameters with -lim.\n"
printf " Expected result 1: Limits are set correctly. \n"
printf " Expected result 2: With reasonable limits the fit is good. \n\n"

rm -f limit.par
rm -f output.fit output.bld

printf "A1_lower := 250\n" >> limit.par
printf "A1_upper := 350\n" >> limit.par
printf "L1_lower := -5\n" >> limit.par
printf "L1_upper := -3\n" >> limit.par

printf "A2_lower := 150\n" >> limit.par
printf "A2_upper := 250\n" >> limit.par
printf "L2_lower := -0.5\n" >> limit.par
printf "L2_upper := -0.3\n" >> limit.par

printf "A3_lower := 50\n" >> limit.par
printf "A3_upper := 150\n" >> limit.par
printf "L3_lower := -0.05\n" >> limit.par
printf "L3_upper := -0.03\n" >> limit.par

printf "A4_lower := 0\n" >> limit.par
printf "A4_upper := 50\n" >> limit.par
printf "L4_lower := 0\n" >> limit.par
printf "L4_upper := 0\n" >> limit.par

printf "DT_lower := 0.9\n" >> limit.par
printf "DT_upper := 1.1\n" >> limit.par

$PROGRAM -d2 -n=4 -lim=limit.par -fast -svg=output.svg -fit=output.bld s4exp.bld output.fit 1>stdout.txt 2>&1
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt constraints "[250,350] [150,250] [50,150] [0,50] [-5,-3] [-0.5,-0.3] [-0.05,-0.03] [0,0] [0.9,1.1]"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

parmatch -parnames=n -abs=0.0001 -rel=2 s4exp.fit output.fit
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.1 -rel=1 s4exp.bld output.bld
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 2.1.1 \n"
printf " Test case 1: Option -delay overwrites limits given with lim file.\n"
printf " Expected result 1: Limits are set correctly. \n"
printf " Expected result 2: With reasonable limits the fit is good. \n\n"

rm -f output.fit output.bld

$PROGRAM -d2 -n=4 -lim=limit.par -delay=1 -fast -svg=output.svg -fit=output.bld s4exp.bld output.fit 1>stdout.txt 2>&1
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt constraints "[250,350] [150,250] [50,150] [0,50] [-5,-3] [-0.5,-0.3] [-0.05,-0.03] [0,0] [1,1]"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

parmatch -parnames=n -abs=0.00001 -rel=0.1 s4exp.fit output.fit
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.1 -rel=1 s4exp.bld output.bld
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n 2.2.0 \n"
printf " Test case: Constraint file works with several TACs with option -n=a.\n"
printf " Expected result: With reasonable limits the fit is good. \n\n"

rm -f limit.par
rm -f output.bld fit_snexp.svg output.fit

printf "A1_lower := 90\n" >> limit.par
printf "A1_upper := 310\n" >> limit.par
printf "L1_lower := -4.5\n" >> limit.par
printf "L1_upper := -0.09\n" >> limit.par

printf "A2_lower := 9\n" >> limit.par
printf "A2_upper := 210\n" >> limit.par
printf "L2_lower := -0.6\n" >> limit.par
printf "L2_upper := -0.001\n" >> limit.par

printf "A3_lower := 90\n" >> limit.par
printf "A3_upper := 300\n" >> limit.par
printf "L3_lower := -0.05\n" >> limit.par
printf "L3_upper := -0.0001\n" >> limit.par

printf "A4_lower := 0\n" >> limit.par
printf "A4_upper := 40\n" >> limit.par
printf "L4_lower := 0\n" >> limit.par
printf "L4_upper := 0\n" >> limit.par

printf "DT_lower := 1.0\n" >> limit.par
printf "DT_upper := 1.0\n" >> limit.par

$PROGRAM -n=a -lim=limit.par -fast -svg=fit_snexp.svg -fit=output.bld s4exp.bld output.fit
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

parmatch -parnames=n -abs=0.0001 -rel=2 s4exp.fit output.fit
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.01 -rel=1 s4exp.bld output.bld
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

# omit previous tests
#fi


printf "\n===================================================================\n"
printf "   Tests with real data \n"
printf "===================================================================\n"


printf "\n 3.0.0 \n"
printf " Test case: Typical carfentanil PTAC. \n"
printf " Expected result: Fitted TAC close to measured TAC. \n\n"

fname=c11-carfentanil_metabcorr
pfname='../ptac/'$fname

$PROGRAM -wf -svg=$fname'.svg' -fit=$fname'.bld' $pfname'.kbq' $fname'.fit'
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=3 -rel=20 $pfname'.kbq' $fname'.bld'
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
interpol -i -x=3,10,20,50 $pfname'.kbq' correct.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
interpol -i -x=3,10,20,50 $fname'.bld' output.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
tacmatch -abs=0.2 -rel=1 correct.dat output.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n 3.1.0 \n"
printf " Test case: Typical flumazenil PTAC. \n"
printf " Expected result: Fitted TAC close to measured TAC. \n\n"

fname=c11-flumazenil_metabcorr
pfname='../ptac/'$fname

rm -f flumazenil.lim

printf "A1_lower := 500\n" >> flumazenil.lim
printf "A1_upper := 2000\n" >> flumazenil.lim
printf "L1_lower := -3\n" >> flumazenil.lim
printf "L1_upper := -0.5\n" >> flumazenil.lim

printf "A2_lower := 1000\n" >> flumazenil.lim
printf "A2_upper := 6000\n" >> flumazenil.lim
printf "L2_lower := -4\n" >> flumazenil.lim
printf "L2_upper := -0.5\n" >> flumazenil.lim

printf "A3_lower := 1\n" >> flumazenil.lim
printf "A3_upper := 40\n" >> flumazenil.lim
printf "L3_lower := -0.2\n" >> flumazenil.lim
printf "L3_upper := -0.01\n" >> flumazenil.lim

printf "A4_lower := 0\n" >> flumazenil.lim
printf "A4_upper := 5\n" >> flumazenil.lim
printf "L4_lower := -0.02\n" >> flumazenil.lim
printf "L4_upper := 0.0\n" >> flumazenil.lim

printf "DT_lower := 0.4\n" >> flumazenil.lim
printf "DT_upper := 0.7\n" >> flumazenil.lim


$PROGRAM -wf -n=4 -lim=flumazenil.lim -svg=$fname'.svg' -fit=$fname'.bld' $pfname'.kbq' $fname'.fit'
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=20 -rel=30 $pfname'.kbq' $fname'.bld'
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
interpol -i -x=3,20,40,90 $pfname'.kbq' correct.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
interpol -i -x=3,20,40,90 $fname'.bld' output.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
tacmatch -abs=2 -rel=15 correct.dat output.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 3.2.0 \n"
printf " Test case: Typical MP4A PTAC. \n"
printf " Expected result: Fitted TAC close to measured TAC. \n\n"

fname=c11-mp4b_metabcorr
pfname='../ptac/'$fname

$PROGRAM -wf -n=4 -svg=$fname'.svg' -fit=$fname'.bld' $pfname'.kbq' $fname'.fit'
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=10 -rel=20 $pfname'.kbq' $fname'.bld'
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
interpol -i -x=3,15,40 $pfname'.kbq' correct.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
interpol -i -x=3,15,40 $fname'.bld' output.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
tacmatch -abs=1 -rel=10 correct.dat output.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 3.3.0 \n"
printf " Test case: Typical raclopride PTAC. \n"
printf " Expected result: Fitted TAC close to measured TAC. \n\n"

fname=c11-raclopride_metabcorr
pfname='../ptac/'$fname

$PROGRAM -wf -n=4 -svg=$fname'.svg' -fit=$fname'.bld' $pfname'.kbq' $fname'.fit'
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=10 -rel=20 $pfname'.kbq' $fname'.bld'
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
interpol -i -x=3,15,40 $pfname'.kbq' correct.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
interpol -i -x=3,15,40 $fname'.bld' output.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
tacmatch -abs=1 -rel=10 correct.dat output.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n 3.4.0 \n"
printf " Test case: Typical WAY100635 PTAC. \n"
printf " Expected result: Fitted TAC close to measured TAC. \n\n"

fname=c11-way100635_metabcorr
pfname='../ptac/'$fname

rm -f way100635.lim

printf "A1_lower := 500\n" >> way100635.lim
printf "A1_upper := 5000\n" >> way100635.lim
printf "L1_lower := -20\n" >> way100635.lim
printf "L1_upper := -1\n" >> way100635.lim

printf "A2_lower := 50\n" >> way100635.lim
printf "A2_upper := 500\n" >> way100635.lim
printf "L2_lower := -10\n" >> way100635.lim
printf "L2_upper := -0.5\n" >> way100635.lim

printf "A3_lower := 10\n" >> way100635.lim
printf "A3_upper := 100\n" >> way100635.lim
printf "L3_lower := -1\n" >> way100635.lim
printf "L3_upper := -0.1\n" >> way100635.lim

printf "A4_lower := 0\n" >> way100635.lim
printf "A4_upper := 50\n" >> way100635.lim
printf "L4_lower := -0.1\n" >> way100635.lim
printf "L4_upper := -0.01\n" >> way100635.lim

printf "DT_lower := 0.33\n" >> way100635.lim
printf "DT_upper := 0.37\n" >> way100635.lim


$PROGRAM -wf -n=4 -lim=way100635.lim -svg=$fname'.svg' -svg1=$fname'_1.svg' -svg2=$fname'_2.svg' -fit=$fname'.bld' $pfname'.kbq' $fname'.fit'
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=2 -rel=20 $pfname'.kbq' $fname'.bld'
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
interpol -i -x=3,15,40 $pfname'.kbq' correct.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
interpol -i -x=3,15,40 $fname'.bld' output.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
tacmatch -abs=1 -rel=10 correct.dat output.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n 3.5.0 \n"
printf " Test case: Typical 18F-choline PTAC. \n"
printf " Expected result: Fitted TAC close to measured TAC. \n\n"

fname=f18-choline_metabcorr
pfname='../ptac/'$fname

rm -f fchol.lim

printf "A1_lower := 500\n" >> fchol.lim
printf "A1_upper := 5000\n" >> fchol.lim
printf "L1_lower := -5\n" >> fchol.lim
printf "L1_upper := -0.1\n" >> fchol.lim

printf "A2_lower := 10\n" >> fchol.lim
printf "A2_upper := 1000\n" >> fchol.lim
printf "L2_lower := -1\n" >> fchol.lim
printf "L2_upper := -0.1\n" >> fchol.lim

printf "A3_lower := 1\n" >> fchol.lim
printf "A3_upper := 100\n" >> fchol.lim
printf "L3_lower := -0.5\n" >> fchol.lim
printf "L3_upper := -0.01\n" >> fchol.lim

printf "A4_lower := 0.1\n" >> fchol.lim
printf "A4_upper := 1.0\n" >> fchol.lim
printf "L4_lower := -0.01\n" >> fchol.lim
printf "L4_upper := -0.0001\n" >> fchol.lim

printf "DT_lower := 0.45\n" >> fchol.lim
printf "DT_upper := 0.50\n" >> fchol.lim


$PROGRAM -wf -n=4 -lim=fchol.lim -svg=$fname'.svg' -svg1=$fname'_1.svg' -svg2=$fname'_2.svg' -fit=$fname'.bld' $pfname'.kbq' $fname'.fit'
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=2 -rel=40 $pfname'.kbq' $fname'.bld'
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
interpol -i -x=3,20,60,100 $pfname'.kbq' correct.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
interpol -i -x=3,20,60,100 $fname'.bld' output.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
tacmatch -abs=1 -rel=10 correct.dat output.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n 3.6.0 \n"
printf " Test case: Typical FBPA PTAC. \n"
printf " Expected result: Fitted TAC close to measured TAC. \n\n"

fname=f18-fbpa_totplasma
pfname='../ptac/'$fname

$PROGRAM -wf -svg=$fname'.svg' -svg1=$fname'_1.svg' -svg2=$fname'_2.svg' -fit=$fname'.bld' $pfname'.kbq' $fname'.fit'
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=2 -rel=10 $pfname'.kbq' $fname'.bld'
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
interpol -i -x=3,15,40 $pfname'.kbq' correct.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
interpol -i -x=3,15,40 $fname'.bld' output.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
tacmatch -abs=1 -rel=10 correct.dat output.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n 3.7.0 \n"
printf " Test case: Typical FDG arterial PTAC. \n"
printf " Expected result: Fitted TAC close to measured TAC. \n\n"

fname=f18-fdg_arteriaplasma
pfname='../ptac/'$fname

$PROGRAM -wf -n=4 -svg=$fname'.svg' -svg1=$fname'_1.svg' -svg2=$fname'_2.svg' -fit=$fname'.bld' $pfname'.kbq' $fname'.fit'
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=15 -rel=25 $pfname'.kbq' $fname'.bld'
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
interpol -i -x=5,15,60 $pfname'.kbq' correct.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
interpol -i -x=5,15,60 $fname'.bld' output.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
tacmatch -abs=1 -rel=10 correct.dat output.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 3.7.1 \n"
printf " Test case: Typical FDG venous PTAC. \n"
printf " Expected result: Fitted TAC close to measured TAC. \n\n"

fname=f18-fdg_venaplasma
pfname='../ptac/'$fname

$PROGRAM -wf -n=4 -svg=$fname'.svg' -svg1=$fname'_1.svg' -svg2=$fname'_2.svg' -fit=$fname'.bld' $pfname'.kbq' $fname'.fit'
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=5 -rel=15 $pfname'.kbq' $fname'.bld'
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
interpol -i -x=5,20,50 $pfname'.kbq' correct.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
interpol -i -x=5,20,50 $fname'.bld' output.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
tacmatch -abs=1 -rel=10 correct.dat output.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n 3.8.0 \n"
printf " Test case: Typical FDOPA PTAC. \n"
printf " Expected result: Fitted TAC close to measured TAC. \n\n"

fname=f18-fdopa_metabcor
pfname='../ptac/'$fname

rm -f fdopa.lim

printf "A1_lower := 500\n" >> fdopa.lim
printf "A1_upper := 2000\n" >> fdopa.lim
printf "L1_lower := -5\n" >> fdopa.lim
printf "L1_upper := -0.5\n" >> fdopa.lim

printf "A2_lower := 500\n" >> fdopa.lim
printf "A2_upper := 5000\n" >> fdopa.lim
printf "L2_lower := -4\n" >> fdopa.lim
printf "L2_upper := -0.1\n" >> fdopa.lim

printf "A3_lower := 1\n" >> fdopa.lim
printf "A3_upper := 100\n" >> fdopa.lim
printf "L3_lower := -0.5\n" >> fdopa.lim
printf "L3_upper := -0.01\n" >> fdopa.lim

printf "A4_lower := 0.1\n" >> fdopa.lim
printf "A4_upper := 10.0\n" >> fdopa.lim
printf "L4_lower := -0.05\n" >> fdopa.lim
printf "L4_upper := -0.0001\n" >> fdopa.lim

printf "DT_lower := 0.5\n" >> fdopa.lim
printf "DT_upper := 0.7\n" >> fdopa.lim

$PROGRAM -wf -n=4 -lim=fdopa.lim -svg=$fname'.svg' -svg1=$fname'_1.svg' -svg2=$fname'_2.svg' -fit=$fname'.bld' $pfname'.kbq' $fname'.fit'
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=25 -rel=25 $pfname'.kbq' $fname'.bld'
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
interpol -i -x=5,30,90,200 $pfname'.kbq' correct.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
interpol -i -x=5,30,90,200 $fname'.bld' output.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
tacmatch -abs=1 -rel=10 correct.dat output.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"




printf "\n 3.9.0 \n"
printf " Test case: Radiowater BTAC from muscle study. \n"
printf " Expected result: Fitted TAC close to measured TAC. \n\n"

fname=o15-h2o_muscle
pfname='../ptac/'$fname

$PROGRAM -svg=$fname'.svg' -svg1=$fname'_1.svg' -svg2=$fname'_2.svg' -fit=$fname'.bld' $pfname'.kbq' $fname'.fit'
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
#tacmatch -abs=150 -rel=90 $pfname'.kbq' $fname'.bld'
#if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
#printf "\n ok \n"
interpol -i -x=60,180,360 $pfname'.kbq' correct.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
interpol -i -x=60,180,360 $fname'.bld' output.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
tacmatch -abs=10 -rel=5 correct.dat output.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 3.9.1 \n"
printf " Test case: Radiowater BTAC from oncology study. \n"
printf " Expected result: Fitted TAC close to measured TAC. \n\n"

fname=o15-h2o_onkol
pfname='../ptac/'$fname

$PROGRAM -svg=$fname'.svg' -svg1=$fname'_1.svg' -svg2=$fname'_2.svg' -fit=$fname'.bld' $pfname'.kbq' $fname'.fit'
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=150 -rel=20 $pfname'.kbq' $fname'.bld'
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
interpol -i -x=90,180,400 $pfname'.kbq' correct.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
interpol -i -x=90,180,400 $fname'.bld' output.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
tacmatch -abs=10 -rel=5 correct.dat output.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n 3.10.0 \n"
printf " Test case 1: O2-study for muscle. \n"
printf " Expected result: Fitted TAC close to measured TAC. \n\n"

fname=o15-o2_metabcorr_delaycorr_muscle
pfname='../ptac/'$fname

$PROGRAM -svg=$fname'.svg' -svg1=$fname'_1.svg' -fit=$fname'.bld' $pfname'.kbq' $fname'.fit'
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
#tacmatch -abs=150 -rel=20 $pfname'.kbq' $fname'.bld'
#if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
#printf "\n ok \n"
interpol -i -x=60,180,440 $pfname'.kbq' correct.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
interpol -i -x=60,180,440 $fname'.bld' output.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
tacmatch -abs=10 -rel=5 correct.dat output.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 3.10.1 \n"
printf " Test case: O2-study for muscle. \n"
printf " Expected result: Fitted TAC close to measured TAC. \n\n"

fname=o15-o2_totblood_muscle
pfname='../ptac/'$fname

$PROGRAM -svg=$fname'.svg' -svg1=$fname'_1.svg' -fit=$fname'.bld' $pfname'.kbq' $fname'.fit'
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
#tacmatch -abs=150 -rel=20 $pfname'.kbq' $fname'.bld'
#if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
#printf "\n ok \n"
interpol -i -x=60,180,450 $pfname'.kbq' correct.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
interpol -i -x=60,180,450 $fname'.bld' output.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
tacmatch -abs=10 -rel=5 correct.dat output.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 3.10.2 \n"
printf " Test case: O2-study for brain. \n"
printf " Expected result: Fitted TAC close to measured TAC. \n\n"

fname=o15-o2_totblood_brain
pfname='../ptac/'$fname

$PROGRAM -svg=$fname'.svg' -svg1=$fname'_1.svg' -fit=$fname'.bld' $pfname'.kbq' $fname'.fit'
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
#tacmatch -abs=150 -rel=20 $pfname'.kbq' $fname'.bld'
#if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
#printf "\n ok \n"
interpol -i -x=60,180,300 $pfname'.kbq' correct.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
interpol -i -x=60,180,300 $fname'.bld' output.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
tacmatch -abs=10 -rel=5 correct.dat output.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n 3.11.0 \n"
printf " Test case 1: Bladder TAC from FDG mouse study. \n"
printf " Test case 2: Option -integ because bladder represents AUC of FDG in plasma. \n"
printf " Expected result 1: Fitted TAC reasonably close to measured TAC. \n"
printf " Expected result 2: Fit can be used to calculate TAC with somewhat representing the shape of plasma TAC.\n\n"

fname=ae392wholebladder

$PROGRAM -d3 -integ -wf -delay=1 -svg=$fname'.svg' -svg1=$fname'_1.svg' -fit=$fname'_fitted.tac' $fname'.tac' $fname'.fit'
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=2.5E+05 -rel=5 $fname'.tac' $fname'_fitted.tac'
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
fit2dat -a=90 $fname'.fit' $fname'_ap.bld'
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tac2svg -mt="PTAC based on bladder" -legend=n $fname'_ap.svg' -l $fname'_ap.bld'
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
fit2dat -a=5 $fname'.fit' $fname'_ap1.bld'
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tac2svg -mt="Initial phase of PTAC based on bladder" -legend=n $fname'_ap1.svg' -l $fname'_ap1.bld'
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


# omit previous tests
#fi


printf "\n===================================================================\n"
printf "   All passed! \n"
printf "===================================================================\n"
exit 0

