#!/bin/bash
#: Title      : test_fit_fexp
#: Date       : 2023-06-29
#: Author     : "Vesa Oikonen" <vesa.oikonen@utu.fi>
#: Options    : None

# Set the name of executable to test
if [ -n "${OS+1}" ] && [ "$OS" = "Windows_NT" ]; then
EXT=.exe;
else
EXT=;
fi
PROGRAM=../../fit_fexp$EXT;
if [ ! -f $PROGRAM ]; then
  printf "Failed: executable does not exist.\n"
  exit 1
fi

printf "=====================================================================\n"
printf "creating test data for %s\n" $PROGRAM
printf "=====================================================================\n"


printf "\n verifying that required data exists \n"

if [ ! -f data1.dat ] || [ ! -f data1_3.dat ] || [ ! -f data1_4.dat ]; then
  printf "Failed: required data does not exist.\n"
  exit 1
fi
printf "ok\n"
if [ ! -f data2.dat ] ; then
  printf "Failed: required data does not exist.\n"
  exit 1
fi
printf "ok\n"
if [ ! -f data3_1.dat ] || [ ! -f data4_1.dat ] ; then
  printf "Failed: required data does not exist.\n"
  exit 1
fi
printf "ok\n"
if [ ! -f data5_1.dat ] || [ ! -f data5_2.dat ] ; then
  printf "Failed: required data does not exist.\n"
  exit 1
fi
printf "ok\n"
if [ ! -f data6_1.dat ] || [ ! -f data6_2.dat ] ; then
  printf "Failed: required data does not exist.\n"
  exit 1
fi
printf "ok\n"


printf "\n creating test data \n"

if [ ! -f mono.fit ] || [ ! -f mono.dat ] ; then
  printf "FIT1        fit_fexp 0.7.4 (c) 2019 by Turku PET Centre\n" > mono.fit
  printf "Date:	2019-12-19 21:41:37\n" >> mono.fit
  printf "Data file:	mono.dat\n" >> mono.fit
  printf "Data unit:	unitless\n" >> mono.fit
  printf "Time unit:	min\n" >> mono.fit
  printf "Nr of VOIs:	1\n" >> mono.fit
  printf "Region Plane	Start	End	dataNr	WSS	parNr	Type	Parameters\n" >> mono.fit
  printf "1 . .	0.000	60.000	8	0	4	0302	0.25	-0.02	0.7	0\n" >> mono.fit

  fit2dat -x=2,5,10,20,40,60 mono.fit mono.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  fit2dat -x=0,2,5,10,20,40,60 mono.fit monoz.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi



printf "\n\n"
printf "=====================================================================\n"
printf "testing %s\n" $PROGRAM
printf "=====================================================================\n"



printf "\n 0.1.0 \n"
printf " Test case: Unknown option. \n"
printf " Expected result: Error. \n"

$PROGRAM -stupidoption data1.dat 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error "invalid option '-stupidoption'."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n 0.2.0 \n"
printf " Test case: Defaults. \n"
printf " Expected result 1: Default values of variables are correct. \n"
printf " Expected result 2: Fitted TAC is close to original. \n"

rm -f output.fit output.dat

$PROGRAM -d2 data1.dat output.fit 1>stdout.txt 2>&1
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

iftisval stdout.txt noDivide "0"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt weighting "0"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt last_col_is_weight "0"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt type "351"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt MRL_check "0"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

fit2dat -f=data1.dat output.fit output.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"


tacmatch -abs=0.01 -rel=1 -x=y -y=y data1.dat output.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"




printf "\n 0.2.1 \n"
printf " Test case: Options are correctly interpreted. \n"
printf " Expected result: Values of variables are correct. \n"

$PROGRAM -d2 -wf -mrl -nd -svg=test.svg data1.dat 1>stdout.txt 2>&1
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

iftisval stdout.txt noDivide "1"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt weighting "2"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt last_col_is_weight "0"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt MRL_check "1"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt svgfile "test.svg"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n 0.3.0 \n"
printf " Test case: Appropriate arguments entered. \n"
printf " Expected result: Arguments are correctly identified. \n"

$PROGRAM -d2 data1.dat test.fit 1>stdout.txt 2>&1
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt dfile "data1.dat"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt ffile "test.fit"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 0.3.1 \n"
printf " Test case: Extra argument entered. \n"
printf " Expected result: Error. \n"

$PROGRAM -d2 data1.dat test.fit extra 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error "too many arguments: 'extra'."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n\n---------------------------------------------------------------\n\n"

printf "\n 1.0.0 \n"
printf " Test case: Input file does not exist. \n"
printf " Expected result: Error. \n"

$PROGRAM -d2 nonexistingfile.dat test.fit 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Error in reading 'nonexistingfile.dat'" "cannot open file"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 1.1.0 \n"
printf " Test case: Input file in wrong format. \n"
printf " Expected result: Error. \n"

echo meh ngh      >  test.dat
echo zzzzhhhh bop >> test.dat
$PROGRAM -d2 test.dat test.fit 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Error in reading 'test.dat'" "wrong format"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 1.2.0 \n"
printf " Test case: Input file contains more than one fraction curve. \n"
printf " Expected result: First is used, and warning is printed. \n\n"

rm -f output.fit output2.fit
rm -f output.dat output2.dat

$PROGRAM data1_3.dat output.fit
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
$PROGRAM data1_3.dat output2.fit 1>stdout.txt 2>&1
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Warning "extra columns in data1_3.dat are ignored."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

fit2dat -f=data1_3.dat output.fit output.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fit2dat -f=data1_3.dat output2.fit output2.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
tacmatch -abs=0.01 -rel=1 -x=y -y=y output.dat output2.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 1.3.0 \n"
printf " Test case: Fraction file contains less than 3 samples. \n"
printf " Expected result: Error. \n\n"

$PROGRAM data1_4.dat output.fit
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n 1.4.0 \n"
printf " Test case: Fraction file contains percentages instead of fractions. \n"
printf " Expected result: Automatic conversion to fractions with warning. \n\n"

rm -f output.fit output.dat

$PROGRAM -svg=data5_1.svg data5_1.dat output.fit 1>stdout.txt 2>&1
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

iftisval stdout.txt Warning "converting percentages to fractions."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

fit2dat -f=data5_1.dat output.fit output.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
taccalc output.dat x 100 output2.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.01 -rel=1 -x=y -y=y data5_1.dat output2.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n 1.4.1 \n"
printf " Test case: Fraction file contains one value larger than 1. \n"
printf "            Option -nd is used to prevent automatic scaling. \n"
printf " Expected result: Scale is not converted. \n\n"

rm -f output.fit output.dat

$PROGRAM -svg=data5_2.svg -nd data5_2.dat output.fit 1>stdout.txt 2>&1
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

iftisval stdout.txt Warning "converting percentages to fractions."
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

fit2dat -f=data5_1.dat output.fit output.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
taccalc data5_1.dat x 0.01 temp.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.01 -rel=1 -x=y -y=y temp.dat output.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"





printf "\n 1.5.0 \n"
printf " Test case: Missing value in fraction file without weights. \n"
printf " Expected result: Missing value gets weight 0, others 1. \n\n"

$PROGRAM -d3 -svg=data6_1.svg data6_1.dat output.fit 1>stdout.txt 2>&1
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

iftisval stdout.txt data_weights "1, 1, 1, 1, 0, 1, 1, 1"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"


printf "\n 1.5.1 \n"
printf " Test case: Missing value in fraction file with weights. \n"
printf " Expected result: Missing value gets weight 0, others as in file. \n\n"

$PROGRAM -d3 -svg=data6_2.svg data6_2.dat output.fit 1>stdout.txt 2>&1
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

iftisval stdout.txt data_weights "1, 2, 3, 0, 5, 6, 7, 7"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"


printf "\n 1.5.2 \n"
printf " Test case: Missing value in fraction file with option -w1. \n"
printf " Expected result: Missing value gets weight 0, others 1. \n\n"

$PROGRAM -d3 -w1 data6_1.dat output.fit 1>stdout.txt 2>&1
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

iftisval stdout.txt data_weights "1, 1, 1, 1, 0, 1, 1, 1"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"



printf "\n\n===============================================================\n\n"


printf "\n 1.6.0 \n"
printf " Test case: Filename for fits is not given. \n"
printf " Expected result: Fits are written in stdout. \n\n"

rm -f output1.fit output2.fit

$PROGRAM data1.dat output1.fit
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

$PROGRAM data1.dat > output2.fit
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

fit2dat -f=data1.dat output1.fit output1.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
fit2dat -f=data2.dat output2.fit output2.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

tacmatch -abs=0.01 -rel=1 -x=y -y=y output1.dat output2.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n\n===============================================================\n\n"


printf "\n 1.7.0 \n"
printf " Test case 1: Option -mono. \n"
printf " Expected result: Correct fit. \n\n"

rm -f output.fit mono.svg

$PROGRAM -mono -svg=mono.svg mono.dat output.fit
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

fit2dat -f=mono.dat output.fit output.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.01 -rel=1 -x=y -y=y mono.dat output.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

parformat -f=IFT output.fit
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval -abs=0.001 output.ift p1 0.25
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval -abs=0.001 output.ift p2 -0.02
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval -abs=0.001 output.ift p3 0.7
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 1.7.1 \n"
printf " Test case 1: Option -mono. \n"
printf " Test case 2: Option -a=1. \n"
printf " Expected result: Fit starts from one. \n\n"

rm -f output.fit mono0.svg

$PROGRAM -mono -a=1 -svg=mono1.svg mono.dat output.fit
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

fit2dat -x=0 output.fit output.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
taclist -ift -nv -nn output.dat > output.ift
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
iftisval output.ift value[1][1] 1
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n 1.7.2 \n"
printf " Test case 1: Option -mono. \n"
printf " Test case 2: Option -a=zero. \n"
printf " Expected result: Correct fit. \n\n"

rm -f output.fit monoz.svg

$PROGRAM -mono -a=zero -svg=monoz.svg monoz.dat output.fit
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

fit2dat -f=monoz.dat output.fit output.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.001 -rel=0.1 -x=y -y=y monoz.dat output.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n\n===============================================================\n\n"


printf "\n 2.0.0 \n"
printf " Test case: File contains weights. \n"
printf " Expected result: Weights are identified. \n\n"

$PROGRAM -d3 data3_1.dat output.fit 1>stdout.txt 2>&1
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

iftisval stdout.txt data_weights "1, 2, 3, 4, 5, 6, 7, 7"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 2.0.1 \n"
printf " Test case: File does not contain weights. \n"
printf " Expected result: Weights are 1. \n\n"

$PROGRAM -d3 data1.dat output.fit 1>stdout.txt 2>&1
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

iftisval stdout.txt data_weights "1, 1, 1, 1, 1, 1, 1, 1"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n 2.0.1 \n"
printf " Test case: File contains (stupid) weights, override by option -w1. \n"
printf " Expected result: Weights are correct. \n\n"

rm -f output.fit output.dat

$PROGRAM -d3 -w1 data4_1.dat output.fit 1>stdout.txt 2>&1
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

iftisval stdout.txt data_weights "1, 1, 1, 1, 1, 1, 1, 1, 1"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

fit2dat -f=data1.dat output.fit output.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.01 -rel=1 -x=y -y=y data1.dat output.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"




printf "\n===================================================================\n"
printf "   All passed! \n"
printf "===================================================================\n"
exit 0

