/******************************************************************************
 * This file is not compiled into the library, but it contains main()
 * which is compiled to an executable, used to test the library functions. 
 *****************************************************************************/
//#pragma pack(1)
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include "libtpcmisc.h"
#include "libtpcimgio.h"
/*****************************************************************************/
#include <unistd.h>
/*****************************************************************************/

/*****************************************************************************/
#ifndef MAXVAL
#define MAXVAL 100000;
#endif
/*****************************************************************************/

/*****************************************************************************/
/* Test function declarations: */
int test_create_img(IMG *img, int dim_x, int dim_y, int dim_z, int dim_t);
int test_img_io(int VERBOSE);
/*****************************************************************************/

/*****************************************************************************/
static char *info[] = {
  "Usage: @P [options]",
  " ",
  "Options:",
  " -stdoptions", // List standard options like --help, -v, etc
  " -t, --test",
  "     Run all tests for library functions.",
  0};
/*****************************************************************************/

/*****************************************************************************/
/** Run unit tests to the library functions
 *  @author Vesa Oikonen
 *  @return 0 if all tests pass, otherwise >0.
 * */
int main(
  /** Nr of arguments */
  int argc,
  /** Pointer to arrays of argument string */
  char *argv[ ]
) {
  int i, help=0, version=0, verbose=1, error=0, test=0;
  int ret;
  char *cptr;

  if(argc==1) {tpcPrintUsage(argv[0], info, stdout); return(0);}
  for(i=1; i<argc; i++) {
    if(tpcProcessStdOptions(argv[i], &help, &version, &verbose)==0) continue;
    cptr=argv[i]; if(*cptr=='-') cptr++; if(*cptr=='-') cptr++;
    if(strncasecmp(cptr, "TEST", 1)==0) {
      test=1; continue;
    } else {
      error++; break;
    }
  }
  if(error>0) {
    fprintf(stderr, "Error: specify --help for usage.\n");
    return(1);
  }
  /* Print help or version? */
  if(help) {tpcPrintUsage(argv[0], info, stdout); return(0);}
  if(version) {tpcPrintBuild(argv[0], stdout); return(0);}

  if(test==0) return(0);

  if(verbose>0) printf("running tests for library functions...\n");

  i=10;

  /* img i/o */
  i++; ret=test_img_io(verbose);
  if(ret!=0) {fprintf(stderr, "failed (%d).\n", ret); return(i);}

  if(verbose>0) printf("\nAll tests passed.\n\n");
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/*!
 * Create IMG for testing purposes.
 *
 * @param img	Pointer to an initiated but not allocated IMG
 * @param dim_x	X dimension (nr of columns)
 * @param dim_y	Y dimension (nr of rows)
 * @param dim_z	Z dimension (nr of planes)
 * @param dim_t time dimension (nr of time frames, or gates)
 * @returns 0 if IMG could be created, or nonzero in case of error.
 */
int test_create_img(IMG *img, int dim_x, int dim_y, int dim_z, int dim_t) {
  int zi, xi, yi, fi;
  float f=-100.0;

  if(img==NULL || img->status!=IMG_STATUS_INITIALIZED) return(1);
  if(imgAllocate(img, dim_z, dim_y, dim_x, dim_t)){
    printf("    \nError in image allocation: imgAllocate() @ img.c\n");
    return(1);
  }
  //printf("  allocated\n");
  img->dimt=dim_t; img->dimx=dim_x; img->dimy=dim_y; img->dimz=dim_z;
  for(zi=0; zi<img->dimz; zi++) img->planeNumber[zi]=zi+1;
  img->type = IMG_TYPE_IMAGE;
  for(fi=0; fi<dim_t; fi++) {
    if(fi==0) img->start[fi]=0.0; else img->start[fi]=img->end[fi-1];
    img->end[fi]=img->start[fi]+6.0;
    img->mid[fi]=0.5*(img->start[fi]+img->end[fi]);
  }
  //printf("  frames set\n");
  for(zi=0; zi<dim_z; zi++) {
    //printf("  zi=%d\n", zi);
    f+=(float)(3*zi);
    for(yi=0; yi<dim_y; yi++) {
      f+=(float)(2*yi);
      for(xi=0; xi<dim_x; xi++) {
        f+=(float)(xi);
        if(f>10000.0) f=-100.0; // prevent error when floats are scaled to short ints
        for(fi=0; fi<dim_t; fi++) {
          img->m[zi][yi][xi][fi]=f+(float)fi;
        }
      }
    }
  }
  //printf("  pixel values set\n"); fflush(stdout);
  img->unit=CUNIT_KBQ_PER_ML;
  img->scanStart=time(NULL);
  img->decayCorrection=IMG_DC_CORRECTED;
  img->isotopeHalflife=HL_C11*60.;
  imgSetDecayCorrFactors(img, 1);
  //printf("  decay correction factors set\n");

  img->xform[0]=NIFTI_XFORM_UNKNOWN; // qform
  img->xform[1]=NIFTI_XFORM_SCANNER_ANAT; // sform

  return(0);
}
/******************************************************************************/

/******************************************************************************/
/*!
   Testbench for image IO functions. Tests include:<br>
   ECAT7 image write and read<br>
   ECAT7 image values write and read<br>
   ECAT6 image volume write and read<br>
   Analyze image volume write and read<br>
   NIfTI-1 image volume write and read<br>

   This functions writes and reads images of huge size.
   Function in file test_libtpcimgio.c writes only small files.
  
   @return 0 in case of success */
int test_img_io(
  /** Verbose level; enter 0 to print nothing. */ 
  int VERBOSE
) {
  int xdim, ydim, zdim, fdim;
  long long pxlNr;
  char *bfname="test_img_io";
  char fname[FILENAME_MAX];
  int ret;
  IMG img, img2;

  if(VERBOSE>0) printf("\n Testing IMG I/O functions\n");
  if(VERBOSE>3) IMG_TEST=VERBOSE-3;
  imgInit(&img); imgInit(&img2);

  /* Create test image for ECAT 7 image formats;
     note that ECAT matrix list limits plane number to < 256 and frame number to < 512 */
  xdim=1024; ydim=1024; zdim=50; fdim=25; // 1 310 720 000 pixels
  pxlNr=xdim*ydim*zdim*fdim;
  if(VERBOSE>1) printf("test image size is %dx%dx%dx%d = %lld\n", xdim, ydim, zdim, fdim, pxlNr);
  if(test_create_img(&img, xdim, ydim, zdim, fdim)) {
    fprintf(stderr, "Error: cannot create test data.\n");
    return(1);
  }
  img.type=IMG_TYPE_IMAGE;


  /* ECAT7 image write and read */
  sprintf(fname, "%s.v", bfname);
  img._dataType=ECAT7_SUNI2;

  if(VERBOSE>1) printf("\n  testing 3D ECAT 7 format\n");
  img._fileFormat=IMG_E7;
  ret=imgWrite(fname, &img); if(ret) {imgEmpty(&img); return(11);}
  ret=imgRead(fname, &img2); if(ret) {imgEmpty(&img); imgEmpty(&img2); return(12);}
  ret=imgMatch(&img, &img2, 0.001);
  if(ret!=0) {
    fprintf(stderr, "Error: Ecat7 image is not the same after write/read (%d)\n", ret);
    if(ret<400 && VERBOSE>2) {
      printf("\n--------\nBefore:\n"); imgInfo(&img);
      printf("\n--------\nAfter:\n"); imgInfo(&img2);
    }
    imgEmpty(&img); imgEmpty(&img2); return(13);
  }
  remove(fname); imgEmpty(&img2);

  if(VERBOSE>1) printf("\n  testing 2D ECAT 7 format\n");
  img._fileFormat=IMG_E7_2D;
  ret=imgWrite(fname, &img); if(ret) {imgEmpty(&img); return(14);}
  ret=imgRead(fname, &img2); if(ret) {imgEmpty(&img); imgEmpty(&img2); return(15);}
  ret=imgMatch(&img, &img2, 0.001);
  if(ret!=0) {
    fprintf(stderr, "Error: Ecat7 image is not the same after write/read (%d)\n", ret);
    if(ret<400 && VERBOSE>2) {
      printf("\n--------\nBefore:\n"); imgInfo(&img);
      printf("\n--------\nAfter:\n"); imgInfo(&img2);
    }
    imgEmpty(&img); imgEmpty(&img2); return(16);
  }
  remove(fname); imgEmpty(&img2);

  imgEmpty(&img); 


  /* Create test image for ECAT 6.3 image formats; at the time, files were much smaller,
     and code does not support large files. */
  xdim=128; ydim=128; zdim=15; fdim=25;
  pxlNr=xdim*ydim*zdim*fdim;
  if(VERBOSE>1) printf("test image size is %dx%dx%dx%d = %lld\n", xdim, ydim, zdim, fdim, pxlNr);
  if(test_create_img(&img, xdim, ydim, zdim, fdim)) {
    fprintf(stderr, "Error: cannot create test data.\n");
    return(1);
  }
  img.type=IMG_TYPE_IMAGE;

  if(VERBOSE>1) printf("  testing ECAT 6.3 format\n");
  sprintf(fname, "%s.img", bfname);
  img._fileFormat=IMG_E63;
  img._dataType=ECAT7_SUNI2;
  img.scanner=SCANNER_ECAT931;
  ret=imgWrite(fname, &img); if(ret) return(21);
  ret=imgRead(fname, &img2); if(ret) return(22);
  /* ECAT6 image values write and read testing */
  ret=imgMatch(&img, &img2, 0.001);
  if(ret!=0) {
    fprintf(stderr, "Error: Ecat6 image is not the same after write/read (%d)\n", ret);
    if(ret<400 && VERBOSE>2) {
      printf("\n--------\nBefore:\n"); imgInfo(&img);
      printf("\n--------\nAfter:\n"); imgInfo(&img2);
    }
    imgEmpty(&img); imgEmpty(&img2);
    return(23);
  }
  remove(fname); imgEmpty(&img2);

  imgEmpty(&img); 



  /* Create a big test image for NIfTI and Analyze image formats */
  xdim=512; ydim=512; zdim=256; fdim=20; // 5.2 Gb NIfTI file, 2.5 Gb Analyze file
  pxlNr=xdim*ydim*zdim*fdim;
  if(VERBOSE>1) printf("test image size is %dx%dx%dx%d = %lld\n", xdim, ydim, zdim, fdim, pxlNr);
  if(test_create_img(&img, xdim, ydim, zdim, fdim)) {
    fprintf(stderr, "Error: cannot create test data.\n");
    return(1);
  }
  img.type=IMG_TYPE_IMAGE;
  img._dataType=0; // Not stored in NIfTI and Analyze headers
  img.unit=0; // Not stored in NIfTI and Analyze headers
  img.scanner=SCANNER_UNKNOWN; // Not stored in NIfTI and Analyze headers


  /* NIfTI single file format write and read */
  niftiRemove(fname, 0, 0);
  if(VERBOSE>1) printf("  testing NIFTI single file format\n");
  strcpy(fname, bfname);
  img._fileFormat=IMG_NIFTI_1S;
  ret=imgWrite(fname, &img); if(ret) return(41);
  int file_format;
  ret=imgFormatDetermine(fname, NULL, NULL, NULL, NULL,
         &file_format, NULL, NULL, NULL, 0);
  if(ret!=0) return(42);
  if(file_format!=IMG_NIFTI_1S) return(43);
  ret=imgRead(fname, &img2); if(ret) return(44);
  niftiRemove(fname, 0, 0);
  ret=imgMatch(&img, &img2, 0.001);
  if(ret!=0) {
    fprintf(stderr, "Error: NIfTI 1S image is not the same after write/read (%d)\n", ret);
    if(ret<400 && VERBOSE>2) {
      printf("\n--------\nBefore:\n"); imgInfo(&img);
      printf("\n--------\nAfter:\n"); imgInfo(&img2);
    }
    imgEmpty(&img); imgEmpty(&img2);
    return(23);
  }
  remove(fname); imgEmpty(&img2);


  /* NIfTI dual file format write and read */
  if(VERBOSE>1) printf("  testing NIFTI dual file format\n");
  strcpy(fname, bfname);
  img._fileFormat=IMG_NIFTI_1D;
  ret=imgWrite(fname, &img); if(ret) return(46);
  ret=imgFormatDetermine(fname, NULL, NULL, NULL, NULL,
         &file_format, NULL, NULL, NULL, 0);
  if(ret!=0) return(46);
  if(file_format!=IMG_NIFTI_1D) return(47);
  ret=imgRead(fname, &img2); if(ret) return(48);
  niftiRemove(fname, 0, 0);
  ret=imgMatch(&img, &img2, 0.001);
  if(ret!=0) {
    fprintf(stderr, "Error: NIfTI 1D is not the same after w/r (%d)\n", ret);
    if(ret<400 && VERBOSE>2) {
      printf("\n--------\nBefore:\n"); imgInfo(&img);
      printf("\n--------\nAfter:\n"); imgInfo(&img2);
    }
    imgEmpty(&img); imgEmpty(&img2);
    return(49);
  }
  imgEmpty(&img2);


  /* Analyze image volume write and read (Big endian) */
  if(VERBOSE>1) printf("  testing Analyze big endian format\n");
  strcpy(fname, bfname);
  img._fileFormat=IMG_ANA; /*ANALYZE_TEST=IMG_TEST=4;*/
  img._dataType=0;
  img.unit=0;
  img.scanner=SCANNER_UNKNOWN;
  ret=imgWrite(fname, &img); if(ret) return(31);
  ret=imgRead(fname, &img2); if(ret) return(32);
  ret=imgMatch(&img, &img2, 0.001);
  niftiRemove(fname, IMG_NIFTI_1D, 0);
  if(ret!=0) {
    fprintf(stderr, "Error: Analyze image (big endian) is not the same after w/r (%d)\n", ret);
    if(ret<400 && VERBOSE>2) {
      printf("\n--------\nBefore:\n"); imgInfo(&img);
      printf("\n--------\nAfter:\n"); imgInfo(&img2);
    }
    imgEmpty(&img); imgEmpty(&img2);
    return(33);
  }
  imgEmpty(&img2);

  /* Analyze image volume write and read (Little endian) */
  if(VERBOSE>1) printf("  testing Analyze little endian format\n");
  strcpy(fname, bfname);
  img._fileFormat=IMG_ANA_L;
  ret=imgWrite(fname, &img); if(ret) return(34);
  ret=imgRead(fname, &img2); if(ret) return(35);
  ret=imgMatch(&img, &img2, 0.001);
  niftiRemove(fname, IMG_NIFTI_1D, 0);
  if(ret!=0) {
    fprintf(stderr, "Error: Analyze image 2 is not the same after w/r (%d)\n", ret);
    if(ret<400 && VERBOSE>2) {
      printf("\n--------\nBefore:\n"); imgInfo(&img);
      printf("\n--------\nAfter:\n"); imgInfo(&img2);
    }
    imgEmpty(&img); imgEmpty(&img2);
    return(36);
  }
  imgEmpty(&img2);



  imgEmpty(&img); 

  if(VERBOSE>0) printf("  ok\n");
  return(0);
}
/******************************************************************************/

/******************************************************************************/
