/**
  @file prmat.c
  @brief Data handling for projection matrix.
  @author Jarkko Johansson
  @date 2004-04-20
*/
/*****************************************************************************/
int PRMAT_TEST;
int PRMAT_VERBOSE;
/*****************************************************************************/
#include "libtpcrec.h"
/*****************************************************************************/

/*****************************************************************************/
/* Initialization and memory handling for prmat data      */
/*****************************************************************************/
/**
   Initializes PRMAT datatype for use.
   To be utilised before any use of PRMAT type variables.
   @param mat pointer to projection matrix to be initialized. 
*/
void prmatInit(PRMAT *mat)
{
  if(PRMAT_VERBOSE) printf("PRMAT:prmatInit() \n"); 
  
  /*init the information of projection matrix*/
  //buffer mat to contain zero
  memset(mat, 0, sizeof(PRMAT)); 
  mat->type=PRMAT_TYPE_NA;
  mat->viewNr=0;
  mat->binNr=0;
  mat->mode=PRMAT_DMODE_01;
  mat->prmatfov= (int*)NULL;
  mat->max=mat->min=0.0;
  mat->scaling_factor=1.0;
  mat->factor_sum=0.0;
  mat->factor_sqr_sum=(float*)NULL;

  /*init look-up table*/
  mat->nrp=0;
  mat->lines=mat->_linesdata=(unsigned int**)NULL; //data array

  /*init projection data*/
  mat->dimr=0;
  mat->dime=(unsigned int*)NULL;
  mat->fact=mat->_factdata=(unsigned short int***)NULL; //data array
  
  mat->status=PRMAT_STATUS_INITIALIZED;
}
/*****************************************************************************/
/**
   Frees the memory allocated for mat.
   All data is cleared.
   @param mat pointer to projection matrix to be emptied.
*/
void prmatEmpty(PRMAT *mat)
{
  unsigned int ir, ip;
  if(PRMAT_VERBOSE) printf("PRMAT:prmatEmpty() \n");
  
  //if mat is allready empty
  if(mat->status==PRMAT_STATUS_INITIALIZED) return; 
  
  //if look-up table is occupied
  if(mat->status==PRMAT_STATUS_LU_OCCUPIED) { 
    /*free the occupied memory*/
    for(ir=0; ir<mat->nrp; ir++) { //every row
      free((unsigned int*)mat->_linesdata[ir]);
    }
    if(mat->nrp > 0) free(mat->_linesdata);
  }

  //if projection matrix is occupied
  if(mat->status==PRMAT_STATUS_PR_OCCUPIED || 
     mat->status==PRMAT_STATUS_BS_OCCUPIED )
  { 
    free((float*)mat->factor_sqr_sum);
    for(ir=0; ir<mat->dimr; ir++){ //every row
      for(ip=0; ip<mat->dime[ir]; ip++){ //every factor
	free((unsigned short int*)mat->_factdata[ir][ip]);
      }
    }

    free((int*)mat->dime);
  
    if(mat->dimr>0) free(mat->_factdata); 
    mat->dimr=0; 
  }
  return;
}
/*****************************************************************************/
/**
   Allocates memory for PRMAT data. Normally used only in SET-functions.   
   Allocates memory for look-up table if set = 0 and for projection matrix 
   otherwise.
   @pre *mat is initialized && rows is positive && *coords list contains 
             as many positive integers as there are rows.
   @post memory is allocated for PRMAT structure mat.
   @param mat pointer to projection matrix for which the allocation is done.
   @param set tells for which part of the structure allocation is to be done.
   @param rows number of rows in a matrix.
   @param coords list of number of entries in each row.
   @return 0 if ok.
 */
int prmatAllocate(PRMAT *mat, int set, unsigned int rows, unsigned int *coords)
{
  unsigned int *iptr, ir, ip;
  if(PRMAT_VERBOSE) printf("PRMAT:prmatAllocate(*mat,%d,*coords) \n",rows);
  
  /*check the arguments*/
  if(mat->status==PRMAT_STATUS_UNINITIALIZED) return(1);
  if(rows<1) return(2);
  
  //If set != 0 allocate for projection matrix.
  if(set){
    /*allocate memory*/
    mat->factor_sqr_sum=calloc(rows,sizeof(float));
    mat->dimr=rows; //for dimr
    mat->dime=calloc(rows,sizeof(int)); 
    mat->_factdata=
     (unsigned short int***)calloc(rows,sizeof(unsigned short int**)); 
    //if not enough memory
    if(mat->_factdata==NULL || mat->dime==NULL) return(4);
    //iterate through the rows
    for(ir=0,iptr=coords; ir<rows; ir++){
      //copy the coords vector in the same loop
      mat->dime[ir]=*iptr; 
      //*iptr is the number of (non-zero) factors in this row
      mat->_factdata[ir]=
       (unsigned short int**)calloc(*iptr,sizeof(unsigned short int*)); 
      //if not enough memory
      if(mat->_factdata[ir]==NULL) return(4); 
      //iterate through the factors in this row
      for(ip=0; ip<(*iptr); ip++){ 
	//NOTE that the leaves are coordinates and factor in this order
	mat->_factdata[ir][ip]=
          (unsigned short int*)calloc(3,sizeof(unsigned short int)); 
	if(mat->_factdata[ir][ip]==NULL) return(4);
      }
      iptr++;
    }
    /*set data pointers; these pointers are used to avoid data loss*/
    mat->fact=mat->_factdata;    
  } else {
    //If set was 0 allocate for look-up table.
    //allocate memory for look up table
    mat->nrp=rows; 
    mat->_linesdata=(unsigned int**)calloc(rows,sizeof(unsigned int*)); 
    //if not enough memory
    if(mat->_linesdata==NULL) return(4);
    //iterate through the rows
    for(ir=0,iptr=coords; ir<rows; ir++){
      //iptr is the number of lors belonging to this pixel
      mat->_linesdata[ir]=(unsigned int*)calloc(*iptr + 2,sizeof(unsigned int)); 
      //if not enough memory
      if(mat->_linesdata[ir]==NULL) return(4); 
      iptr++;
    }
    //set data pointers; these pointers are used to avoid data loss
    mat->lines=mat->_linesdata;    
  }
  return(0);
}
/*****************************************************************************/
/* GET procedures for PRMAT datatype. USE ONLY THESE FOR GETTING. */
/*****************************************************************************/
/* GET FUNCTIONS FOR THE PARAMETERS */
/*****************************************************************************/
/**
   Returns the number of views (angles) in the geometrics.
   @pre mat is initialized. 
   @param mat pointer to a projection matrix.
   @return the number of views in the geometrics.
*/
unsigned int prmatGetNV(PRMAT *mat)
{
  return mat->viewNr;
}
/*****************************************************************************/
/**
   Returns the number of bins (distances) in the geometrics.
   @pre mat is initialized. 
   @param mat pointer to a projection matrix.
   @return the number of bins in the geometrics.
*/
unsigned int prmatGetNB(PRMAT *mat)
{
  return mat->binNr;
}
/*****************************************************************************/
/**
   Returns the image dimension in the geometrics.
   @pre mat is initialized. 
   @param mat pointer to a projection matrix.
   @return the image dimension in the geometrics.
*/
unsigned int prmatGetID(PRMAT *mat)
{
  return mat->imgDim;
}
/*****************************************************************************/
/* GET FUNCTIONS FOR ACCESSING LOOK-UP TABLE*/
/*****************************************************************************/
/**
   Returns number of rows (pixels) in the look-up table.
   @pre mat is initialized. 
   @param mat pointer to a projection matrix.
   @return number of pixels in look-up table, some negative value if ERROR.
*/
unsigned int prmatGetPIX(PRMAT *mat)
{
  return mat->nrp;
}
/*****************************************************************************/
/**
   Returns coordinates of a pixel in place 'row' in the look-up table.
   @pre 0<=row<=prmatGetPIX(*mat) && mat is initialized.
   @param mat pointer to a projection matrix.
   @param row index of the pixel whose coordinates are to be returned.
   @return coordinates of a pixel in the look-up table, some negative value 
    if ERROR.
*/
unsigned int prmatGetPixCoord(PRMAT *mat,int row)
{
  return mat->lines[row][0];
}
/*****************************************************************************/
/**
   Returns the number of lines of response (rays) intersecting the pixel in 
   place 'row'.
   @pre 0<=row<=prmatGetPIX(*mat) && mat is initialized.
   @param mat pointer to a projection matrix.
   @param row index of the pixel for which the number of lors is returned.
   @return number of lines of response intersecting the pixel, some negative 
    value if ERROR.
*/
unsigned int prmatGetRays(PRMAT *mat,int row)
{
  return mat->lines[row][1];
}
/*****************************************************************************/
/**
   Returns coordinates of a line of response in place (row,ind) in the 
   look-up table.
   @pre 0<=row<=prmatGetPIX(*mat) && 0<=ind<=prmatGetRays(*mat,row) &&
    mat is initialized. 
   @param mat pointer to a projection matrix.
   @param row the row index.
   @param ind the (non-zero) lor index.
   @return int coordinates (view*binNr + bin) of the referred line of response,
    some negative value if ERROR.
 */
unsigned int prmatGetBinView(PRMAT *mat, int row, int ind)
{
  return mat->lines[row][ind+2];
}
/*****************************************************************************/
/* GET FUNCTIONS FOR ACCESSING PROJECTION MATRIX BY A LINE OF RESPONSE.*/
/*****************************************************************************/
/**
   Returns number of rows (lines of response) in the given projection matrix.
   @pre mat is initialized. 
   @param mat pointer to the projection matrix.
   @return number of rows in the given projection matrix, 
    some negative value if ERROR.
*/
unsigned int prmatGetRows(PRMAT *mat)
{
  //NOTE no checking on parameters to accelerate iteration
  return(mat->dimr);
}
/*****************************************************************************/
/**
   Returns number of pixels intersected by the given line of response.
   @pre 0<=row<=prmatGetLORS(*mat) && mat is initialized.
   @param mat pointer to the projection matrix.
   @param row index of the line of response for which the number of pixels 
    is returned.
   @return number of pixels intersected by given line of response, 
    some negative value if ERROR.
*/
unsigned int prmatGetPixels(PRMAT *mat, int row)
{
  return(mat->dime[row]);
}
/*****************************************************************************/
/**
   Returns coordinate of a pixel in place (row,pix) in the given projection 
   matrix.
   @pre 0<=row<=prmatGetLORS(*mat) && 0<=pix<=prmatGetPixels(row,*mat)
    && mat is initialized. 
   @param mat pointer to a projection matrix.
   @param row the row index.
   @param pix the (non-zero) pixel index.
   @return coordinate of the referred pixel, some negative value if ERROR.
*/
unsigned int prmatGetCoord(PRMAT *mat, int row, int pix)
{
  return (int)(mat->fact[row][pix][1])*mat->imgDim + (int)(mat->fact[row][pix][0]);
}
/*****************************************************************************/
/**
   Returns the x-coordinate of a pixel in place (row,pix) in the given 
   projection matrix.
   @pre 0<=row<=prmatGetLORS(*mat) && 0<=pix<=prmatGetPixels(row,*mat)
    && mat is initialized. 
   @param mat pointer to a projection matrix.
   @param row the row index.
   @param pix the (non-zero) pixel index.
   @return coordinate of the referred pixel, some negative value if ERROR.
*/
unsigned int prmatGetXCoord(PRMAT *mat, int row, int pix)
{
  return (int)(mat->fact[row][pix][0]);
}
/*****************************************************************************/
/**
   Returns the y-coordinate of a pixel in place (row,pix) in the given 
   projection matrix.
   @pre 0<=row<=prmatGetLORS(*mat) && 0<=pix<=prmatGetPixels(row,*mat)
    && mat is initialized. 
   @param mat pointer to a projection matrix.
   @param row the row index.
   @param pix the (non-zero) pixel index.
   @return coordinate of the referred pixel, some negative value if ERROR.
*/
unsigned int prmatGetYCoord(PRMAT *mat, int row, int pix)
{
  return (int)(mat->fact[row][pix][1]);
}
/*****************************************************************************/
/**
   Returns factor (weighting coefficient) for a pixel in place (row,pix) in 
   given projection matrix.
   @pre 0<=row<=prmatGetLORS(*mat) && 0<=pix<=prmatGetPixels(row,*mat)
    && mat is initialized.
   @param mat pointer to a projection matrix.
   @param row the row index.
   @param pix the (non-zero) pixel index.
   @return factor for the referred pixel, some negative value if ERROR.
*/
float prmatGetFactor(PRMAT *mat, int row, int pix)
{
  return((float)(mat->scaling_factor)*(mat->fact[row][pix][2]));
}
/*****************************************************************************/
/**
   Returns major semiaxe of the FOV in given projection matrix.
   @pre mat is initialized.
   @param mat pointer to a projection matrix.
   @return major semiaxe of the FOV in given projection matrix
*/
float prmatGetMajor(PRMAT *mat)
{
  return(mat->prmatfov[0]);
}
/*****************************************************************************/
/**
   Returns minor semiaxe of the FOV in given projection matrix.
   @pre mat is initialized.
   @param mat pointer to a projection matrix.
   @return minor semiaxe of the FOV in given projection matrix
*/
float prmatGetMinor(PRMAT *mat)
{
  return(mat->prmatfov[1]);
}
/*****************************************************************************/
/**
   Returns minimal value in given projection matrix.
   @pre mat is initialized.
   @param mat pointer to a projection matrix.
   @return minimal value in given projection matrix.
*/
float prmatGetMin(PRMAT *mat)
{
  return(mat->min);
}
/*****************************************************************************/
/**
   Returns greatest value in given projection matrix.
   @pre mat is initialized.
   @param mat pointer to a projection matrix.
   @return maximal value in given projection matrix.
*/
float prmatGetMax(PRMAT *mat) {
  return(mat->max);
}
/*****************************************************************************/
/**
   Returns sum of all factors in given projection matrix.
   For initializing image matrix.
   @pre mat is initialized.
   @param mat pointer to a projection matrix.
   @return sum of all factors in given projection matrix.
*/
float prmatGetFactorSum(PRMAT *mat)
{
  return(mat->factor_sum);
}
/*****************************************************************************/
/**
   Returns square sum of all factors in given projection matrix in given row.
   @pre mat is initialized && 0<=row<=prmatGetLORS(*mat).
   @param mat pointer to a projection matrix.
   @param row row index for accessing factors.
   @return square sum of all factors in given projection matrix in given row.
*/
float prmatGetFactorSqrSum(PRMAT *mat, int row)
{
  return(mat->factor_sqr_sum[row]);
}
/*****************************************************************************/
/* ARCHIVING METHODS */
/*****************************************************************************/
/**
   Adds the given projection matrix to the file called prTDXXYY.prm. 
   Where T is the type of the scanner 
   (E=ECAT931 and G=GE Advance), D is the type of discretisation
   (0='0/1', 1='length of intersection' and 2='exact area'),
   XX is the major semiaxe of the FOV, and YY the minor semiaxe.
   If file already exists it is NOT created or rewritten.
   @pre mat is initialized (contains a projection matrix). 
   @post mat is saved in the file  prTDXXYY.prm. 
   @param mat pointer to the projection matrix to be saved.
   @return 0 if OK.
*/
int prmatSaveMatrix(PRMAT *mat)
{
  char matrixfile[FILENAME_MAX], semi_x_c[2], semi_y_c[2], mod[2];
  int semi_x=mat->prmatfov[0], semi_y=mat->prmatfov[1], mode=mat->mode, itmp;
  unsigned short int x_coordinate, y_coordinate, value, p_coord, lors;
  unsigned int lor, row, fac, ind;
  float ftmp;
  FILE *fp;
  
  //Compose the file name to write
  sprintf(semi_x_c,"%i",semi_x);
  sprintf(semi_y_c,"%i",semi_y);
  //decide the type
  if(mat->type==PRMAT_TYPE_ECAT931) strcpy(matrixfile,"prE");
  else 
    if(mat->type==PRMAT_TYPE_GE) strcpy(matrixfile,"prG");
    else strcpy(matrixfile,"prN");
  //decide the mode 
  sprintf(mod,"%i",mode);
  strcat(matrixfile,mod);
  strcat(matrixfile,semi_x_c);
  strcat(matrixfile,semi_y_c);
  strcat(matrixfile,".prm");
  
  if(access(matrixfile, 0) != -1) {
    if(PRMAT_VERBOSE) printf("PRMAT:File %s already exists.\n", matrixfile);
    return(0);
  }
  
  fp=fopen(matrixfile,"wb+");
  if(fp==NULL) return(1);

  // Save status.
  fwrite(&(mat->status),sizeof(int),1,fp);  

  // Save type of the scanner.
  fwrite(&(mat->type),sizeof(int),1,fp);  

  // Save scanner geometrics.
  fwrite(&(mat->viewNr),sizeof(int),1,fp);  
  fwrite(&(mat->binNr),sizeof(int),1,fp);  
  fwrite(&(mat->imgDim),sizeof(int),1,fp);  

  // Save mode.
  fwrite(&(mat->mode),sizeof(int),1,fp);  

  // Save FOV.
  fwrite(&(mat->prmatfov[0]),sizeof(int),1,fp);
  fwrite(&(mat->prmatfov[1]),sizeof(int),1,fp); 

  // Save min, max, sum and scaling factor.
  fwrite(&(mat->min),sizeof(float),1,fp);
  fwrite(&(mat->max),sizeof(float),1,fp);
  fwrite(&(mat->factor_sum),sizeof(float),1,fp);
  fwrite(&(mat->scaling_factor),sizeof(float),1,fp);  

  // If projection matrix is set.
  if(mat->status >= PRMAT_STATUS_BS_OCCUPIED){

    fwrite(&(mat->dimr),sizeof(int),1,fp);
    //square sums
    for(row=0;row<prmatGetRows(mat);row++){
      ftmp=prmatGetFactorSqrSum(mat,row);
      fwrite(&ftmp,sizeof(float),1,fp);
    }
    //dime
    for(row=0;row<prmatGetRows(mat);row++){
      itmp=prmatGetPixels(mat,row);
      fwrite(&itmp,sizeof(int),1,fp);
    }
    //the actual data _factdata
    for(row=0;row<prmatGetRows(mat);row++){
      //coordinates and values in every row
      for(fac=0;fac<prmatGetPixels(mat,row);fac++){
	x_coordinate=prmatGetXCoord(mat,row,fac);
	fwrite(&x_coordinate,sizeof(unsigned short int),1,fp);
	y_coordinate=prmatGetYCoord(mat,row,fac);
	fwrite(&y_coordinate,sizeof(unsigned short int),1,fp);
	//NOTE that we wan't to save the values in unsigned short ints
	value=mat->fact[row][fac][2];
	fwrite(&value,sizeof(unsigned short int),1,fp);
      }
    }
  }// END OF SAVING PROJECTION MATRIX.

  // If look-up table is set.
  if(mat->status == PRMAT_STATUS_LU_OCCUPIED){
    //save first the statical parts of the structure
    fwrite(&(mat->nrp),sizeof(int),1,fp);
    //save lines: first coordinates of pixels inside the fov,
    for(row=0;row<prmatGetPIX(mat);row++){
      p_coord=prmatGetPixCoord(mat,row);
      fwrite(&p_coord,sizeof(unsigned short int),1,fp);
    }
    // then number of lors hitting a pixel.
    for(row=0; row<prmatGetPIX(mat); row++){
      lors=prmatGetRays(mat,row);
      fwrite(&lors,sizeof(unsigned short int),1,fp);
    }

    //save lors
    for(row=0;row<prmatGetPIX(mat);row++){
      for(ind=0; ind<prmatGetRays(mat,row); ind++){
	lor = prmatGetBinView(mat,row,ind);
	fwrite(&lor,sizeof(unsigned int),1,fp);
      }
    }
  }// END OF SAVING LOOK-UP TABLE.

  fclose(fp);
  return(0);
}
/*****************************************************************************/
/**
   Reads one projection matrix from the given file to the given PRMAT 
   structure mat.
   @pre mat is initialized.
   @post a projection matrix is read from the file fname. 
   @param fname file name to read.
   @param mat pointer to the projection matrix datatype.
   @return 0 if OK.
*/
int prmatReadMatrix(char *fname, PRMAT *mat)
{
  unsigned int fov[2], *dimentry=NULL, *nrlor, *iptr, dimtmp, count=0;
  unsigned int row, lor=0, ind, fac;
  unsigned short int  *nrlor_c, *nrlor_l, *usiptr, *usiptr_c;
  unsigned short int x_coordinate=0, y_coordinate=0, value=0;
  float *sqrsum=NULL, *fptr, sqrtmp;
  FILE *fp;

  if(PRMAT_VERBOSE) printf("\n PRMAT:prmatReadMatrix(%s,mat) \n",fname);
  if(mat->status==PRMAT_STATUS_UNINITIALIZED) return(1);
  prmatEmpty(mat);

  fp=fopen(fname,"rb");
  if(fp==NULL) return(1);

  //read first the statical parts of the structure
  if(fread(&(mat->status),sizeof(int),1,fp)==0) {fclose(fp); return(2);}
  if(PRMAT_VERBOSE) printf("PRMAT: status: %i \n",mat->status);
  if(fread(&(mat->type),sizeof(int),1,fp)==0) {fclose(fp); return(2);}
  if(PRMAT_VERBOSE) printf("PRMAT: scanner type: %i \n",mat->type);
  if(fread(&(mat->viewNr),sizeof(int),1,fp)==0) {fclose(fp); return(2);}
  if(fread(&(mat->binNr),sizeof(int),1,fp)==0) {fclose(fp); return(2);}
  if(fread(&(mat->imgDim),sizeof(int),1,fp)==0) {fclose(fp); return(2);}
  if(PRMAT_VERBOSE) 
    printf("PRMAT: scanner geometrics: (%i,%i,%i) \n",
           mat->viewNr,mat->binNr,mat->imgDim);
  if(fread(&(mat->mode),sizeof(int),1,fp)==0) {fclose(fp); return(2);}
  if(PRMAT_VERBOSE) printf("PRMAT: discretisation mode: %i \n",mat->mode);
  if(fread(fov,sizeof(int),1,fp)==0) {fclose(fp); return(2);}
  if(fread((fov+1),sizeof(int),1,fp)==0) {fclose(fp); return(2);}
  mat->prmatfov=calloc(2,sizeof(int)); 
  mat->prmatfov[0]=fov[0];
  mat->prmatfov[1]=fov[1];
  if(PRMAT_VERBOSE) 
    printf("PRMAT: FOV: (%i,%i) \n", mat->prmatfov[0],mat->prmatfov[1]);
  // Read min, max, sum and scaling factor.
  if(fread(&(mat->min),sizeof(float),1,fp)==0) {fclose(fp); return(2);}
  if(PRMAT_VERBOSE) printf("PRMAT: min: %f \n",mat->min);
  if(fread(&(mat->max),sizeof(float),1,fp)==0) {fclose(fp); return(2);}
  if(PRMAT_VERBOSE) printf("PRMAT: max: %f \n",mat->max);
  if(fread(&(mat->factor_sum),sizeof(float),1,fp)==0) {fclose(fp); return(2);}
  if(PRMAT_VERBOSE) printf("PRMAT: sum of all factors: %f \n",mat->factor_sum);
  if(fread(&(mat->scaling_factor),sizeof(float),1,fp)==0) {fclose(fp); return(2);}
  if(PRMAT_VERBOSE) printf("PRMAT: scaling factor: %f \n",mat->scaling_factor);  
 
  // If projection matrix has been occupied.
  if(mat->status >= PRMAT_STATUS_BS_OCCUPIED){
    if(PRMAT_VERBOSE) printf("PRMAT: reading projection matrix \n");  
    if(fread(&(mat->dimr),sizeof(int),1,fp)==0) {fclose(fp); return(2);}
    if(PRMAT_VERBOSE) 
      printf("PRMAT: number of rows in the projection matrix: %i \n",mat->dimr);  
    //read square sums
    sqrsum=calloc(mat->dimr,sizeof(float));  
    for(row=0, fptr=sqrsum;row<(mat->dimr); row++, fptr++) {
      if(fread(&sqrtmp,sizeof(float),1,fp)==0) {fclose(fp); return(2);}
      *fptr=sqrtmp;
    }
    //read dime
    dimentry=(unsigned int*)calloc(mat->dimr,sizeof(int));  
    count = 0;
    for(row=0,iptr=dimentry;row<(mat->dimr);row++,iptr++){
      if(fread(&dimtmp,sizeof(int),1,fp)==0) {fclose(fp); return(2);}
      *iptr=dimtmp;
      count += dimtmp;
    }

    //ready to allocate; NOTE that dime is also set in function prmatAllocate()    
    prmatAllocate(mat,1,mat->dimr,dimentry);
    if(PRMAT_VERBOSE)
      printf("PRMAT:prmatAllocate(mat) done in prmatReadMatrix(%s,mat)\n",fname);

    //now we can set square sums
    for(row=0, fptr=sqrsum;row<(mat->dimr);row++, fptr++){
      mat->factor_sqr_sum[row]=*fptr;
    }
    //read the actual data _factdata
    for(row=0;row<(mat->dimr);row++){
      //read coordinates and values in every row
      for(fac=0;fac<prmatGetPixels(mat,row);fac++){
	if(fread(&x_coordinate,sizeof(unsigned short int),1,fp)==0) {
          fclose(fp); return(2);}
	mat->fact[row][fac][0]=x_coordinate;
	if(fread(&y_coordinate,sizeof(unsigned short int),1,fp)==0) {
          fclose(fp); return(2);}
	mat->fact[row][fac][1]=y_coordinate;
	if(fread(&value,sizeof(unsigned short int),1,fp)==0) {
          fclose(fp); return(2);}
	mat->fact[row][fac][2]=value;
      }
    }
  }// END OF READING PROJECTIONS

  // If look-up table has been occupied.
  if(mat->status == PRMAT_STATUS_LU_OCCUPIED){
    if(PRMAT_VERBOSE) printf("PRMAT: reading look-up table \n");  
    if(fread(&(mat->nrp),sizeof(int),1,fp)==0) {fclose(fp); return(2);}
    if(PRMAT_VERBOSE)
      printf("PRMAT: number of pixels inside the FOV: %i \n",mat->nrp);
    //read nrlor
    nrlor_c=(unsigned short int*)calloc(mat->nrp,sizeof(unsigned short int));  
    nrlor_l=(unsigned short int*)calloc(mat->nrp,sizeof(unsigned short int));  

    //first coordinates of pixels.
    usiptr = nrlor_c;
    for(row=0; row<mat->nrp; row++, usiptr++){
      // Read next coordinate from file.
      if(fread(usiptr,sizeof(unsigned short int),1,fp)==0) {fclose(fp); return(2);}
    }
    //then number of lors hitting a pixel
    usiptr = nrlor_l;
    for(row=0; row<mat->nrp; row++, usiptr++){
      // Read next lornr from file.
      if(fread(usiptr,sizeof(unsigned short int),1,fp)==0) {fclose(fp); return(2);}
    }

    if(PRMAT_VERBOSE){
      // Get number of lors in lut.
      usiptr = nrlor_l;
      count = 0;
      for(row=0; row<mat->nrp; row++, usiptr++){
	count += *usiptr;
      }
      printf("PRMAT: nr of lors in lut:  %i \n",count);
      //printf("  nr of lors in ave:  %i \n",count/mat->nrp);
    }

    // Copy list usi format to integer list.
    nrlor = (unsigned int*)calloc(mat->nrp,sizeof(int));
    usiptr = nrlor_l;
    iptr = nrlor;
    for(row=0; row<mat->nrp; row++, iptr++){
      *iptr = (int)*usiptr++;
    }    

    // Now we can allocate memory for look-up table.
    prmatAllocate(mat, 0, mat->nrp, nrlor);
    if(PRMAT_VERBOSE)
      printf("PRMAT:prmatAllocate(mat) done in prmatReadMatrix(%s,mat)\n",fname);
    // And fill mat->lines.
    usiptr_c = nrlor_c;
    usiptr = nrlor_l;
    for(row=0;row<mat->nrp;row++ , usiptr++, usiptr_c++){
      mat->lines[row][0] = *usiptr_c;
      mat->lines[row][1] = *usiptr;
    }

    // Read the data _linesdata.
    for(row=0;row<prmatGetPIX(mat);row++){
      for(ind=0; ind<prmatGetRays(mat,row); ind++){
	if(fread(&lor,sizeof(unsigned int),1,fp)==0) {fclose(fp); return(2);}
	mat->lines[row][ind+2] = lor;
      }
    }

    free((unsigned short int*)nrlor_l);
    free((unsigned short int*)nrlor_c);
  }// END OF READING LOOK-UP TABLE

  if(PRMAT_VERBOSE) printf("PRMAT: prmatReadMatrix() done. \n");
  free((float*)sqrsum); free((int*)dimentry);

  fclose(fp);

  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
