/** @file pxl2mask.c
 *  @brief Make mask image from specified pixel(s).
 *  @copyright (c) Turku PET Centre
 *  @author Vesa Oikonen
 */
/// @cond
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <string.h>
#include <unistd.h>
#include <time.h>
/*****************************************************************************/
#include "libtpcmisc.h"
#include "libtpcimgio.h"
#include "libtpccurveio.h"
#include "libtpcimgp.h"
#include "libtpcmodext.h"
/*****************************************************************************/

/*****************************************************************************/
static char *info[] = {
  "Make mask image from specified pixel(s).",
  "Pixels can be specified as volume range definition files, or as file",
  "containing list of pixel coordinates.",
  "A template image (not modified) is needed to get mask dimensions.",
  " ",
  "Usage: @P [Options] templatefile maskfile pixel(s)",
  " ",
  "Options:",
  " -stdoptions", // List standard options like --help, -v, etc
  " ",
  "Volume range definition (vrd) file is an ASCII text file, which contains",
  "pixel coordinates (x y z; 1..dimension) of the two opposite corners of",
  "the extracted image volume, for example:",
  "     corner1 := 63 57 26",
  "     corner2 := 84 71 44",
  "One or more pixel coordinates (x y z; 1..dimension) can be listed in file,",
  "for example:",
  "     24,52,13",
  "     25,52,14",
  " ",
  "See also: pxl2tac, mask2pxl, imgmask, imgmaxp, maskdila",
  " ",
  "Keywords: image, pixel, mask",
  0};
/*****************************************************************************/

/*****************************************************************************/
/* Turn on the globbing of the command line, since it is disabled by default in
   mingw-w64 (_dowildcard=0); in MinGW32 define _CRT_glob instead, if necessary;
   In Unix&Linux wildcard command line processing is enabled by default. */
/*
#undef _CRT_glob
#define _CRT_glob -1
*/
int _dowildcard = -1;
/*****************************************************************************/

/*****************************************************************************/
/**
 *  Main
 */
int main(int argc, char **argv)
{
  int        ai, help=0, version=0, verbose=1;
  int        ret, fileNr=0, firstfile=0;
  char       maskfile[FILENAME_MAX], tempfile[FILENAME_MAX];
  char      *cptr, tmp[128], pxlfile[FILENAME_MAX];
  IMG        img;


  /*
   *  Get arguments
   */
  if(argc==1) {tpcPrintUsage(argv[0], info, stderr); return(1);}
  maskfile[0]=tempfile[0]=pxlfile[0]=(char)0;
  /* Options */
  for(ai=1; ai<argc; ai++) if(*argv[ai]=='-') { /* options */
    cptr=argv[ai]+1; if(*cptr=='-') cptr++; if(cptr==NULL) continue;
    if(tpcProcessStdOptions(argv[ai], &help, &version, &verbose)==0) continue;
    /* We should not be here */
    fprintf(stderr, "Error: invalid option '%s'.\n", argv[ai]);
    return(1);
  } else break;
  
  /* Print help or version? */
  if(help==2) {tpcHtmlUsage(argv[0], info, ""); return(0);}
  if(help) {tpcPrintUsage(argv[0], info, stdout); return(0);}
  if(version) {tpcPrintBuild(argv[0], stdout); return(0);}
  
  /* Process other arguments, starting from the first non-option */
  if(ai<argc) {strlcpy(tempfile, argv[ai], FILENAME_MAX); ai++;}
  if(ai<argc) {strlcpy(maskfile, argv[ai], FILENAME_MAX); ai++;}
  for(; ai<argc; ai++) { // pixel def file(s)
    if(firstfile==0) firstfile=ai;
    fileNr++;
  }

  /* Is something missing or wrong? */
  if(!maskfile[0] || fileNr<1) {
    fprintf(stderr, "Error: missing command-line argument; use option --help\n");
    return(1);
  }

  /* In verbose mode print arguments and options */
  if(verbose>1) {
    printf("tempfile := %s\n", tempfile);
    printf("maskfile := %s\n", maskfile);
    printf("fileNr := %d\n", fileNr);
    fflush(stdout);
  }


  /*
   *  Read the contents of the template PET file to img data structure
   */
  {
    IMG timg;
    imgInit(&timg);
    if(verbose>0) printf("reading %s\n", tempfile);
    ret=imgRead(tempfile, &timg);
    if(ret) {
      fprintf(stderr, "Error: %s\n", timg.statmsg);
      if(verbose>1) printf("ret=%d\n", ret);
      imgEmpty(&timg); return(2);
    }
    if(verbose>0) {
      printf("pet_dimx := %d\n", timg.dimx);
      printf("pet_dimy := %d\n", timg.dimy);
      printf("pet_dimz := %d\n", timg.dimz);
      //printf("pet_dimt := %d\n", timg.dimt);
    }
    imgInit(&img);
    ret=imgAllocateWithHeader(&img, timg.dimz, timg.dimy, timg.dimx, 1, &timg);
    imgEmpty(&timg);
    if(ret!=0) {
      fprintf(stderr, "Error: cannot allocate mask image.\n");
      if(verbose>1) printf("ret=%d\n", ret);
      imgEmpty(&img); return(2);
    }
  }


  /*
   *  Read pixel definitions
   */
  if(verbose==1) printf("reading pixel positions\n");
  IMG_PIXELS pxl;
  pxlInit(&pxl);
  for(ai=firstfile; ai<argc; ai++) {
    strlcpy(pxlfile, argv[ai], FILENAME_MAX);
    if(verbose>1) printf("reading %s\n", pxlfile);
    /* First, try to read file as pixel list */
    if((ret=pxlRead(&pxl, pxlfile, tmp))==0) {
      continue;
    } else if(verbose>1) {
      printf("could not read as pixel list: %s\n", tmp);
      if(verbose>2) printf("ret := %d\n", ret);
    }
    /* Then try to read as Read Volume Range Definition File */
    VOL_RANGE vol_range={0,0,0,0,0,0};
    IMG_PIXEL p={0,0,0,0};
    if((ret=vrdRead(pxlfile, &vol_range, tmp))==0) {
      vrdReorder(&vol_range);
      if(verbose>1) {
        printf("vol_range.x := %d - %d\n", vol_range.x1, vol_range.x2);
        printf("vol_range.y := %d - %d\n", vol_range.y1, vol_range.y2);
        printf("vol_range.z := %d - %d\n", vol_range.z1, vol_range.z2);
      }
      /* Add range pixels to the list */
      long long n=vrdVxlNr(&vol_range);
      if(n<1) {
        fprintf(stderr, "Warning: no pixels defined in %s\n", pxlfile);
      } else {
        if(pxlAllocateMore(&pxl, n)!=0) {
          fprintf(stderr, "Error: out of memory.\n");
          imgEmpty(&img); pxlFree(&pxl); return(3);
        }
        for(p.z=vol_range.z1; p.z<=vol_range.z2; p.z++)
          for(p.x=vol_range.x1; p.x<=vol_range.x2; p.x++)
            for(p.y=vol_range.y1; p.y<=vol_range.y2; p.y++) 
              pxlAdd(&pxl, &p);
      }
      continue;
    } else if(verbose>1) {
      printf("could not read as vrd file: %s\n", tmp);
      if(verbose>2) printf("ret := %d\n", ret);
    }
    continue;
  } // next pixel list or vrd file
  if(pxl.pxlNr<1) {
    fprintf(stderr, "Error: no pixels.\n");
    imgEmpty(&img); pxlFree(&pxl); return(4);
  }
  if(verbose>6) {
    printf("list of pixels:\n");
    pxlWrite(&pxl, stdout, NULL);
  }
  /* Remove any duplicates */
  ret=pxlRmDuplicates(&pxl);
  if(ret>0 && verbose>0) {
    printf("%d pixel duplicates removed.\n", ret);
  }
  if(verbose>1) {
    printf("list of pixels:\n");
    pxlWrite(&pxl, stdout, NULL);
  }

  /* Check that pixels are inside image dimensions */
  ret=0;
  for(long long int i=0; i<pxl.pxlNr; i++) {
    if(pxl.p[i].z<1 || pxl.p[i].z>img.dimz) {
      fprintf(stderr, "Error: pixel outside image z dimension.\n");
      ret++;
    }
    if(pxl.p[i].x<1 || pxl.p[i].x>img.dimx) {
      fprintf(stderr, "Error: pixel outside image x dimension.\n");
      ret++;
    }
    if(pxl.p[i].y<1 || pxl.p[i].y>img.dimy) {
      fprintf(stderr, "Error: pixel outside image x dimension.\n");
      ret++;
    }
  }
  if(ret!=0) {
    imgEmpty(&img); pxlFree(&pxl); return(4);
  }


  /*
   *  Set mask pixels
   */
  if(verbose>0) printf("setting mask contents\n");
  for(int zi=1; zi<img.dimz; zi++)
    for(int yi=1; yi<img.dimy; yi++)
      for(int xi=1; xi<img.dimx; xi++)
        img.m[zi][yi][xi][0]=(float)0.0;
  for(long long int i=0; i<pxl.pxlNr; i++)
    img.m[pxl.p[i].z-1][pxl.p[i].y-1][pxl.p[i].x-1][0]=(float)1.0;


  /*
   *  Write the mask image
   */
  if(verbose>1) fprintf(stdout, "writing mask in %s\n", maskfile);
  ret=imgWrite(maskfile, &img);
  if(ret) {
    fprintf(stderr, "Error: %s\n", img.statmsg);
    imgEmpty(&img); return(11);
  }
  imgEmpty(&img);
  if(verbose>0) printf("mask written.\n\n");

  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/// @endcond
