/** @file ecat2nii.c
 *  @brief Convert ECAT 6.3 or 7 image format to NIfTI-1.
 *  @copyright (c) Turku PET Centre
 *  @author Vesa Oikonen
 */
/// @cond
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <string.h>
#include <time.h>
#include <sys/stat.h>
#include <unistd.h>
/*****************************************************************************/
#include "libtpcmisc.h"
#include "libtpcimgio.h"
#include "libtpcimgp.h"
/*****************************************************************************/

/*****************************************************************************/
static char *info[] = {
  "Converts PET images from ECAT 6.3 or 7 to NIfTI-1 format.",
  "Conversion can also be done using ImageConverter (.NET application).",
  " ",
  "Image byte order is determined by the computer where the program is run.",
  "NIfTI image format does not contain information on the frame times.",
  "Frame times can be retrieved from SIF file, which can be created optionally.",
  "SIF can also be created later using other software.",
  " ",
  "Usage: @P [Options] ecatfile(s)",
  " ",
  "Options:",
  " -O=<output path>",
  "     Data directory for NIfTI files, if other than the current working path.",
  " -dual",
  "     Save the image in dual file format (the header and voxel data in",
  "     separate files *.hdr and *.img); single file format (*.nii)",
  "     is the default.",
  " -sif",
  "     SIF is saved with NIfTI; note that existing SIF will be overwritten.",
  " -stdoptions", // List standard options like --help, -v, etc
  " ",
  "Example:",
  "  @P *.v",
  " ",
  "See also: nii2ecat, nii_lhdr, ecat2ana, eframe, img2flat",
  " ",
  "Keywords: image, format conversion, ECAT, NIfTI",
  0};
/*****************************************************************************/

/*****************************************************************************/
/* Turn on the globbing of the command line, since it is disabled by default in
   mingw-w64 (_dowildcard=0); in MinGW32 define _CRT_glob instead, if necessary;
   In Unix&Linux wildcard command line processing is enabled by default. */
/*
#undef _CRT_glob
#define _CRT_glob -1
*/
int _dowildcard = -1;
/*****************************************************************************/

/*****************************************************************************/
/**
 *  Main
 */
int main(int argc, char **argv)
{
  int      ai, help=0, version=0, verbose=1;
  int      ret, errorNr=0, fi=0, fileNr=0, ffi=0;
  int      fileformat=IMG_NIFTI_1S; /* IMG_NIFTI_1S = single file format,
                                       IMG_NIFTI_1D = dual file format */ 
  int      save_sif=0; // SIF is made (1) or not made (0)
  char     ecatfile[FILENAME_MAX], dbname[FILENAME_MAX], *cptr;
  char     outputdir[FILENAME_MAX], dbdir[FILENAME_MAX];
  IMG      img;
  float    fmin, fmax;
  SIF      sif;
  int      prev_fileformat=0;


  /*
   *  Get arguments
   */
  if(argc==1) {tpcPrintUsage(argv[0], info, stderr); return(1);}
  outputdir[0]=ecatfile[0]=dbname[0]=dbdir[0]=(char)0;
  imgInit(&img); sifInit(&sif);
  /* Options */
  for(ai=1; ai<argc; ai++) if(*argv[ai]=='-') { /* options */
    cptr=argv[ai]+1; if(*cptr=='-') cptr++; if(cptr==NULL) continue;
    if(tpcProcessStdOptions(argv[ai], &help, &version, &verbose)==0) continue;
    if(strncasecmp(cptr, "DUAL", 3)==0) {
      fileformat=IMG_NIFTI_1D; continue;
    } else if(strncasecmp(cptr, "O=", 2)==0) {
      cptr+=2;
      if(strlen(cptr)>0) strcpy(outputdir, cptr); else strcpy(outputdir, ".");
      continue;
    } else if(strcasecmp(cptr, "SIF")==0) {
      save_sif=1; continue;
    }
    fprintf(stderr, "Error: invalid option '%s'\n", argv[ai]);
    return(1);
  } else break;
  
  /* Print help or version? */
  if(help==2) {tpcHtmlUsage(argv[0], info, ""); return(0);}
  if(help) {tpcPrintUsage(argv[0], info, stdout); return(0);}
  if(version) {tpcPrintBuild(argv[0], stdout); return(0);}

  /* Process other arguments, starting from the first non-option */
  for(; ai<argc; ai++) {
    /* Other files are ECAT files which must exist */
    if(access(argv[ai], 0) == -1) {
      fprintf(stderr, "Error: file %s does not exist\n", argv[ai]);
      return(1);
    }
    if(ffi<1) ffi=ai;
    fileNr++;
  }

  /* Is something missing? */
  if(fileNr<1) {
    fprintf(stderr, "Error: no ECAT files were specified.\n");
    return(1);
  }

  /* In verbose mode print arguments and options */
  if(verbose>1) {
    printf("program := %s\n", argv[0]);
    if(fileformat==IMG_NIFTI_1D) printf("fileformat := 1D\n");
    else printf("fileformat := 1S\n");
    printf("save_sif := %d\n", save_sif);
    printf("outputdir := %s\n", outputdir);
    printf("fileNr := %d\n", fileNr);
  }
  if(verbose>3) IMG_TEST=verbose-3; else IMG_TEST=0;


  /*
   *  Process each ECAT file separately
   */
  if(verbose>1) printf("processing...\n");
  fileNr=0;
  for(ai=ffi; ai<argc; ai++) {

    strlcpy(ecatfile, argv[ai], FILENAME_MAX);
    if(verbose>0) {fprintf(stdout, "%s : \n", ecatfile); fflush(stdout);}
    imgEmpty(&img);
    sifEmpty(&sif);

    /*
     *  Make NIfTI database name
     */
    /*  Make output directory name, if it was not given */
    if(strlen(outputdir)<1) {
      strcpy(dbdir, ".");
    } else {
      strlcpy(dbdir, outputdir, FILENAME_MAX);
      /* Remove trailing slashes */
      ret=strlen(dbdir)-1;
      if(dbdir[ret]=='/' || dbdir[ret]=='\\') dbdir[ret]=(char)0;
    }
    /* Create the subdirectory */
    if(access(dbdir, 0) == -1) {
      if(verbose>0) {
        fprintf(stdout, "  Creating subdirectory %s\n", dbdir);
        fflush(stdout);
      }
#ifdef WIN32
      ret=mkdir(dbdir);
#else
      ret=mkdir(dbdir, 00775);
#endif
      if(ret!=0) {
        fprintf(stderr, "  Error: cannot create subdirectory.\n");
        fflush(stderr); errorNr++; continue;
      }
    }
    /* Combine path and filename */
    cptr=strrchr(ecatfile, '/'); if(cptr==NULL) cptr=strrchr(ecatfile, '\\');
    if(cptr==NULL) cptr=ecatfile; else cptr++;
#ifdef WIN32
    snprintf(dbname, FILENAME_MAX, "%s\\%s", dbdir, cptr);
#else
    snprintf(dbname, FILENAME_MAX, "%s/%s", dbdir, cptr);
#endif
    cptr=strrchr(dbname, '.'); if(cptr!=NULL) cptr[0]=(char)0;
    if(verbose>1) printf("  NIfTI db_name: '%s'\n", dbname);

    /* Delete previous NIfTI */
    /* It does not need to be valid NIfTI format, just that the filenames
       match */
    if(verbose>1) printf("  checking if file exists\n");
    char hdrfile[FILENAME_MAX], imgfile[FILENAME_MAX];
    ret=niftiCreateFNames(dbname, hdrfile, imgfile, NULL, fileformat);
    if(ret!=0) {
      fprintf(stderr, "  Error: invalid NIfTI name %s\n", dbname);
      fprintf(stdout, "  No conversion is done for %s\n", ecatfile);
      fflush(stdout); fflush(stderr);
      errorNr++; continue;
    }
    if(fileformat==IMG_NIFTI_1S) {
      /* If single file format, then just try to delete */
      ret=niftiRemove(dbname, fileformat, verbose-3);
    } else {
      /* Dual file format; check that we are not accidentally deleting
         ECAT 6.3 or other .img file that is not NIfTI or Analyze */
      if(access(hdrfile, 0)!=-1) {
        /* hdr file exists, thus delete it and possible img file */
        ret=niftiRemove(dbname, fileformat, verbose-3);
      } else if(access(imgfile, 0)!=-1) {
        /* only img exists */
        fprintf(stderr, "  Error: %s would be overwritten.\n", imgfile);
        fprintf(stdout, "  No conversion is done for %s\n", ecatfile);
        fflush(stdout); fflush(stderr);
        errorNr++; continue;
      }
    }


    /*
     *  Get the global min and max pixel values;
     *  those are needed for the header, and for scaling pixels
     */
    if(verbose>1) fprintf(stdout, "  searching min and max in %s\n", ecatfile);
    ret=imgReadMinMax(ecatfile, &fmin, &fmax);
    if(ret) {
      fprintf(stderr, "Error: %s\n", imgStatus(ret)); 
      errorNr++; fflush(stderr); continue;
    }
    if(verbose>2) 
      printf("    global_min := %g\n    global_max := %g\n", fmin, fmax);

    /*
     *  Get the frame number and allocate memory for SIF, if requested
     */
    if(save_sif) {
      /* Read information from input image for SIF */
      if(verbose>1) printf("  reading header information\n");
      ret=imgReadHeader(ecatfile, &img, img._fileFormat);
      if(ret) {
        fprintf(stderr, "Error: %s\n", imgStatus(ret));
        errorNr++; fflush(stderr); continue;
      }
      /* Allocate SIF to store frame times */
      ret=sifSetmem(&sif, img.dimt); if(ret!=0) {
        fprintf(stderr, "Error: out of memory.\n");
        errorNr++; fflush(stderr); continue;
      }
      sif.colNr=4; sif.version=1;
      imgEmpty(&img);
    }

    /*
     *  Conversion of ECAT file, one frame at a time
     */
    if(verbose>0) {
      fprintf(stdout, "  processing %s\n", ecatfile); fflush(stdout);}
    fi=0; imgEmpty(&img);
    while((ret=imgReadFrame(ecatfile, fi+1, &img, 0)) == 0) {
      if(verbose>1) {printf("    frame %d\n", fi+1); fflush(stdout);}
      /* Set the write file format including byte order */
      if(fi==0) // must be saved for reading next frame
        prev_fileformat=img._fileFormat;
      img._fileFormat=fileformat;

      /* Write the frame in NIfTI format */
      ret=imgWriteNiftiFrame(dbname, fi+1, &img, 0, fmin, fmax, verbose-3);
      if(ret) niftiRemove(dbname, fileformat, verbose-4);
      //printf("ret := %d\n", ret);
      if(ret!=STATUS_OK) {
        if(verbose>2) fprintf(stderr, "Error: %s.\n", imgStatus(ret));
        break;
      }
      if(verbose>2) {printf("    frame %d written.\n", fi+1); fflush(stdout);}
      /* Set SIF contents for this frame, if requested */
      if(save_sif) {
        sif.x1[fi]=img.start[0]; sif.x2[fi]=img.end[0];
        if(fi==0) {
          /* Set isotope */
          strcpy(sif.isotope_name, imgIsotope(&img) );
          /* Set studynumber */
          strcpy(sif.studynr, img.studyNr);
          /* Set scan start time */
          sif.scantime=img.scanStart;
        }
      }
      /* Prepare to the next frame */
      img._fileFormat=prev_fileformat;
      fi++;
      if(verbose>2) {
        printf("    about to read frame %d\n", fi+1); fflush(stdout);}
    } // next frame
    if(verbose>0) {
      if(ret==STATUS_NOMATRIX) {
        fprintf(stdout, "  %d frame(s) processed.\n", fi); fflush(stdout);
      }
    }
    if(ret!=STATUS_OK && ret!=STATUS_NOMATRIX) {
      fprintf(stderr, "Error: %s\n", imgStatus(ret)); // imgInfo(&img);
      niftiRemove(dbname, fileformat, verbose-2);
      errorNr++; fflush(stderr); continue;
    }
    imgEmpty(&img);

    /* Save SIF, if requested */
    if(save_sif) {
      char temp[FILENAME_MAX+10];
      sprintf(temp, "%s.sif", dbname);
      ret=sifWrite(&sif, temp);
      if(ret!=0) {
        fprintf(stderr, "Error: cannot write %s\n", temp);
        errorNr++; fflush(stderr); continue;
      }
      sifEmpty(&sif);
    }

  } /* next file */

  if(errorNr>0) return(errorNr+10);
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/// @endcond
