#!/bin/bash
#: Title      : test_p2t_v3c
#: Date       : 2023-06-29
#: Author     : "Vesa Oikonen" <vesa.oikonen@utu.fi>
#: Options    : None

# Set the name of executable to test
if [ -n "${OS+1}" ] && [ "$OS" = "Windows_NT" ]; then
EXT=.exe;
else
EXT=;
fi
PROGRAM=../../p2t_v3c$EXT;
if [ ! -f $PROGRAM ]; then
  printf "Failed: executable does not exist.\n"
  exit 1
fi


printf "=====================================================================\n"
printf "creating test data for %s\n" $PROGRAM
printf "=====================================================================\n"


printf "\n verifying that required data exists \n"

if [ ! -f bolus.dat ] || [ ! -f bolus_short.dat ] || [ ! -f stead100.dat ]; then
  printf "Failed: required data does not exist.\n"
  exit 1
fi
if [ ! -f bolus_toofew.dat ] || [ ! -f bolus_wrongorder.dat ] ; then
  printf "Failed: required data does not exist.\n"
  exit 1
fi
if [ ! -f correct_pb_v3cm.dat ] || [ ! -f correct_pb_vp3cm.dat ] ; then
  printf "Failed: required data does not exist.\n"
  exit 1
fi
if [ ! -f correct_ss_v3cm.dat ] || [ ! -f correct_ss_vp3cm.dat ] ; then
  printf "Failed: required data does not exist.\n"
  exit 1
fi

if [ ! -f multi.dat ] ; then
  tacadd -ovr multi.dat bolus.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacadd -force multi.dat bolus.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi

if [ ! -f bolus_auc.dat ] ; then
  interpol -i -x=1000 bolus.dat bolus_auc.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi



printf "\n\n"
printf "=====================================================================\n"
printf "testing %s\n" $PROGRAM
printf "=====================================================================\n"



printf "\n 0.1.0 \n"
printf " Test case: Unknown option. \n"
printf " Expected result: Error. \n\n"

$PROGRAM -stupidoption bolus.dat bolus.dat 0.2 0.4 0 0 0 0 0 10 1 tissue.dat 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error "invalid option '-stupidoption'."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n 0.1.1 \n"
printf " Test case: Extra argument. \n"
printf " Expected result: Error. \n\n"

$PROGRAM bolus.dat bolus.dat 0.2 0.4 0 0 0 0 0 10 1 tissue.dat extra 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error "invalid argument 'extra'."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n 0.1.2 \n"
printf " Test case: Missing argument(s). \n"
printf " Expected result: Error. \n\n"

$PROGRAM bolus.dat 0.2 0.4 0 0 0 0 0 10 1 tissue.dat 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

$PROGRAM bolus.dat bolus.dat 0.2 0.4 0 0 0 0 0 10 1 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

$PROGRAM bolus.dat bolus.dat 0.2 0.4 0  0 0 0 10 1 tissue.dat tissue.dat 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

$PROGRAM bolus.dat  tissue.dat 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

$PROGRAM   1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

printf "\n passed. \n\n"


printf "\n-------------------------------------------------------------------\n"

printf "\n 0.2.0 \n"
printf " Test case: Correct input, no options. \n"
printf " Expected result 1: Values of variables are correct. \n"
printf " Expected result 2: Output file is created. \n\n"

rm -f output.dat
taccalc bolus_short.dat x 0.5 blood_short.dat
$PROGRAM -d2 bolus_short.dat blood_short.dat 0.2 0.4 0 0 0 0 0 10 1 output.dat 1>stdout.txt 2>&1
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

iftisval stdout.txt pfile "bolus_short.dat"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt bfile "blood_short.dat"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt sfile "output.dat"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt parameters "0.2 0.4 0 0 0 0"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt f "0"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Vb "0.1"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt fA "0.01"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt save_only_total "0"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt parallel "0"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

if [ ! -f output.dat ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 0.2.1 \n"
printf " Test case: Options -sub and -nosub. \n"
printf " Expected result: Options are correctly identified. \n\n"

rm -f output.dat
$PROGRAM -d2 -sub bolus_short.dat bolus_short.dat 0.2 0.4 0 0 0 0 0 10 1 output.dat 1>stdout.txt 2>&1
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt save_only_total "0"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

rm -f output.dat
$PROGRAM -d2 -nosub bolus_short.dat bolus_short.dat 0.2 0.4 0 0 0 0 0 10 1 output.dat 1>stdout.txt 2>&1
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt save_only_total "1"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 0.2.2 \n"
printf " Test case: Options -parallel and -series. \n"
printf " Expected result: Options are correctly identified. \n\n"

rm -f output.dat
$PROGRAM -d2 -parallel bolus_short.dat bolus_short.dat 0.2 0.4 0 0 0 0 0 10 1 output.dat 1>stdout.txt 2>&1
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt parallel "1"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

rm -f output.dat
$PROGRAM -d2 -series bolus_short.dat bolus_short.dat 0.2 0.4 0 0 0 0 0 10 1 output.dat 1>stdout.txt 2>&1
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt parallel "0"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

rm -f output.dat
$PROGRAM -d2 -paral bolus_short.dat bolus_short.dat 0.2 0.4 0 0 0 0 0 10 1 output.dat 1>stdout.txt 2>&1
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt parallel "1"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

rm -f output.dat
$PROGRAM -d2 -ser bolus_short.dat bolus_short.dat 0.2 0.4 0 0 0 0 0 10 1 output.dat 1>stdout.txt 2>&1
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt parallel "0"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n===================================================================\n"
printf "   Input file \n"
printf "===================================================================\n"


printf "\n 1.0.0 \n"
printf " Test case: Non-existing input file. \n"
printf " Expected result: Error. \n\n"

rm -f output.dat
$PROGRAM nonexisting.dat bolus.dat 0.2 0.4 0 0 0 0 0 10 1 output.dat 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Error in reading 'nonexisting.dat'" "cannot open file"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

rm -f output.dat
$PROGRAM bolus.dat nonexisting.dat 0.2 0.4 0 0 0 0 0 10 1 output.dat 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Error in reading 'nonexisting.dat'" "cannot open file"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n 1.0.1 \n"
printf " Test case: Wrong input file format. \n"
printf " Expected result: Error. \n\n"

printf "abc defg\nbubudum\nmeh\n" > bad.dat
rm -f output.dat
$PROGRAM bad.dat bolus.dat 0.2 0.4 0 0 0 0 0 10 1 output.dat 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Error in reading 'bad.dat'" "wrong format"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

rm -f output.dat
$PROGRAM bolus.dat bad.dat 0.2 0.4 0 0 0 0 0 10 1 output.dat 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Error in reading 'bad.dat'" "wrong format"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 1.0.2 \n"
printf " Test case: Input file contains more than one TAC. \n"
printf " Expected result: Error. \n\n"

rm -f output.dat
$PROGRAM multi.dat bolus.dat 0.2 0.4 0 0 0 0 0 10 1 output.dat 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Error" "plasma data contains more than one curve."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

rm -f output.dat
$PROGRAM bolus.dat multi.dat 0.2 0.4 0 0 0 0 0 10 1 output.dat 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Error" "blood data contains more than one curve."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 1.0.3 \n"
printf " Test case: Input file contains less than 3 samples. \n"
printf " Expected result: Error. \n\n"

rm -f output.dat
$PROGRAM bolus_toofew.dat bolus.dat 0.2 0.4 0 0 0 0 0 10 1 output.dat 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Error" "too few samples in plasma data."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

rm -f output.dat
$PROGRAM bolus.dat bolus_toofew.dat 0.2 0.4 0 0 0 0 0 10 1 output.dat 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Error" "too few samples in blood data."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

rm -f output.dat
$PROGRAM bolus_toofew.dat bolus_toofew.dat 0.2 0.4 0 0 0 0 0 10 1 output.dat 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Error" "too few samples in plasma data."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n 1.0.4 \n"
printf " Test case: Input file samples are not in ascending order. \n"
printf " Expected result: Error (from simulation routine). \n\n"

rm -f output.dat
$PROGRAM bolus_wrongorder.dat bolus.dat 0.2 0.4 0 0 0 0 0 10 1 output.dat 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

rm -f output.dat
$PROGRAM bolus.dat bolus_wrongorder.dat 0.2 0.4 0 0 0 0 0 10 1 output.dat 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

rm -f output.dat
$PROGRAM bolus_wrongorder.dat bolus_wrongorder.dat 0.2 0.4 0 0 0 0 0 10 1 output.dat 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n===================================================================\n"
printf "   Rate constants \n"
printf "===================================================================\n"

printf "\n 1.1.0 \n"
printf " Test case 1: All but the first rate constant can be set to zero. \n"
printf " Test case 2: Negative rate constants are not allowed. \n"
printf " Test case 3: Rate constant must be samller than 1.0E+08. \n"
printf " Expected result 1: Error if first is zero or less. \n"
printf " Expected result 2: Error if any rate constant is negative. \n"
printf " Expected result 3: Error if any rate constant is too large. \n\n"

rm -f output.dat
$PROGRAM bolus.dat bolus.dat 0 0.4 0 0 0 0 0 10 1 output.dat 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Error" "k1 must be > 0."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

rm -f output.dat
$PROGRAM -parallel -d2 bolus_short.dat bolus_short.dat 0.2 0 0 0 0 0 0 10 1 output.dat 1>stdout.txt 2>&1
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "parameters" "0.2 0 0 0 0 0"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

rm -f output.dat
$PROGRAM -series -d2 bolus_short.dat bolus_short.dat 0.2 0 0 0 0 0 0 10 1 output.dat 1>stdout.txt 2>&1
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "parameters" "0.2 0 0 0 0 0"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



rm -f output.dat
$PROGRAM bolus.dat bolus.dat -0.1 0.2 0 0 0 0 0 10 1 output.dat 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
$PROGRAM bolus.dat bolus.dat 0.1 -0.2 0 0 0 0 0 10 1 output.dat 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
$PROGRAM bolus.dat bolus.dat 0.1 0.2 -0.05 0 0 0 0 10 1 output.dat 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
$PROGRAM bolus.dat bolus.dat 0.1 0.2 0.05 -0.1 0 0 0 10 1 output.dat 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
$PROGRAM bolus.dat bolus.dat 0.1 0.2 0.1 0.05 -0.01 0 0 10 1 output.dat 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
$PROGRAM bolus.dat bolus.dat 0.1 0.2 0.05 0.1 0.2 -0.000001 0 10 1 output.dat 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
$PROGRAM bolus.dat bolus.dat 0.1 0.2 0 0 0 0 -0.1 10 1 output.dat 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


rm -f output.dat
$PROGRAM bolus.dat bolus.dat 2.0E+08 0.2 0 0 0 0 0 10 1 output.dat 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
$PROGRAM bolus.dat bolus.dat 0.1 2.0E+08 0 0 0 0 0 10 1 output.dat 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
$PROGRAM bolus.dat bolus.dat 0.1 0.2 2.0E+08 0 0 0 0 10 1 output.dat 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
$PROGRAM bolus.dat bolus.dat 0.1 0.2 0.05 2.0E+08 0 0 0 10 1 output.dat 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
$PROGRAM bolus.dat bolus.dat 0.1 0.2 0.1 0.05 2.0E+08 0 0 10 1 output.dat 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
$PROGRAM bolus.dat bolus.dat 0.1 0.2 0.05 0.1 0.2 2.0E+08 0 10 1 output.dat 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n===================================================================\n"
printf "   Output file \n"
printf "===================================================================\n"

printf "\n 1.2.0 \n"
printf " Test case: Output file exists. \n"
printf " Expected result: Backup is made from the existing one. \n\n"
rm -f output.dat output.dat.bak
$PROGRAM -nosub bolus_short.dat bolus_short.dat 0.4 0.2 0.2 0.2 0.1 0.2 0 10 1 output.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
$PROGRAM -nosub bolus_short.dat bolus_short.dat 0.3 0.3 0.3 0.3 0.3 0.3 0 10 2 output.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -x=n -y=y -xunit=n -yunit=n output.dat output.dat.bak
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"




printf "\n===================================================================\n"
printf "   Validity in perfect steady-state and bolus \n"
printf "===================================================================\n"

printf "\n 2.0.0 \n"
printf " Test case: Steady-state and 3TCM in series. \n"
printf " Expected result: Correct concentration level is reached. \n\n"

rm -f output.dat temp.dat test.dat
$PROGRAM -nosub stead100.dat stead100.dat 0.5 5.0 0.25 0.5 0.6 0.1 0.0 10 30 output.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
taccbv --force -tv output.dat stead100.dat 10 temp.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
taccut temp.dat 999.9 1000.1 test.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
tacmatch -x=y -y=y -xunit=n -yunit=n -tacnames=n -abs=0.001 -rel=0.01 correct_ss_v3cm.dat test.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n 2.0.1 \n"
printf " Test case: Steady-state and 3TCM in parallel. \n"
printf " Expected result: Correct concentration level is reached. \n\n"

rm -f output.dat temp.dat test.dat
$PROGRAM -nosub -parallel stead100.dat stead100.dat 0.5 5.0 0.25 0.5 0.6 0.1 0.0 10 30 output.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
taccbv --force -tv output.dat stead100.dat 10 temp.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
taccut temp.dat 999.9 1000.1 test.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
tacmatch -x=y -y=y -xunit=n -yunit=n -tacnames=n -abs=0.001 -rel=0.01 correct_ss_vp3cm.dat test.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n-------------------------------------------------------------------\n"

printf "\n 2.1.0 \n"
printf " Test case: Bolus and 3TCM in series. \n"
printf " Expected result: Correct tissue-to-plasma AUC ratio is reached. \n\n"

rm -f output.dat output_auc.dat temp.dat
$PROGRAM -series -nosub bolus.dat bolus.dat 0.5 5.0 0.25 0.5 0.6 0.1 0.0 10 30 output.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
taccbv --force -tv output.dat bolus.dat 10 temp.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
interpol -i -x=1000 temp.dat output_auc.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
taccalc --force output_auc.dat : bolus_auc.dat temp.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
tacmatch -x=y -y=y -xunit=n -yunit=n -tacnames=n -abs=0.001 -rel=0.01 correct_pb_v3cm.dat temp.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 2.1.1 \n"
printf " Test case: Bolus and 3TCM in parallel. \n"
printf " Expected result: Correct tissue-to-plasma AUC ratio is reached. \n\n"

rm -f output.dat output_auc.dat temp.dat
$PROGRAM -parallel -nosub bolus.dat bolus.dat 0.5 5.0 0.25 0.5 0.6 0.1 0.0 10 30 output.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
taccbv --force -tv output.dat bolus.dat 10 temp.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
interpol -i -x=1000 temp.dat output_auc.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
taccalc --force output_auc.dat : bolus_auc.dat temp.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
tacmatch -x=y -y=y -xunit=n -yunit=n -tacnames=n -abs=0.001 -rel=0.01 correct_pb_vp3cm.dat temp.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n===================================================================\n"
printf "   Perfusion effect \n"
printf "===================================================================\n"


printf "\n 3.0.0 \n"
printf " Test case 1: Bolus and 3TCM in series with perfusion effect. \n"
printf " Test case 2: Quantitative result is not yet tested. \n"
printf " Expected result: TACs can be simulated and plotted. \n\n"

rm -f sim1.dft simu1.dft
$PROGRAM -series bolus.dat bolus.dat 0.2 0.4 0.3 0.33 0.6 0.1 0.5 10 20 sim1.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
ainterp sim1.dft 0 simu1.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tac2svg -legend=y -x2=30 simu1.svg simu1.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"


printf "\n 3.0.1 \n"
printf " Test case 1: Bolus and 3TCM in series with perfusion effect. \n"
printf " Test case 2: Quantitative result is not yet tested. \n"
printf " Expected result: TACs can be simulated and plotted. \n\n"

rm -f sim2.dft simu2.dft
$PROGRAM -parallel bolus.dat bolus.dat 0.2 0.4 0.3 0.33 0.6 0.1 0.5 10 20 sim2.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
ainterp sim2.dft 0 simu2.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tac2svg -legend=y -x2=30 simu2.svg simu2.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"




printf "\n===================================================================\n"
printf "   Testing against simulation in spreadsheet \n"
printf "===================================================================\n"


printf "\n 4.0.0 \n"
printf " Test case: Bolus and 3TCM in series with Vb but no perfusion effect. \n"
printf " Expected result: simulated PET TAC is correct. \n\n"

$PROGRAM -series -nosub input_3cm.dat input_3cm.dat 0.4 0.6 0.3 0.4 0.08 0.01 0 4 100 sims.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tac2svg -legend=n -mt="3TCM-S" sims.svg -l correct_3cms.dat sims.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -x=y -y=y -xunit=y -yunit=y -tacnames=n -abs=0.001 -rel=0.01 correct_3cms.dat sims.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 4.1.0 \n"
printf " Test case: Bolus and 3TCM in parallel with Vb but no perfusion effect. \n"
printf " Expected result: simulated PET TAC is correct. \n\n"

$PROGRAM -parallel -nosub input_3cm.dat input_3cm.dat 0.4 0.6 0.2 0.05 0.2 0.2 0 4 100 simp.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tac2svg -legend=n -mt="3TCM-P" simp.svg -l correct_3cmp.dat simp.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -x=y -y=y -xunit=y -yunit=y -tacnames=n -abs=0.001 -rel=0.01 correct_3cmp.dat simp.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"




printf "\n===================================================================\n"
printf "   All passed! \n"
printf "===================================================================\n"
exit 0

